# 🐳 Docker Setup - Projeto Allure

Este documento explica como configurar e executar o projeto Allure usando Docker.

## 📋 Pré-requisitos

- Docker (versão 20.10 ou superior)
- Docker Compose (versão 2.0 ou superior)
- Git

## 🚀 Início Rápido

### 1. Configuração Inicial

```bash
# Clone o repositório (se ainda não tiver)
git clone <seu-repositorio>
cd allure_project

# Copie o arquivo de configuração
cp env.example .env

# Edite o arquivo .env com suas configurações
nano .env
```

### 2. Executar o Projeto

#### Modo Desenvolvimento
```bash
# Usando o script de inicialização
./scripts/start.sh dev

# Ou diretamente com docker-compose
docker-compose -f docker-compose.dev.yml up --build
```

#### Modo Produção
```bash
# Usando o script de inicialização
./scripts/start.sh prod

# Ou diretamente com docker-compose
docker-compose up --build -d
```

## 🔧 Comandos Disponíveis

O script `scripts/start.sh` oferece os seguintes comandos:

```bash
./scripts/start.sh dev      # Iniciar em modo desenvolvimento
./scripts/start.sh prod     # Iniciar em modo produção
./scripts/start.sh build    # Construir as imagens Docker
./scripts/start.sh stop     # Parar todos os containers
./scripts/start.sh clean    # Limpar containers e volumes
./scripts/start.sh logs     # Mostrar logs dos containers
./scripts/start.sh shell    # Acessar shell do container web
./scripts/start.sh migrate  # Executar migrações do banco
./scripts/start.sh help     # Mostrar ajuda
```

## 📁 Estrutura de Arquivos Docker

```
allure_project/
├── Dockerfile                 # Imagem da aplicação Django
├── docker-compose.yml         # Configuração para produção
├── docker-compose.dev.yml     # Configuração para desenvolvimento
├── nginx.conf                 # Configuração do Nginx
├── .dockerignore             # Arquivos ignorados no build
├── env.example               # Exemplo de variáveis de ambiente
├── scripts/
│   └── start.sh              # Script de inicialização
└── allure_project/
    └── settings_production.py # Configurações para produção
```

## 🌐 Acessos

Após iniciar o projeto:

- **Aplicação**: http://localhost:8000
- **Banco de dados**: localhost:5432
- **Nginx** (produção): http://localhost:80

## ⚙️ Configurações

### Variáveis de Ambiente (.env)

```bash
# Django
DEBUG=True
SECRET_KEY=your-secret-key-here
ALLOWED_HOSTS=localhost,127.0.0.1,0.0.0.0

# Banco de Dados
DATABASE_URL=postgresql://allure_user:allure_password@db:5432/allure_db

# Email
EMAIL_HOST=smtp.gmail.com
EMAIL_PORT=587
EMAIL_USE_TLS=True
EMAIL_HOST_USER=your-email@gmail.com
EMAIL_HOST_PASSWORD=your-app-password

# Mercado Pago
MERCADO_PAGO_ACCESS_TOKEN=your-mercadopago-token
MERCADO_PAGO_SANDBOX=True
MODO_TESTE_PAGAMENTO=True
SITE_URL=http://localhost:8000
```

### Modos de Execução

#### Desenvolvimento
- Hot reload habilitado
- Debug ativado
- Banco SQLite (opcional) ou PostgreSQL
- Volumes montados para desenvolvimento

#### Produção
- Gunicorn como servidor WSGI
- Nginx como proxy reverso
- PostgreSQL como banco de dados
- Configurações de segurança ativadas
- Compressão e cache otimizados

## 🗄️ Banco de Dados

### PostgreSQL (Recomendado para Produção)

O projeto está configurado para usar PostgreSQL em produção:

```bash
# Acessar o banco
docker-compose exec db psql -U allure_user -d allure_db

# Fazer backup
docker-compose exec db pg_dump -U allure_user allure_db > backup.sql

# Restaurar backup
docker-compose exec -T db psql -U allure_user -d allure_db < backup.sql
```

### SQLite (Desenvolvimento)

Para usar SQLite em desenvolvimento, modifique o `settings.py`:

```python
DATABASES = {
    'default': {
        'ENGINE': 'django.db.backends.sqlite3',
        'NAME': BASE_DIR / 'db.sqlite3',
    }
}
```

## 📊 Monitoramento e Logs

### Visualizar Logs

```bash
# Logs de todos os serviços
docker-compose logs -f

# Logs de um serviço específico
docker-compose logs -f web
docker-compose logs -f db
docker-compose logs -f nginx
```

### Acessar Container

```bash
# Shell do container web
docker-compose exec web bash

# Shell do banco de dados
docker-compose exec db psql -U allure_user -d allure_db
```

## 🔄 Deploy

### Deploy Local

```bash
# 1. Configurar variáveis de ambiente
cp env.example .env
# Editar .env com configurações de produção

# 2. Iniciar em modo produção
./scripts/start.sh prod

# 3. Executar migrações
./scripts/start.sh migrate
```

### Deploy em Servidor

1. **Preparar servidor**:
   ```bash
   # Instalar Docker e Docker Compose
   curl -fsSL https://get.docker.com -o get-docker.sh
   sh get-docker.sh
   
   # Instalar Docker Compose
   sudo curl -L "https://github.com/docker/compose/releases/download/v2.20.0/docker-compose-$(uname -s)-$(uname -m)" -o /usr/local/bin/docker-compose
   sudo chmod +x /usr/local/bin/docker-compose
   ```

2. **Configurar projeto**:
   ```bash
   # Clonar repositório
   git clone <seu-repositorio>
   cd allure_project
   
   # Configurar variáveis
   cp env.example .env
   # Editar .env com configurações do servidor
   ```

3. **Iniciar aplicação**:
   ```bash
   ./scripts/start.sh prod
   ```

## 🛠️ Troubleshooting

### Problemas Comuns

1. **Porta já em uso**:
   ```bash
   # Verificar processos usando a porta
   lsof -i :8000
   lsof -i :5432
   
   # Parar containers
   ./scripts/start.sh stop
   ```

2. **Erro de permissão**:
   ```bash
   # Dar permissão ao script
   chmod +x scripts/start.sh
   ```

3. **Problemas com volumes**:
   ```bash
   # Limpar volumes
   ./scripts/start.sh clean
   ```

4. **Erro de migração**:
   ```bash
   # Executar migrações manualmente
   docker-compose exec web python manage.py migrate
   ```

### Comandos Úteis

```bash
# Ver status dos containers
docker-compose ps

# Reiniciar um serviço específico
docker-compose restart web

# Reconstruir uma imagem específica
docker-compose build web

# Ver uso de recursos
docker stats

# Limpar sistema Docker
docker system prune -a
```

## 📚 Recursos Adicionais

- [Documentação do Docker](https://docs.docker.com/)
- [Documentação do Docker Compose](https://docs.docker.com/compose/)
- [Django Deployment](https://docs.djangoproject.com/en/stable/howto/deployment/)
- [Gunicorn Documentation](https://gunicorn.org/)

## 🤝 Contribuição

Para contribuir com melhorias no setup Docker:

1. Faça um fork do projeto
2. Crie uma branch para sua feature
3. Implemente as mudanças
4. Teste localmente
5. Submeta um pull request

## 📞 Suporte

Em caso de problemas:

1. Verifique os logs: `./scripts/start.sh logs`
2. Consulte a seção de troubleshooting
3. Abra uma issue no repositório
4. Entre em contato com a equipe de desenvolvimento
