# Sistema de Comunicação - Allure

## Visão Geral

O sistema de comunicação do Allure é um módulo completo para gerenciar notificações por email com gatilhos automáticos baseados em eventos do sistema.

## Características Principais

### ✅ **Funcionalidades Implementadas**

- **Templates de Email**: Sistema completo de templates HTML responsivos
- **Gatilhos Automáticos**: Disparo automático baseado em eventos
- **Fila de Processamento**: Sistema assíncrono para envio de emails
- **Logs Detalhados**: Rastreamento completo de todos os emails
- **Interface Admin**: Painel administrativo completo
- **Templates Responsivos**: Design moderno e mobile-friendly

### 🎯 **Eventos Cobertos**

1. **Cadastro de Modelo**
   - Confirmação de cadastro
   - Bem-vinda à plataforma

2. **Aprovação de Perfil**
   - Perfil aprovado
   - Perfil rejeitado

3. **Pagamentos**
   - Pagamento aprovado
   - Pagamento pendente
   - Pagamento falhou

4. **Planos de Anúncio**
   - Plano ativado
   - Plano expirando (7 e 3 dias)
   - Plano expirado

5. **Boost e Premium**
   - Ativação de boost
   - Ativação de premium
   - Expiração de serviços

6. **Outros**
   - Recuperação de senha
   - Contato de clientes
   - Newsletters

## Estrutura do Sistema

### 📁 **Arquivos Principais**

```
communication/
├── models.py          # Modelos de dados
├── services.py        # Serviços de envio
├── triggers.py        # Sistema de gatilhos
├── admin.py          # Interface administrativa
├── views.py          # Views do sistema
├── urls.py           # URLs do módulo
└── templates/
    ├── emails/       # Templates de email
    └── communication/ # Templates do admin
```

### 🗄️ **Modelos de Dados**

1. **EmailTemplate**: Templates de email
2. **EmailLog**: Log de emails enviados
3. **EmailTrigger**: Configuração de gatilhos
4. **EmailQueue**: Fila de processamento

## Como Usar

### 🚀 **Configuração Inicial**

1. **Executar migrações**:
```bash
python manage.py makemigrations communication
python manage.py migrate
```

2. **Configurar templates padrão**:
```bash
python manage.py setup_communication
```

3. **Configurar email em produção**:
```python
# settings.py
EMAIL_BACKEND = 'django.core.mail.backends.smtp.EmailBackend'
EMAIL_HOST = 'smtp.gmail.com'
EMAIL_PORT = 587
EMAIL_USE_TLS = True
EMAIL_HOST_USER = 'seu-email@gmail.com'
EMAIL_HOST_PASSWORD = 'sua-senha-app'
```

### 📧 **Enviando Emails**

#### **Método 1: Gatilhos Automáticos**
```python
from communication.services import trigger_modelo_cadastrado

# Disparar gatilho automaticamente
trigger_modelo_cadastrado(modelo)
```

#### **Método 2: Envio Direto**
```python
from communication.services import EmailService
from communication.models import EmailTemplate

template = EmailTemplate.objects.get(tipo='cadastro_sucesso')
contexto = {
    'nome': 'Maria Silva',
    'email': 'maria@exemplo.com',
    'data_cadastro': '15/01/2024'
}

EmailService.send_email(
    template=template,
    destinatario_email='maria@exemplo.com',
    contexto=contexto
)
```

#### **Método 3: Fila Assíncrona**
```python
from communication.services import EmailService

EmailService.queue_email(
    template=template,
    destinatario_email='maria@exemplo.com',
    contexto=contexto,
    delay_minutos=30  # Enviar em 30 minutos
)
```

### ⚙️ **Processando a Fila**

```python
from communication.services import QueueProcessor

# Processar 10 emails da fila
stats = QueueProcessor.process_queue(batch_size=10)
print(f"Processados: {stats['processados']}")
print(f"Enviados: {stats['enviados']}")
print(f"Falharam: {stats['falharam']}")
```

### 🎛️ **Gatilhos Personalizados**

```python
from communication.services import TriggerService

# Disparar gatilho personalizado
contexto = {
    'nome': 'João Silva',
    'email': 'joao@exemplo.com',
    'evento_especifico': 'valor'
}

TriggerService.process_trigger('evento_personalizado', contexto)
```

## Templates de Email

### 🎨 **Estrutura dos Templates**

Todos os templates herdam de `emails/base.html` que inclui:
- Design responsivo
- Cores da marca Allure
- Header e footer padronizados
- Suporte a variáveis dinâmicas

### 📝 **Variáveis Disponíveis**

Cada template tem suas variáveis específicas, mas todas incluem:
- `{{ nome }}`: Nome do destinatário
- `{{ email }}`: Email do destinatário
- `{{ site_url }}`: URL do site
- `{{ site_nome }}`: Nome do site
- `{{ data_atual }}`: Data atual
- `{{ hora_atual }}`: Hora atual

### ✏️ **Criando Novos Templates**

1. **Criar arquivo HTML**:
```html
<!-- communication/templates/emails/meu_template.html -->
{% extends "emails/base.html" %}

{% block content %}
<h2>Meu Template</h2>
<p>Olá {{ nome }},</p>
<p>Conteúdo do email...</p>
{% endblock %}
```

2. **Registrar no admin**:
```python
EmailTemplate.objects.create(
    nome='Meu Template',
    tipo='meu_tipo',
    assunto='Assunto: {{ nome }}',
    conteudo_html=render_to_string('emails/meu_template.html'),
    variaveis_disponiveis=['nome', 'email']
)
```

## Interface Administrativa

### 📊 **Dashboard**
- Estatísticas gerais
- Emails recentes
- Fila de processamento
- Templates mais usados

### 📋 **Gerenciamento**
- **Templates**: Criar, editar, preview
- **Logs**: Histórico de emails
- **Gatilhos**: Configurar eventos
- **Fila**: Monitorar processamento

### 🔧 **Funcionalidades**
- Preview de templates
- Envio de emails de teste
- Processamento manual da fila
- Estatísticas detalhadas

## Comandos de Management

### 🛠️ **Comandos Disponíveis**

```bash
# Configurar sistema inicial
python manage.py setup_communication

# Processar fila (para cron job)
python manage.py process_queue

# Limpar logs antigos
python manage.py cleanup_email_logs
```

## Integração com Eventos Existentes

### 🔗 **Pontos de Integração**

O sistema já está integrado com:

1. **Cadastro de Modelo** (`accounts/views.py`)
2. **Pagamentos** (`payments/views.py`)
3. **Aprovação de Perfis** (via signals)
4. **Webhooks do Mercado Pago**

### 📡 **Signals Configurados**

- `post_save` em `Modelo`
- `post_save` em `TransacaoPagamento`
- Gatilhos manuais para eventos específicos

## Monitoramento e Logs

### 📈 **Métricas Disponíveis**

- Total de emails enviados
- Taxa de entrega
- Emails por template
- Performance da fila
- Erros de envio

### 🔍 **Logs Detalhados**

Cada email gera um log com:
- Status do envio
- Data/hora
- Erro (se houver)
- Contexto usado
- Destinatário

## Configurações de Produção

### ⚙️ **Recomendações**

1. **SMTP**: Use provedor confiável (SendGrid, Mailgun, etc.)
2. **Fila**: Configure Celery para processamento assíncrono
3. **Monitoramento**: Configure alertas para falhas
4. **Backup**: Mantenha backup dos templates

### 🔒 **Segurança**

- Templates são sanitizados
- Emails são validados
- Rate limiting implementado
- Logs de segurança

## Troubleshooting

### ❌ **Problemas Comuns**

1. **Emails não enviando**:
   - Verificar configurações SMTP
   - Checar logs de erro
   - Validar templates

2. **Fila não processando**:
   - Executar `process_queue` manualmente
   - Verificar configurações de cron
   - Checar logs do sistema

3. **Templates não renderizando**:
   - Validar sintaxe Django
   - Verificar variáveis disponíveis
   - Testar com preview

### 🆘 **Suporte**

Para problemas específicos:
1. Verificar logs em `communication/models.py`
2. Usar interface admin para debug
3. Testar com emails de teste
4. Consultar documentação Django

## Próximos Passos

### 🚀 **Melhorias Futuras**

- [ ] Integração com Celery
- [ ] Templates A/B testing
- [ ] Analytics de abertura
- [ ] Segmentação de usuários
- [ ] Automação de campanhas
- [ ] Integração com WhatsApp
- [ ] SMS notifications

---

**Sistema desenvolvido para Allure - Plataforma de Ensaios Fotográficos**
