from django.shortcuts import render, redirect, get_object_or_404
from django.contrib.auth import authenticate, login, logout
from django.contrib.auth.decorators import login_required
from django.contrib.auth.models import User
from django.contrib import messages
from django.core.mail import send_mail
from django.conf import settings
from django.utils.crypto import get_random_string
from django.urls import reverse
from models.models import Modelo, FotoModelo, FotoAprovacao, Servico, Categoria
from django.http import JsonResponse
from datetime import datetime
import json

# Importar sistema de comunicação
from communication.services import trigger_modelo_cadastrado, trigger_modelo_aprovado


def login_view(request):
    """View de login"""
    if request.method == 'POST':
        username_or_email = request.POST.get('username')
        password = request.POST.get('password')
        
        # Tentar autenticar por username primeiro
        user = authenticate(request, username=username_or_email, password=password)
        
        # Se não funcionar, tentar por email
        if user is None:
            try:
                user_obj = User.objects.get(email=username_or_email)
                user = authenticate(request, username=user_obj.username, password=password)
            except User.DoesNotExist:
                user = None
        
        if user is not None:
            login(request, user)
            
            # Redirecionar para a página solicitada ou home
            next_url = request.GET.get('next', 'core:home')
            return redirect(next_url)
        else:
            messages.error(request, 'Usuário ou senha inválidos.')
    
    return render(request, 'accounts/login.html')


def logout_view(request):
    """View de logout"""
    logout(request)
    messages.success(request, 'Logout realizado com sucesso.')
    return redirect('core:home')


def cadastro_modelo_etapa1(request):
    """Cadastro inicial da modelo - Etapa 1"""
    if request.method == 'POST':
        try:
            # Dados básicos
            nome = request.POST.get('nome')
            sobrenome = request.POST.get('sobrenome')
            cpf = request.POST.get('cpf')
            data_nascimento = request.POST.get('data_nascimento')
            estado = request.POST.get('estado')
            cidade = request.POST.get('cidade')
            email = request.POST.get('email')
            telefone = request.POST.get('telefone')
            username = request.POST.get('username')
            nome_exibicao = request.POST.get('nome_exibicao')
            idade = request.POST.get('idade')
            whatsapp = request.POST.get('whatsapp')
            telegram = request.POST.get('telegram')
            link_contato = request.POST.get('link_contato')
            
            # Tipos de serviço selecionados
            categoria_servico = request.POST.getlist('categoria_servico')
            
            # Locais de atendimento selecionados
            local_atendimento = request.POST.getlist('local_atendimento')
            
            # Validar dados obrigatórios
            if not all([nome, sobrenome, cpf, data_nascimento, estado, cidade, email, telefone, username, nome_exibicao, idade]):
                messages.error(request, 'Todos os campos obrigatórios devem ser preenchidos.')
                return render(request, 'accounts/cadastro_modelo_etapa1.html', {'categorias': Categoria.objects.all()})
            
            # Validar tipos de serviço
            if not categoria_servico:
                messages.error(request, 'Você deve selecionar pelo menos um tipo de serviço.')
                return render(request, 'accounts/cadastro_modelo_etapa1.html', {'categorias': Categoria.objects.all()})
            
            # Validar locais de atendimento
            if not local_atendimento:
                messages.error(request, 'Você deve selecionar pelo menos um local de atendimento.')
                return render(request, 'accounts/cadastro_modelo_etapa1.html', {'categorias': Categoria.objects.all()})
            
            # Validar CPF único
            if Modelo.objects.filter(cpf=cpf).exists():
                messages.error(request, 'Este CPF já está cadastrado.')
                return render(request, 'accounts/cadastro_modelo_etapa1.html', {'categorias': Categoria.objects.all()})
            
            # Validar email único
            if User.objects.filter(email=email).exists() or Modelo.objects.filter(email_contato=email).exists():
                messages.error(request, 'Este email já está em uso.')
                return render(request, 'accounts/cadastro_modelo_etapa1.html', {'categorias': Categoria.objects.all()})
            
            # Validar username único
            if User.objects.filter(username=username).exists():
                messages.error(request, 'Este nome de usuário já está em uso.')
                return render(request, 'accounts/cadastro_modelo_etapa1.html', {'categorias': Categoria.objects.all()})
            
            # Obter senha escolhida pela modelo
            password = request.POST.get('password')
            
            # Criar usuário
            user = User.objects.create_user(
                username=username,
                email=email,
                password=password, # Use password chosen by model
                first_name=nome,
                last_name=sobrenome
            )
            
            # Converter data de nascimento
            data_nasc = datetime.strptime(data_nascimento, '%Y-%m-%d').date()
            
            # Criar modelo com status pendente de aprovação
            modelo = Modelo.objects.create(
                user=user,
                nome=nome,
                sobrenome=sobrenome,
                cpf=cpf,
                data_nascimento=data_nasc,
                estado=estado,
                cidade=cidade,
                nome_exibicao=nome_exibicao,
                idade=int(idade),
                telefone=telefone,
                whatsapp=whatsapp,
                telegram=telegram,
                link_contato=link_contato,
                email_contato=email,
                status='pendente_aprovacao'
            )
            
            # Salvar tipos de serviço e locais de atendimento
            if categoria_servico:
                modelo.set_categorias_servico(categoria_servico)
            
            if local_atendimento:
                modelo.set_locais_atendimento(local_atendimento)
            
            modelo.save()
            
            # Processar fotos para aprovação (exatamente 8 fotos)
            fotos_galeria = request.FILES.getlist('fotos_galeria')
            print(f"DEBUG: Fotos recebidas no backend: {len(fotos_galeria)}")
            print(f"DEBUG: Nomes dos arquivos: {[f.name for f in fotos_galeria]}")
            print(f"DEBUG: Modelo criado: {modelo.id} - {modelo.nome_exibicao}")
            
            if not fotos_galeria or len(fotos_galeria) != 8:
                messages.error(request, f'Você deve enviar exatamente 8 fotos para a galeria. Atualmente enviou: {len(fotos_galeria) if fotos_galeria else 0} foto(s).')
                modelo.delete()
                user.delete()
                return render(request, 'accounts/cadastro_modelo_etapa1.html', {'categorias': Categoria.objects.all()})
            
            # Salvar fotos para aprovação
            fotos_salvas = 0
            for i, foto in enumerate(fotos_galeria):
                try:
                    FotoAprovacao.objects.create(
                        modelo=modelo,
                        imagem=foto,
                        ordem=i
                    )
                    fotos_salvas += 1
                    print(f"Foto {i} salva com sucesso: {foto.name}")
                except Exception as e:
                    print(f"Erro ao salvar foto {i}: {e}")
                    continue
            
            print(f"Total de fotos salvas: {fotos_salvas}")
            
            if fotos_salvas > 0:
                messages.success(request, f'{fotos_salvas} foto(s) salva(s) para aprovação!')
            
            # Salvar modelo
            modelo.save()
            
            # Disparar gatilho de cadastro (será processado automaticamente via signals)
            # O gatilho será disparado automaticamente quando o modelo for salvo
            
            # Redirecionar para página de sucesso
            return render(request, 'accounts/cadastro_sucesso_etapa1.html', {
                'modelo': modelo
            })
            
        except Exception as e:
            messages.error(request, f'Erro ao realizar cadastro: {str(e)}')
            return render(request, 'accounts/cadastro_modelo_etapa1.html', {'categorias': Categoria.objects.all()})
    
    return render(request, 'accounts/cadastro_modelo_etapa1.html', {'categorias': Categoria.objects.all()})




def get_context():
    """Retorna o contexto comum para o template de cadastro"""
    return {
        'categorias': Categoria.objects.filter(ativo=True).order_by('ordem', 'nome'),
        'etnia_choices': Modelo.ETNIA_CHOICES,
    }


@login_required
def painel_modelo(request):
    """Painel da modelo logada"""
    try:
        modelo = request.user.modelo_profile
        
        # Verificar status de pagamento
        status_pagamento = None
        alerta_pagamento = None
        plano_ativo = None
        
        if modelo.status == 'pendente_aprovacao':
            status_pagamento = 'pendente_aprovacao'
            alerta_pagamento = {
                'tipo': 'info',
                'titulo': 'Perfil em Aprovação',
                'mensagem': 'Seu perfil está sendo analisado pela nossa equipe. Você receberá uma notificação assim que for aprovado.',
                'acao': 'Aguardando Aprovação',
                'url': None  # Sem URL, apenas mensagem informativa
            }
        elif modelo.status == 'cadastro_inicial':
            status_pagamento = 'pendente_selecao'
            alerta_pagamento = {
                'tipo': 'warning',
                'titulo': 'Selecione um Plano',
                'mensagem': 'Para ativar seu perfil e começar a receber clientes, você precisa selecionar um plano de anúncio.',
                'acao': 'Selecionar Plano',
                'url': 'payments:selecionar_plano'
            }
        elif modelo.status == 'pendente_pagamento':
            status_pagamento = 'pendente_aprovacao'
            alerta_pagamento = {
                'tipo': 'info',
                'titulo': 'Pagamento em Análise',
                'mensagem': 'Seu pagamento está sendo processado. Você receberá uma confirmação em breve.',
                'acao': 'Ver Status',
                'url': 'payments:selecionar_plano'
            }
        elif modelo.plano_ativo:
            status_pagamento = 'ativo'
            plano_ativo = {
                'nome': modelo.plano_ativo.nome,
                'tipo': modelo.plano_ativo.tipo,
                'data_inicio': modelo.plano_data_inicio,
                'data_fim': modelo.plano_data_fim
            }
        
        # Estatísticas básicas
        total_fotos = modelo.fotos.count()
        servicos_ativos = modelo.servicos.filter(ativo=True).count()
        
        context = {
            'modelo': modelo,
            'status_pagamento': status_pagamento,
            'alerta_pagamento': alerta_pagamento,
            'plano_ativo': plano_ativo,
            'total_fotos': total_fotos,
            'servicos_ativos': servicos_ativos,
        }
        
        return render(request, 'accounts/painel_modelo.html', context)
        
    except Modelo.DoesNotExist:
        messages.error(request, 'Perfil de modelo não encontrado.')
        return redirect('core:home')


@login_required
def editar_perfil(request):
    """Editar perfil da modelo"""
    try:
        modelo = request.user.modelo_profile
    except Modelo.DoesNotExist:
        messages.error(request, 'Perfil de modelo não encontrado.')
        return redirect('core:home')
    
    if request.method == 'POST':
        try:
            # Atualizar dados básicos
            modelo.nome = request.POST.get('nome')
            modelo.sobrenome = request.POST.get('sobrenome')
            modelo.cpf = request.POST.get('cpf')
            modelo.data_nascimento = request.POST.get('data_nascimento')
            modelo.nome_exibicao = request.POST.get('nome_exibicao')
            modelo.idade = int(request.POST.get('idade'))
            
            # Atualizar email do usuário
            user = modelo.user
            user.email = request.POST.get('email')
            user.username = request.POST.get('username')
            user.save()
            
            # Contatos
            modelo.telefone = request.POST.get('telefone', '')
            modelo.whatsapp = request.POST.get('whatsapp', '')
            modelo.telegram = request.POST.get('telegram', '')
            modelo.link_contato = request.POST.get('link_contato', '')
            modelo.email_contato = request.POST.get('email_contato', '')
            
            # Localização
            modelo.estado = request.POST.get('estado')
            modelo.cidade = request.POST.get('cidade')
            
            # Características físicas
            modelo.etnia = request.POST.get('etnia')
            modelo.altura = request.POST.get('altura') or None
            modelo.peso = request.POST.get('peso') or None
            
            # Medidas adicionais
            modelo.busto = request.POST.get('busto') or None
            modelo.cintura = request.POST.get('cintura') or None
            modelo.quadril = request.POST.get('quadril') or None
            modelo.calcado = request.POST.get('calcado') or None
            
            # Informações profissionais
            categoria_servico = request.POST.getlist('categoria_servico')
            if categoria_servico:
                modelo.set_categorias_servico(categoria_servico)
            
            modelo.tipo_ensaio = request.POST.get('tipo_ensaio')
            
            local_atendimento = request.POST.getlist('local_atendimento')
            if local_atendimento:
                modelo.set_locais_atendimento(local_atendimento)
            
            modelo.sobre_mim = request.POST.get('sobre_mim', '')
            
            # Biografia (campo adicional)
            modelo.biografia = request.POST.get('biografia', '')
            
            modelo.save()
            
            messages.success(request, 'Perfil atualizado com sucesso!')
            return redirect('accounts:painel_modelo')
            
        except Exception as e:
            messages.error(request, f'Erro ao atualizar perfil: {str(e)}')
    
    return render(request, 'accounts/editar_perfil.html', {'modelo': modelo})


@login_required
def editar_fotos(request):
    """Editar fotos da modelo"""
    try:
        modelo = request.user.modelo_profile
    except Modelo.DoesNotExist:
        messages.error(request, 'Perfil de modelo não encontrado.')
        return redirect('core:home')
    
    if request.method == 'POST':
        try:
            # Upload de nova foto de perfil
            if 'foto_perfil' in request.FILES:
                modelo.foto_perfil = request.FILES['foto_perfil']
                modelo.save()
                messages.success(request, 'Foto de perfil atualizada com sucesso!')
            
            # Processar fotos da galeria
            fotos_galeria = request.FILES.getlist('fotos_galeria')
            categorias_galeria = request.POST.getlist('categorias_galeria')
            autorizar_redes_sociais = request.POST.getlist('autorizar_redes_sociais')
            
            if fotos_galeria:
                # Limitar a 10 fotos na galeria
                fotos_processadas = 0
                for i, foto in enumerate(fotos_galeria):
                    if fotos_processadas >= 10:
                        break
                    
                    # Validar tamanho (máximo 5MB)
                    if foto.size > 5 * 1024 * 1024:
                        continue
                    
                    # Obter categoria (padrão: galeria)
                    categoria = categorias_galeria[i] if i < len(categorias_galeria) else 'galeria'
                    
                    # Verificar se autoriza redes sociais
                    autorizar = f'foto_{i}' in autorizar_redes_sociais
                        
                    try:
                        FotoModelo.objects.create(
                            modelo=modelo,
                            imagem=foto,
                            categoria=categoria,
                            autorizar_redes_sociais=autorizar,
                            ordem=fotos_processadas
                        )
                        fotos_processadas += 1
                    except Exception as e:
                        continue
            
                if fotos_processadas > 0:
                    messages.success(request, f'{fotos_processadas} foto(s) adicionada(s) à galeria!')
            
            # Remover fotos selecionadas
            fotos_remover = request.POST.getlist('fotos_remover')
            if fotos_remover:
                for foto_id in fotos_remover:
                    try:
                        foto = FotoModelo.objects.get(id=foto_id, modelo=modelo)
                        foto.delete()
                    except FotoModelo.DoesNotExist:
                        continue
                
                messages.success(request, 'Fotos removidas com sucesso!')
            
            return redirect('accounts:editar_fotos')
            
        except Exception as e:
            messages.error(request, f'Erro ao atualizar fotos: {str(e)}')
    
    context = {
        'modelo': modelo,
        'fotos_galeria': modelo.fotos.all().order_by('ordem')
    }
    
    return render(request, 'accounts/editar_fotos.html', context)


def cadastro_sucesso(request):
    """View para página de sucesso do cadastro"""
    credentials = request.session.get('cadastro_credentials', {})
    if not credentials:
        return redirect('accounts:cadastro_modelo')
    
    # Limpar credenciais da sessão após exibir
    del request.session['cadastro_credentials']
    
    context = {
        'credentials': credentials
    }
    return render(request, 'accounts/cadastro_sucesso.html', context)


def check_username(request):
    """View para verificar se username está disponível"""
    username = request.GET.get('username', '').strip()
    
    if len(username) < 3:
        return JsonResponse({'available': False, 'message': 'Username deve ter pelo menos 3 caracteres'})
    
    # Verificar se username já existe
    exists = User.objects.filter(username=username).exists()
    
    return JsonResponse({
        'available': not exists,
        'message': 'Username disponível' if not exists else 'Username já existe'
    })


def verificar_email(request):
    """View para verificar se email já existe (AJAX)"""
    if request.method == 'POST':
        try:
            data = json.loads(request.body)
            email = data.get('email')
            
            if email:
                # Verificar se email já existe no User
                email_existe = User.objects.filter(email=email).exists()
                
                return JsonResponse({
                    'email_existe': email_existe,
                    'email': email
                })
            else:
                return JsonResponse({
                    'error': 'Email não fornecido'
                }, status=400)
                
        except json.JSONDecodeError:
            return JsonResponse({
                'error': 'Dados inválidos'
            }, status=400)
        except Exception as e:
            return JsonResponse({
                'error': str(e)
            }, status=500)
    
    return JsonResponse({
        'error': 'Método não permitido'
    }, status=405)


def esqueci_senha(request):
    """View para recuperação de senha"""
    if request.method == 'POST':
        email = request.POST.get('email')
        
        try:
            user = User.objects.get(email=email)
            
            # Gerar nova senha temporária
            nova_senha = get_random_string(8)
            user.set_password(nova_senha)
            user.save()
            
            # Enviar email (em desenvolvimento, apenas mostrar no console)
            send_mail(
                'Recuperação de Senha - Allure',
                f'Sua nova senha temporária é: {nova_senha}\\n\\nPor favor, altere sua senha após fazer login.',
                settings.DEFAULT_FROM_EMAIL,
                [email],
                fail_silently=False,
            )
            
            messages.success(request, 'Nova senha enviada para seu email.')
            return redirect('accounts:login')
            
        except User.DoesNotExist:
            messages.error(request, 'Email não encontrado.')
    
    return render(request, 'accounts/esqueci_senha.html')

