"""
Comando para configurar o sistema de comunicação com templates padrão
"""
from django.core.management.base import BaseCommand
from django.contrib.auth.models import User
from communication.models import EmailTemplate, EmailTrigger
from django.template.loader import render_to_string
import os


class Command(BaseCommand):
    help = 'Configura o sistema de comunicação com templates padrão'

    def handle(self, *args, **options):
        self.stdout.write('Configurando sistema de comunicação...')
        
        # Buscar usuário admin para ser o criador dos templates
        admin_user = User.objects.filter(is_superuser=True).first()
        
        # Templates padrão
        templates_data = [
            {
                'nome': 'Cadastro Realizado com Sucesso',
                'tipo': 'cadastro_sucesso',
                'assunto': 'Bem-vinda ao Allure, {{ nome }}! 🎉',
                'variaveis': ['nome', 'email', 'data_cadastro', 'modelo_id'],
                'template_file': 'emails/cadastro_sucesso.html'
            },
            {
                'nome': 'Perfil Aprovado',
                'tipo': 'perfil_aprovado',
                'assunto': 'Parabéns! Seu perfil foi aprovado 🎉',
                'variaveis': ['nome', 'email', 'data_aprovacao', 'modelo_id'],
                'template_file': 'emails/perfil_aprovado.html'
            },
            {
                'nome': 'Perfil Rejeitado',
                'tipo': 'perfil_rejeitado',
                'assunto': 'Informações sobre seu perfil - Allure',
                'variaveis': ['nome', 'email', 'motivo_rejeicao', 'modelo_id'],
                'template_file': 'emails/perfil_rejeitado.html'
            },
            {
                'nome': 'Pagamento Aprovado',
                'tipo': 'pagamento_aprovado',
                'assunto': 'Pagamento aprovado - {{ descricao }} ✅',
                'variaveis': ['nome', 'email', 'valor', 'descricao', 'data_pagamento', 'tipo_pagamento'],
                'template_file': 'emails/pagamento_aprovado.html'
            },
            {
                'nome': 'Pagamento Pendente',
                'tipo': 'pagamento_pendente',
                'assunto': 'Pagamento em análise - {{ descricao }}',
                'variaveis': ['nome', 'email', 'valor', 'descricao', 'tipo_pagamento'],
                'template_file': 'emails/pagamento_pendente.html'
            },
            {
                'nome': 'Pagamento Falhou',
                'tipo': 'pagamento_falhou',
                'assunto': 'Problema com pagamento - {{ descricao }}',
                'variaveis': ['nome', 'email', 'valor', 'descricao', 'motivo_falha', 'tipo_pagamento'],
                'template_file': 'emails/pagamento_falhou.html'
            },
            {
                'nome': 'Plano Ativado',
                'tipo': 'plano_ativado',
                'assunto': 'Plano ativado com sucesso! 🎯',
                'variaveis': ['nome', 'email', 'plano_nome', 'plano_tipo', 'data_inicio', 'data_fim'],
                'template_file': 'emails/plano_ativado.html'
            },
            {
                'nome': 'Plano Expirando',
                'tipo': 'plano_expirando',
                'assunto': 'Seu plano expira em {{ dias_restantes }} dias ⏰',
                'variaveis': ['nome', 'email', 'plano_nome', 'data_expiracao', 'dias_restantes'],
                'template_file': 'emails/plano_expirando.html'
            },
            {
                'nome': 'Plano Expirado',
                'tipo': 'plano_expirado',
                'assunto': 'Seu plano expirou - Renove agora!',
                'variaveis': ['nome', 'email', 'plano_nome', 'data_expiracao'],
                'template_file': 'emails/plano_expirado.html'
            },
            {
                'nome': 'Boost Ativado',
                'tipo': 'boost_ativado',
                'assunto': 'Boost ativado! Seu perfil está em destaque 🚀',
                'variaveis': ['nome', 'email', 'duracao_dias', 'data_inicio', 'data_fim'],
                'template_file': 'emails/boost_ativado.html'
            },
            {
                'nome': 'Boost Expirado',
                'tipo': 'boost_expirado',
                'assunto': 'Seu boost expirou - Ative novamente!',
                'variaveis': ['nome', 'email', 'data_expiracao'],
                'template_file': 'emails/boost_expirado.html'
            },
            {
                'nome': 'Premium Ativado',
                'tipo': 'premium_ativado',
                'assunto': 'Upgrade Premium ativado! 👑',
                'variaveis': ['nome', 'email', 'plano', 'data_inicio', 'data_fim'],
                'template_file': 'emails/premium_ativado.html'
            },
            {
                'nome': 'Premium Expirado',
                'tipo': 'premium_expirado',
                'assunto': 'Seu Premium expirou - Renove agora!',
                'variaveis': ['nome', 'email', 'data_expiracao'],
                'template_file': 'emails/premium_expirado.html'
            },
            {
                'nome': 'Recuperação de Senha',
                'tipo': 'recuperacao_senha',
                'assunto': 'Redefinir sua senha - Allure',
                'variaveis': ['nome', 'email', 'token', 'link_recuperacao'],
                'template_file': 'emails/recuperacao_senha.html'
            },
            {
                'nome': 'Contato de Cliente',
                'tipo': 'contato_cliente',
                'assunto': 'Nova mensagem de cliente - {{ assunto }}',
                'variaveis': ['nome_cliente', 'email_cliente', 'mensagem', 'assunto', 'data_contato'],
                'template_file': 'emails/contato_cliente.html'
            },
            {
                'nome': 'Bem-vindo',
                'tipo': 'bem_vindo',
                'assunto': 'Bem-vinda ao Allure! 🎉',
                'variaveis': ['nome', 'email'],
                'template_file': 'emails/bem_vindo.html'
            }
        ]
        
        # Criar templates
        templates_criados = 0
        for template_data in templates_data:
            try:
                # Renderizar template HTML
                template_path = f"communication/templates/{template_data['template_file']}"
                conteudo_html = self._render_template(template_path)
                
                # Criar ou atualizar template
                template, created = EmailTemplate.objects.get_or_create(
                    tipo=template_data['tipo'],
                    defaults={
                        'nome': template_data['nome'],
                        'assunto': template_data['assunto'],
                        'conteudo_html': conteudo_html,
                        'variaveis_disponiveis': template_data['variaveis'],
                        'ativo': True,
                        'criado_por': admin_user
                    }
                )
                
                if created:
                    templates_criados += 1
                    self.stdout.write(
                        self.style.SUCCESS(f'✓ Template criado: {template.nome}')
                    )
                else:
                    self.stdout.write(
                        self.style.WARNING(f'⚠ Template já existe: {template.nome}')
                    )
                    
            except Exception as e:
                self.stdout.write(
                    self.style.ERROR(f'✗ Erro ao criar template {template_data["nome"]}: {str(e)}')
                )
        
        # Criar gatilhos padrão
        self.stdout.write('\nCriando gatilhos padrão...')
        from communication.triggers import criar_gatilhos_padrao
        criar_gatilhos_padrao()
        
        self.stdout.write(
            self.style.SUCCESS(
                f'\n✅ Sistema de comunicação configurado com sucesso!\n'
                f'   - {templates_criados} templates criados\n'
                f'   - Gatilhos padrão configurados\n'
                f'   - Sistema pronto para uso!'
            )
        )
    
    def _render_template(self, template_path):
        """Renderiza um template HTML"""
        try:
            # Contexto básico para renderização
            contexto = {
                'nome': '{{ nome }}',
                'email': '{{ email }}',
                'site_url': '{{ site_url }}',
                'site_nome': '{{ site_nome }}',
                'data_atual': '{{ data_atual }}',
                'hora_atual': '{{ hora_atual }}',
            }
            
            # Tentar renderizar o template
            return render_to_string(template_path, contexto)
            
        except Exception as e:
            # Se não conseguir renderizar, retornar HTML básico
            return f"""
            <div style="font-family: Arial, sans-serif; max-width: 600px; margin: 0 auto;">
                <h2>{{ assunto }}</h2>
                <p>Olá {{ nome }},</p>
                <p>Este é um template padrão. O conteúdo específico será adicionado em breve.</p>
                <p>Erro na renderização: {str(e)}</p>
            </div>
            """
