"""
Views para o sistema de comunicação
"""
from django.shortcuts import render, redirect, get_object_or_404
from django.contrib.auth.decorators import login_required, user_passes_test
from django.contrib import messages
from django.http import JsonResponse
from django.core.paginator import Paginator
from django.db.models import Count, Q
from django.utils import timezone
from datetime import timedelta

from .models import EmailTemplate, EmailLog, EmailQueue, EmailTrigger
from .services import EmailService, QueueProcessor


def is_admin(user):
    """Verifica se o usuário é admin"""
    return user.is_authenticated and user.is_staff


@login_required
@user_passes_test(is_admin)
def dashboard_comunicacao(request):
    """Dashboard do sistema de comunicação"""
    try:
        # Estatísticas gerais
        stats = {
            'total_templates': EmailTemplate.objects.count(),
            'templates_ativos': EmailTemplate.objects.filter(ativo=True).count(),
            'total_emails_enviados': EmailLog.objects.filter(status='enviado').count(),
            'emails_hoje': EmailLog.objects.filter(
                data_envio__date=timezone.now().date()
            ).count(),
            'fila_pendente': EmailQueue.objects.filter(status='pendente').count(),
            'fila_falhou': EmailQueue.objects.filter(status='falhou').count(),
        }
        
        # Emails recentes
        emails_recentes = EmailLog.objects.select_related('template').order_by('-data_envio')[:10]
        
        # Fila de processamento
        fila_pendente = EmailQueue.objects.select_related('template').filter(
            status='pendente'
        ).order_by('prioridade', 'data_agendamento')[:10]
        
        # Templates mais usados
        templates_populares = EmailTemplate.objects.annotate(
            total_emails=Count('logs')
        ).order_by('-total_emails')[:5]
        
        context = {
            'stats': stats,
            'emails_recentes': emails_recentes,
            'fila_pendente': fila_pendente,
            'templates_populares': templates_populares,
        }
        
        return render(request, 'communication/dashboard.html', context)
        
    except Exception as e:
        messages.error(request, f'Erro ao carregar dashboard: {str(e)}')
        return redirect('admin:index')


@login_required
@user_passes_test(is_admin)
def preview_template(request, template_id):
    """Preview de um template de email"""
    try:
        template = get_object_or_404(EmailTemplate, id=template_id)
        
        # Contexto de exemplo
        contexto_exemplo = {
            'nome': 'Maria Silva',
            'email': 'maria@exemplo.com',
            'site_url': 'https://allure.com.br',
            'site_nome': 'Allure',
            'data_atual': timezone.now().strftime('%d/%m/%Y'),
            'hora_atual': timezone.now().strftime('%H:%M'),
        }
        
        # Adicionar variáveis específicas do template
        for variavel in template.get_variaveis():
            if variavel not in contexto_exemplo:
                contexto_exemplo[variavel] = f'[Exemplo: {variavel}]'
        
        # Renderizar template
        assunto, conteudo_html, conteudo_texto = EmailService.render_template(
            template, contexto_exemplo
        )
        
        context = {
            'template': template,
            'assunto': assunto,
            'conteudo_html': conteudo_html,
            'conteudo_texto': conteudo_texto,
            'contexto_exemplo': contexto_exemplo,
        }
        
        return render(request, 'communication/preview_template.html', context)
        
    except Exception as e:
        messages.error(request, f'Erro ao gerar preview: {str(e)}')
        return redirect('admin:communication_emailtemplate_changelist')


@login_required
@user_passes_test(is_admin)
def test_email(request, template_id):
    """Enviar email de teste"""
    if request.method == 'POST':
        try:
            template = get_object_or_404(EmailTemplate, id=template_id)
            email_teste = request.POST.get('email_teste')
            
            if not email_teste:
                messages.error(request, 'Email de teste é obrigatório.')
                return redirect('communication:preview_template', template_id=template_id)
            
            # Contexto de exemplo
            contexto_exemplo = {
                'nome': 'Usuário Teste',
                'email': email_teste,
                'site_url': 'https://allure.com.br',
                'site_nome': 'Allure',
                'data_atual': timezone.now().strftime('%d/%m/%Y'),
                'hora_atual': timezone.now().strftime('%H:%M'),
            }
            
            # Adicionar variáveis específicas do template
            for variavel in template.get_variaveis():
                if variavel not in contexto_exemplo:
                    contexto_exemplo[variavel] = f'[Teste: {variavel}]'
            
            # Enviar email
            email_log = EmailService.send_email(
                template=template,
                destinatario_email=email_teste,
                contexto=contexto_exemplo,
                destinatario_nome='Usuário Teste'
            )
            
            if email_log.status == 'enviado':
                messages.success(request, f'Email de teste enviado para {email_teste}')
            else:
                messages.error(request, f'Erro ao enviar email: {email_log.erro_mensagem}')
            
            return redirect('communication:preview_template', template_id=template_id)
            
        except Exception as e:
            messages.error(request, f'Erro ao enviar email de teste: {str(e)}')
            return redirect('communication:preview_template', template_id=template_id)
    
    return redirect('communication:preview_template', template_id=template_id)


@login_required
@user_passes_test(is_admin)
def processar_fila(request):
    """Processar fila de emails via AJAX"""
    if request.method == 'POST':
        try:
            batch_size = int(request.POST.get('batch_size', 10))
            
            stats = QueueProcessor.process_queue(batch_size=batch_size)
            
            return JsonResponse({
                'success': True,
                'stats': stats
            })
            
        except Exception as e:
            return JsonResponse({
                'success': False,
                'error': str(e)
            })
    
    return JsonResponse({'success': False, 'error': 'Método não permitido'})


@login_required
@user_passes_test(is_admin)
def logs_emails(request):
    """Lista de logs de emails com filtros"""
    try:
        # Filtros
        status = request.GET.get('status', '')
        template_id = request.GET.get('template', '')
        data_inicio = request.GET.get('data_inicio', '')
        data_fim = request.GET.get('data_fim', '')
        
        # Query base
        queryset = EmailLog.objects.select_related('template', 'user', 'modelo').order_by('-data_envio')
        
        # Aplicar filtros
        if status:
            queryset = queryset.filter(status=status)
        
        if template_id:
            queryset = queryset.filter(template_id=template_id)
        
        if data_inicio:
            queryset = queryset.filter(data_envio__date__gte=data_inicio)
        
        if data_fim:
            queryset = queryset.filter(data_envio__date__lte=data_fim)
        
        # Paginação
        paginator = Paginator(queryset, 50)
        page_number = request.GET.get('page')
        logs = paginator.get_page(page_number)
        
        # Opções para filtros
        templates = EmailTemplate.objects.filter(ativo=True).order_by('nome')
        status_choices = EmailLog.STATUS_CHOICES
        
        context = {
            'logs': logs,
            'templates': templates,
            'status_choices': status_choices,
            'filtros': {
                'status': status,
                'template': template_id,
                'data_inicio': data_inicio,
                'data_fim': data_fim,
            }
        }
        
        return render(request, 'communication/logs_emails.html', context)
        
    except Exception as e:
        messages.error(request, f'Erro ao carregar logs: {str(e)}')
        return redirect('communication:dashboard')


@login_required
@user_passes_test(is_admin)
def estatisticas(request):
    """Página de estatísticas detalhadas"""
    try:
        # Período padrão: últimos 30 dias
        data_inicio = timezone.now() - timedelta(days=30)
        
        # Estatísticas por status
        stats_status = EmailLog.objects.filter(
            data_envio__gte=data_inicio
        ).values('status').annotate(
            total=Count('id')
        ).order_by('status')
        
        # Estatísticas por template
        stats_template = EmailLog.objects.filter(
            data_envio__gte=data_inicio
        ).values(
            'template__nome', 'template__tipo'
        ).annotate(
            total=Count('id')
        ).order_by('-total')[:10]
        
        # Estatísticas por dia
        stats_diario = EmailLog.objects.filter(
            data_envio__gte=data_inicio
        ).extra(
            select={'dia': 'date(data_envio)'}
        ).values('dia').annotate(
            total=Count('id')
        ).order_by('dia')
        
        # Taxa de entrega
        total_enviados = EmailLog.objects.filter(
            data_envio__gte=data_inicio,
            status='enviado'
        ).count()
        
        total_entregues = EmailLog.objects.filter(
            data_envio__gte=data_inicio,
            status='entregue'
        ).count()
        
        taxa_entrega = (total_entregues / total_enviados * 100) if total_enviados > 0 else 0
        
        context = {
            'stats_status': stats_status,
            'stats_template': stats_template,
            'stats_diario': stats_diario,
            'taxa_entrega': round(taxa_entrega, 2),
            'total_enviados': total_enviados,
            'total_entregues': total_entregues,
            'periodo_dias': 30,
        }
        
        return render(request, 'communication/estatisticas.html', context)
        
    except Exception as e:
        messages.error(request, f'Erro ao carregar estatísticas: {str(e)}')
        return redirect('communication:dashboard')