#!/usr/bin/env python
"""
Script para copiar arquivos de mídia do projeto antigo para o novo
"""
import os
import shutil
from pathlib import Path

# Configurações
OLD_UPLOADS_PATH = '../allure_antigo/uploads'
NEW_MEDIA_PATH = 'media'

def copy_model_photos():
    """Copia as fotos das modelos"""
    print("🔄 Copiando fotos das modelos...")
    
    old_modelos_path = os.path.join(OLD_UPLOADS_PATH, 'modelos')
    new_modelos_path = os.path.join(NEW_MEDIA_PATH, 'modelos')
    
    if not os.path.exists(old_modelos_path):
        print(f"❌ Diretório de modelos antigo não encontrado: {old_modelos_path}")
        return
    
    # Criar diretório de destino
    os.makedirs(new_modelos_path, exist_ok=True)
    
    # Listar todas as pastas de modelos
    for modelo_dir in os.listdir(old_modelos_path):
        modelo_path = os.path.join(old_modelos_path, modelo_dir)
        
        if os.path.isdir(modelo_path):
            print(f"📁 Copiando fotos da modelo {modelo_dir}...")
            
            # Criar diretório de destino para a modelo
            new_modelo_path = os.path.join(new_modelos_path, modelo_dir)
            os.makedirs(new_modelo_path, exist_ok=True)
            
            # Copiar todos os arquivos da pasta
            for file_name in os.listdir(modelo_path):
                if file_name.lower().endswith(('.jpg', '.jpeg', '.png', '.gif', '.webp', '.mp4', '.mov')):
                    old_file = os.path.join(modelo_path, file_name)
                    new_file = os.path.join(new_modelo_path, file_name)
                    
                    if not os.path.exists(new_file):
                        shutil.copy2(old_file, new_file)
                        print(f"  ✅ {file_name}")
    
    print("🎉 Fotos das modelos copiadas com sucesso!")

def copy_banners():
    """Copia os banners"""
    print("🔄 Copiando banners...")
    
    old_banners_path = os.path.join(OLD_UPLOADS_PATH, 'banners')
    new_banners_path = os.path.join(NEW_MEDIA_PATH, 'banners')
    
    if not os.path.exists(old_banners_path):
        print(f"❌ Diretório de banners antigo não encontrado: {old_banners_path}")
        return
    
    # Criar diretório de destino
    os.makedirs(new_banners_path, exist_ok=True)
    
    # Copiar todos os arquivos de banner
    for file_name in os.listdir(old_banners_path):
        if file_name.lower().endswith(('.jpg', '.jpeg', '.png', '.gif', '.webp')):
            old_file = os.path.join(old_banners_path, file_name)
            new_file = os.path.join(new_banners_path, file_name)
            
            if not os.path.exists(new_file):
                shutil.copy2(old_file, new_file)
                print(f"  ✅ {file_name}")
    
    print("🎉 Banners copiados com sucesso!")

def copy_documents():
    """Copia os documentos das modelos"""
    print("🔄 Copiando documentos...")
    
    old_modelos_path = os.path.join(OLD_UPLOADS_PATH, 'modelos')
    new_docs_path = os.path.join(NEW_MEDIA_PATH, 'modelos', 'documentos')
    
    if not os.path.exists(old_modelos_path):
        print(f"❌ Diretório de modelos antigo não encontrado: {old_modelos_path}")
        return
    
    # Criar diretório de destino
    os.makedirs(new_docs_path, exist_ok=True)
    
    # Procurar por documentos na pasta raiz de modelos
    for file_name in os.listdir(old_modelos_path):
        if file_name.startswith(('doc_', 'selfie_')) and file_name.lower().endswith(('.jpg', '.jpeg', '.png', '.heic')):
            old_file = os.path.join(old_modelos_path, file_name)
            new_file = os.path.join(new_docs_path, file_name)
            
            if not os.path.exists(new_file):
                shutil.copy2(old_file, new_file)
                print(f"  ✅ {file_name}")
    
    print("🎉 Documentos copiados com sucesso!")

def main():
    """Função principal"""
    print("🚀 Iniciando cópia de arquivos de mídia...")
    print("=" * 50)
    
    try:
        copy_model_photos()
        copy_banners()
        copy_documents()
        
        print("\n" + "=" * 50)
        print("🎉 Cópia de arquivos concluída com sucesso!")
        print("📁 Arquivos copiados para:")
        print(f"   - Fotos: {os.path.join(NEW_MEDIA_PATH, 'modelos')}")
        print(f"   - Banners: {os.path.join(NEW_MEDIA_PATH, 'banners')}")
        print(f"   - Documentos: {os.path.join(NEW_MEDIA_PATH, 'modelos', 'documentos')}")
        
    except Exception as e:
        print(f"❌ Erro durante a cópia: {e}")
        return False
    
    return True

if __name__ == '__main__':
    main() 