from django.core.management.base import BaseCommand
from django.contrib.auth.models import User
from models.models import Modelo


class Command(BaseCommand):
    help = 'Remove usuários que não têm modelo correspondente (usuários órfãos)'

    def add_arguments(self, parser):
        parser.add_argument(
            '--dry-run',
            action='store_true',
            help='Mostra o que seria deletado sem executar a exclusão',
        )
        parser.add_argument(
            '--force',
            action='store_true',
            help='Força a exclusão sem confirmação',
        )

    def handle(self, *args, **options):
        dry_run = options['dry_run']
        force = options['force']
        
        # Encontrar usuários órfãos
        usuarios_orfos = []
        for user in User.objects.all():
            if not Modelo.objects.filter(user=user).exists():
                usuarios_orfos.append(user)
        
        if not usuarios_orfos:
            self.stdout.write(
                self.style.SUCCESS('✅ Nenhum usuário órfão encontrado!')
            )
            return
        
        # Mostrar usuários que seriam deletados
        self.stdout.write(
            self.style.WARNING(f'🔍 Encontrados {len(usuarios_orfos)} usuários órfãos:')
        )
        
        for user in usuarios_orfos:
            self.stdout.write(
                f'  • ID: {user.id} | Username: {user.username} | Email: {user.email} | Data: {user.date_joined.strftime("%d/%m/%Y")}'
            )
        
        if dry_run:
            self.stdout.write(
                self.style.SUCCESS(f'\n📋 Modo DRY-RUN: Nenhum usuário foi deletado')
            )
            return
        
        # Confirmação
        if not force:
            confirm = input(f'\n❓ Tem certeza que deseja deletar {len(usuarios_orfos)} usuários órfãos? (sim/não): ')
            if confirm.lower() not in ['sim', 's', 'yes', 'y']:
                self.stdout.write(
                    self.style.WARNING('❌ Operação cancelada pelo usuário')
                )
                return
        
        # Executar exclusão
        try:
            for user in usuarios_orfos:
                username = user.username
                email = user.email
                user.delete()
                self.stdout.write(
                    f'🗑️  Usuário deletado: {username} ({email})'
                )
            
            self.stdout.write(
                self.style.SUCCESS(f'\n✅ Sucesso! {len(usuarios_orfos)} usuários órfãos foram deletados')
            )
            
            # Mostrar estatísticas finais
            total_users = User.objects.count()
            total_modelos = Modelo.objects.count()
            self.stdout.write(f'\n📊 Estatísticas finais:')
            self.stdout.write(f'  • Usuários restantes: {total_users}')
            self.stdout.write(f'  • Modelos existentes: {total_modelos}')
            
        except Exception as e:
            self.stdout.write(
                self.style.ERROR(f'❌ Erro ao deletar usuários: {str(e)}')
            )
