from django.core.management.base import BaseCommand
from models.models import PlanoAnuncio
from payments.models import PlanoAnuncioPagamento, TransacaoPagamento


class Command(BaseCommand):
    help = 'Testa se o sistema de planos está funcionando corretamente'

    def handle(self, *args, **options):
        self.stdout.write(
            self.style.SUCCESS('🧪 Iniciando testes do sistema de planos...')
        )
        
        # Teste 1: Verificar se os planos existem
        self.stdout.write('\n📋 Teste 1: Verificando planos existentes...')
        try:
            planos = PlanoAnuncio.objects.all()
            self.stdout.write(
                self.style.SUCCESS(f'✅ Encontrados {planos.count()} planos:')
            )
            for plano in planos:
                self.stdout.write(f'   • {plano.nome} - R$ {plano.preco} - {plano.duracao_dias} dias')
        except Exception as e:
            self.stdout.write(
                self.style.ERROR(f'❌ Erro ao buscar planos: {str(e)}')
            )
        
        # Teste 2: Verificar se o admin está funcionando
        self.stdout.write('\n🔧 Teste 2: Verificando registro no admin...')
        try:
            from django.contrib.admin import site
            if PlanoAnuncio in site._registry:
                self.stdout.write(
                    self.style.SUCCESS('✅ PlanoAnuncio registrado no admin padrão')
                )
            else:
                self.stdout.write(
                    self.style.WARNING('⚠️ PlanoAnuncio NÃO registrado no admin padrão')
                )
        except Exception as e:
            self.stdout.write(
                self.style.ERROR(f'❌ Erro ao verificar admin: {str(e)}')
            )
        
        # Teste 3: Verificar se o modelo está funcionando
        self.stdout.write('\n📊 Teste 3: Verificando funcionalidades do modelo...')
        try:
            plano_teste = PlanoAnuncio.objects.first()
            if plano_teste:
                preco_mensal = plano_teste.get_preco_mensal()
                self.stdout.write(
                    self.style.SUCCESS(f'✅ Plano teste: {plano_teste.nome}')
                )
                self.stdout.write(f'   • Preço mensal: R$ {preco_mensal:.2f}')
                self.stdout.write(f'   • Destaque: {"Sim" if plano_teste.destaque_perfil else "Não"}')
                self.stdout.write(f'   • Boost: {"Sim" if plano_teste.boost_visualizacoes else "Não"}')
            else:
                self.stdout.write(
                    self.style.WARNING('⚠️ Nenhum plano encontrado para teste')
                )
        except Exception as e:
            self.stdout.write(
                self.style.ERROR(f'❌ Erro ao testar modelo: {str(e)}')
            )
        
        # Teste 4: Verificar URLs
        self.stdout.write('\n🌐 Teste 4: Verificando URLs...')
        try:
            from django.urls import reverse
            from django.urls.exceptions import NoReverseMatch
            
            try:
                url_planos = reverse('payments:selecionar_plano')
                self.stdout.write(
                    self.style.SUCCESS(f'✅ URL de planos: {url_planos}')
                )
            except NoReverseMatch:
                self.stdout.write(
                    self.style.ERROR('❌ URL de planos não encontrada')
                )
                
            try:
                url_admin = reverse('admin:models_planoanuncio_changelist')
                self.stdout.write(
                    self.style.SUCCESS(f'✅ URL do admin: {url_admin}')
                )
            except NoReverseMatch:
                self.stdout.write(
                    self.style.WARNING('⚠️ URL do admin não encontrada (pode ser normal)')
                )
                
        except Exception as e:
            self.stdout.write(
                self.style.ERROR(f'❌ Erro ao verificar URLs: {str(e)}')
            )
        
        self.stdout.write(
            self.style.SUCCESS('\n🎉 Testes concluídos!')
        )
