from django.core.management.base import BaseCommand
from django.contrib.auth.models import User
from django.db.models import Count
from models.models import Modelo


class Command(BaseCommand):
    help = 'Verifica emails e CPFs duplicados no sistema'

    def handle(self, *args, **options):
        self.stdout.write(self.style.SUCCESS('🔍 Verificando duplicatas no sistema...\n'))
        
        # Verificar emails duplicados na tabela User
        self.stdout.write(self.style.WARNING('📧 Verificando emails duplicados na tabela User:'))
        emails_duplicados_user = User.objects.values('email').annotate(
            count=Count('email')
        ).filter(count__gt=1)
        
        if emails_duplicados_user:
            for item in emails_duplicados_user:
                email = item['email']
                count = item['count']
                self.stdout.write(f'  • Email: {email} - {count} usuários')
                
                # Mostrar usuários com este email
                usuarios = User.objects.filter(email=email)
                for user in usuarios:
                    self.stdout.write(f'    - ID: {user.id} | Username: {user.username} | Ativo: {user.is_active}')
        else:
            self.stdout.write('  ✅ Nenhum email duplicado encontrado na tabela User')
        
        # Verificar CPFs duplicados na tabela Modelo
        self.stdout.write(self.style.WARNING('\n🆔 Verificando CPFs duplicados na tabela Modelo:'))
        cpfs_duplicados = Modelo.objects.values('cpf').annotate(
            count=Count('cpf')
        ).filter(count__gt=1)
        
        if cpfs_duplicados:
            for item in cpfs_duplicados:
                cpf = item['cpf']
                count = item['count']
                self.stdout.write(f'  • CPF: {cpf} - {count} modelos')
                
                # Mostrar modelos com este CPF
                modelos = Modelo.objects.filter(cpf=cpf)
                for modelo in modelos:
                    self.stdout.write(f'    - ID: {modelo.id} | Nome: {modelo.nome} | Status: {modelo.status}')
        else:
            self.stdout.write('  ✅ Nenhum CPF duplicado encontrado na tabela Modelo')
        
        # Verificar emails duplicados na tabela Modelo
        self.stdout.write(self.style.WARNING('\n📧 Verificando emails duplicados na tabela Modelo:'))
        emails_duplicados_modelo = Modelo.objects.values('email_contato').annotate(
            count=Count('email_contato')
        ).filter(count__gt=1)
        
        if emails_duplicados_modelo:
            for item in emails_duplicados_modelo:
                email = item['email_contato']
                count = item['count']
                self.stdout.write(f'  • Email: {email} - {count} modelos')
                
                # Mostrar modelos com este email
                modelos = Modelo.objects.filter(email_contato=email)
                for modelo in modelos:
                    self.stdout.write(f'    - ID: {modelo.id} | Nome: {modelo.nome} | Status: {modelo.status}')
        else:
            self.stdout.write('  ✅ Nenhum email duplicado encontrado na tabela Modelo')
        
        # Verificar conflitos entre User e Modelo
        self.stdout.write(self.style.WARNING('\n⚠️  Verificando conflitos entre User e Modelo:'))
        conflitos = 0
        
        for user in User.objects.all():
            if user.email:
                modelos_com_email = Modelo.objects.filter(email_contato=user.email)
                if modelos_com_email.exists():
                    conflitos += 1
                    self.stdout.write(f'  • Conflito: Email {user.email} usado por:')
                    self.stdout.write(f'    - User: {user.username} (ID: {user.id})')
                    for modelo in modelos_com_email:
                        self.stdout.write(f'    - Modelo: {modelo.nome} (ID: {modelo.id})')
        
        if conflitos == 0:
            self.stdout.write('  ✅ Nenhum conflito encontrado entre User e Modelo')
        
        # Estatísticas finais
        self.stdout.write(self.style.SUCCESS('\n📊 Estatísticas do sistema:'))
        self.stdout.write(f'  • Total de usuários: {User.objects.count()}')
        self.stdout.write(f'  • Total de modelos: {Modelo.objects.count()}')
        self.stdout.write(f'  • Usuários ativos: {User.objects.filter(is_active=True).count()}')
        self.stdout.write(f'  • Usuários inativos: {User.objects.filter(is_active=False).count()}')
        
        # Usuários órfãos
        usuarios_orfos = 0
        for user in User.objects.all():
            if not Modelo.objects.filter(user=user).exists():
                usuarios_orfos += 1
        
        self.stdout.write(f'  • Usuários órfãos: {usuarios_orfos}')
        
        if usuarios_orfos > 0:
            self.stdout.write(
                self.style.WARNING(f'\n💡 Dica: Use o comando "python manage.py limpar_usuarios_orfos" para limpar usuários órfãos')
            )
