from django.test import TestCase, Client
from django.urls import reverse
from django.contrib.auth.models import User
from django.core.files.uploadedfile import SimpleUploadedFile
from django.db import IntegrityError
from models.models import Modelo, FotoModelo, Banner
from core.forms import ModeloRegistrationForm, FotoGaleriaForm


class CoreViewsTestCase(TestCase):
    def setUp(self):
        self.client = Client()
        self.user = User.objects.create_user(
            username='testuser',
            email='test@example.com',
            password='testpass123'
        )
        
        # Criar modelo de teste
        self.modelo = Modelo.objects.create(
            user=self.user,
            nome='Nome',
            sobrenome='Teste',
            cpf='123.456.789-00',
            data_nascimento='1998-01-01',
            nome_exibicao='Modelo Teste',
            idade=25,
            altura=170,
            peso=60,
            cidade='São Paulo',
            estado='SP',
            sobre_mim='Descrição de teste',
            categoria_servico='["JOB"]',
            etnia='branca',
            tipo_ensaio='solo',
            local_atendimento='["local_proprio"]',
            status='aprovado'
        )
        
        # Criar foto de teste
        self.image = SimpleUploadedFile(
            "test_image.jpg",
            b"fake-image-content",
            content_type="image/jpeg"
        )
        self.foto = FotoModelo.objects.create(
            modelo=self.modelo,
            imagem=self.image,
            categoria='galeria'
        )

    def test_home_view(self):
        """Testa a página inicial"""
        response = self.client.get(reverse('core:home'))
        self.assertEqual(response.status_code, 200)
        self.assertTemplateUsed(response, 'core/home.html')

    def test_modelos_list_view(self):
        """Testa a listagem de modelos"""
        response = self.client.get(reverse('core:modelos_list'))
        self.assertEqual(response.status_code, 200)
        self.assertTemplateUsed(response, 'core/modelos_list.html')
        # Verificar se o contexto contém page_obj (paginação)
        self.assertIn('page_obj', response.context)

    def test_modelo_detail_view(self):
        """Testa os detalhes de um modelo"""
        response = self.client.get(reverse('core:modelo_detail', args=[self.modelo.slug]))
        self.assertEqual(response.status_code, 200)
        self.assertTemplateUsed(response, 'core/modelo_detail.html')
        self.assertEqual(response.context['modelo'], self.modelo)

    def test_modelo_detail_view_inactive(self):
        """Testa acesso a modelo inativo"""
        self.modelo.status = 'suspenso'
        self.modelo.save()
        response = self.client.get(reverse('core:modelo_detail', args=[self.modelo.slug]))
        self.assertEqual(response.status_code, 404)

    def test_seja_modelo_view_get(self):
        """Testa a página de cadastro de modelo (GET)"""
        response = self.client.get(reverse('core:seja_modelo'))
        self.assertEqual(response.status_code, 200)
        self.assertTemplateUsed(response, 'core/seja_modelo.html')

    def test_seja_modelo_view_post(self):
        """Testa o cadastro de um novo modelo"""
        image = SimpleUploadedFile(
            "test_image.jpg",
            b"fake-image-content",
            content_type="image/jpeg"
        )
        
        data = {
            'username': 'novomodelo',
            'email': 'novo@example.com',
            'password1': 'testpass123',
            'password2': 'testpass123',
            'nome_exibicao': 'Novo Modelo',
            'idade': 25,
            'estado': 'SP',
            'cidade': 'São Paulo',
            'etnia': 'branca',
            'categoria': 'JOB',
            'tipo_ensaio': 'solo',
            'local_atendimento': 'local_proprio',
            'sobre_mim': 'Nova descrição',
            'aceito_termos': True,
        }
        
        files = {
            'foto_perfil': image,
            'documento_rg': image,
            'documento_selfie': image,
        }
        
        response = self.client.post(reverse('core:seja_modelo'), data, files=files)
        self.assertEqual(response.status_code, 302)  # Redirecionamento após sucesso
        
        # Verificar se o modelo foi criado
        novo_modelo = Modelo.objects.filter(nome_exibicao='Novo Modelo').first()
        self.assertIsNotNone(novo_modelo)

    def test_search_modelos(self):
        """Testa a busca de modelos"""
        response = self.client.get(reverse('core:modelos_list'), {'q': 'Teste'})
        self.assertEqual(response.status_code, 200)
        self.assertIn('page_obj', response.context)

    def test_filter_modelos_by_category(self):
        """Testa o filtro por categoria"""
        response = self.client.get(reverse('core:modelos_list'), {'categoria': 'JOB'})
        self.assertEqual(response.status_code, 200)
        self.assertIn('page_obj', response.context)


class CoreFormsTestCase(TestCase):
    def setUp(self):
        self.user = User.objects.create_user(
            username='testuser',
            email='test@example.com',
            password='testpass123'
        )

    def test_modelo_form_valid(self):
        """Testa formulário de modelo válido"""
        form_data = {
            'username': 'testuser2',
            'email': 'test2@example.com',
            'password1': 'testpass123',
            'password2': 'testpass123',
            'nome_exibicao': 'Modelo Teste',
            'idade': 25,
            'estado': 'SP',
            'cidade': 'São Paulo',
            'etnia': 'branca',
            'categoria': 'JOB',
            'tipo_ensaio': 'solo',
            'local_atendimento': 'local_proprio',
            'aceito_termos': True,
        }
        
        image = SimpleUploadedFile(
            "test_image.jpg",
            b"fake-image-content",
            content_type="image/jpeg"
        )
        
        files = {
            'foto_perfil': image,
            'documento_rg': image,
            'documento_selfie': image,
        }
        
        form = ModeloRegistrationForm(data=form_data, files=files)
        self.assertTrue(form.is_valid())

    def test_modelo_form_invalid_age(self):
        """Testa validação de idade inválida"""
        form_data = {
            'username': 'testuser2',
            'email': 'test2@example.com',
            'password1': 'testpass123',
            'password2': 'testpass123',
            'nome_exibicao': 'Modelo Teste',
            'idade': 16,  # Idade inválida
            'estado': 'SP',
            'cidade': 'São Paulo',
            'etnia': 'branca',
            'categoria': 'JOB',
            'tipo_ensaio': 'solo',
            'local_atendimento': 'local_proprio',
            'aceito_termos': True,
        }
        
        image = SimpleUploadedFile(
            "test_image.jpg",
            b"fake-image-content",
            content_type="image/jpeg"
        )
        
        files = {
            'foto_perfil': image,
            'documento_rg': image,
            'documento_selfie': image,
        }
        
        form = ModeloRegistrationForm(data=form_data, files=files)
        self.assertFalse(form.is_valid())
        self.assertIn('idade', form.errors)

    def test_modelo_form_missing_required_fields(self):
        """Testa formulário com campos obrigatórios faltando"""
        form_data = {
            'username': 'testuser2',
            'email': 'test2@example.com',
            'password1': 'testpass123',
            'password2': 'testpass123',
            # Faltando campos obrigatórios
        }
        
        form = ModeloRegistrationForm(data=form_data)
        self.assertFalse(form.is_valid())

    def test_foto_form_valid(self):
        """Testa formulário de foto válido"""
        user = User.objects.create_user(username='testuser3', password='testpass123')
        modelo = Modelo.objects.create(
            user=user,
            nome_exibicao='Modelo Teste',
            idade=25,
            estado='SP',
            cidade='São Paulo',
            categoria='JOB',
            etnia='branca',
            tipo_ensaio='solo',
            local_atendimento='local_proprio',
        )
        
        image = SimpleUploadedFile(
            "test_image.jpg",
            b"fake-image-content",
            content_type="image/jpeg"
        )
        
        form_data = {
            'categoria': 'galeria',
        }
        
        form = FotoGaleriaForm(data=form_data, files={'imagem': image})
        self.assertTrue(form.is_valid())


class CoreModelsTestCase(TestCase):
    def setUp(self):
        self.user = User.objects.create_user(
            username='testuser',
            email='test@example.com',
            password='testpass123'
        )

    def test_modelo_creation(self):
        """Testa criação de modelo"""
        modelo = Modelo.objects.create(
            user=self.user,
            nome_exibicao='Modelo Teste',
            idade=25,
            altura=170,
            peso=60,
            cidade='São Paulo',
            estado='SP',
            sobre_mim='Descrição de teste',
            categoria='JOB',
            etnia='branca',
            tipo_ensaio='solo',
            local_atendimento='local_proprio',
        )
        self.assertEqual(modelo.nome_exibicao, 'Modelo Teste')
        self.assertEqual(modelo.status, 'pendente')

    def test_modelo_str_representation(self):
        """Testa representação string do modelo"""
        modelo = Modelo.objects.create(
            user=self.user,
            nome_exibicao='Modelo Teste',
            idade=25,
            estado='SP',
            cidade='São Paulo',
            categoria='JOB',
            etnia='branca',
            tipo_ensaio='solo',
            local_atendimento='local_proprio',
        )
        self.assertEqual(str(modelo), 'Modelo Teste')

    def test_banner_str_representation(self):
        """Testa representação string do banner"""
        banner = Banner.objects.create(
            titulo='Banner Teste',
            imagem=SimpleUploadedFile("banner.jpg", b"fake-content", content_type="image/jpeg"),
            link='https://example.com',
            ordem=1
        )
        self.assertEqual(str(banner), 'Banner Teste')


class CoreIntegrationTestCase(TestCase):
    def setUp(self):
        self.client = Client()
        self.user = User.objects.create_user(
            username='testuser',
            email='test@example.com',
            password='testpass123'
        )

    def test_complete_modelo_workflow(self):
        """Testa o fluxo completo de cadastro e visualização de modelo"""
        # Login
        self.client.login(username='testuser', password='testpass123')
        
        # Criar modelo
        modelo = Modelo.objects.create(
            user=self.user,
            nome='Nome',
            sobrenome='Teste',
            cpf='123.456.789-01',
            data_nascimento='1998-01-01',
            nome_exibicao='Modelo Teste',
            idade=25,
            estado='SP',
            cidade='São Paulo',
            categoria_servico='["JOB"]',
            etnia='branca',
            tipo_ensaio='solo',
            local_atendimento='["local_proprio"]',
            status='aprovado'
        )
        
        # Acessar detalhes do modelo
        response = self.client.get(reverse('core:modelo_detail', args=[modelo.slug]))
        self.assertEqual(response.status_code, 200)
        self.assertEqual(response.context['modelo'], modelo)

    def test_unauthorized_access(self):
        """Testa acesso não autorizado"""
        # Tentar acessar página que requer login sem estar logado
        response = self.client.get(reverse('core:modelo_detail', args=['modelo-inexistente']))
        self.assertEqual(response.status_code, 404)  # Modelo não existe


class CoreSecurityTestCase(TestCase):
    def setUp(self):
        self.client = Client()
        self.user = User.objects.create_user(
            username='testuser',
            email='test@example.com',
            password='testpass123'
        )

    def test_xss_protection(self):
        """Testa proteção contra XSS"""
        # Testar com dados que poderiam conter XSS
        form_data = {
            'username': 'testuser2',
            'email': 'test2@example.com',
            'password1': 'testpass123',
            'password2': 'testpass123',
            'nome_exibicao': '<script>alert("xss")</script>',
            'idade': 25,
            'estado': 'SP',
            'cidade': 'São Paulo',
            'etnia': 'branca',
            'categoria': 'JOB',
            'tipo_ensaio': 'solo',
            'local_atendimento': 'local_proprio',
            'aceito_termos': True,
        }
        
        image = SimpleUploadedFile(
            "test_image.jpg",
            b"fake-image-content",
            content_type="image/jpeg"
        )
        
        files = {
            'foto_perfil': image,
            'documento_rg': image,
            'documento_selfie': image,
        }
        
        form = ModeloRegistrationForm(data=form_data, files=files)
        # O formulário deve ser válido, mas o XSS será escapado no template
        self.assertTrue(form.is_valid())

    def test_csrf_protection(self):
        """Testa proteção CSRF"""
        # Testar POST sem CSRF token
        data = {
            'username': 'testuser2',
            'email': 'test2@example.com',
            'password1': 'testpass123',
            'password2': 'testpass123',
            'nome_exibicao': 'Modelo Teste',
            'idade': 25,
            'estado': 'SP',
            'cidade': 'São Paulo',
            'etnia': 'branca',
            'categoria': 'JOB',
            'tipo_ensaio': 'solo',
            'local_atendimento': 'local_proprio',
            'aceito_termos': True,
        }
        
        # Usar client sem CSRF para testar
        client = Client(enforce_csrf_checks=True)
        response = client.post(reverse('core:seja_modelo'), data)
        self.assertEqual(response.status_code, 403)  # CSRF error
