#!/usr/bin/env python
"""
Script para debugar os modelos e seus status no banco de dados
"""
import os
import django

# Configurar Django
os.environ.setdefault('DJANGO_SETTINGS_MODULE', 'allure_project.settings')
django.setup()

from models.models import Modelo

def debug_modelos():
    """Debuga os modelos e seus status"""
    print("🔍 DEBUG DOS MODELOS NO BANCO DE DADOS")
    print("=" * 50)
    
    # Contar total de modelos
    total = Modelo.objects.count()
    print(f"📊 Total de modelos: {total}")
    
    if total == 0:
        print("❌ Nenhum modelo encontrado no banco de dados!")
        return
    
    # Contar por status
    print("\n📈 Contagem por status:")
    status_counts = {}
    for modelo in Modelo.objects.all():
        status = modelo.status
        if status not in status_counts:
            status_counts[status] = 0
        status_counts[status] += 1
    
    for status, count in status_counts.items():
        print(f"  - {status}: {count}")
    
    # Mostrar modelos pendentes
    print("\n⏳ Modelos pendentes (detalhado):")
    pendentes = Modelo.objects.filter(
        status__in=['cadastro_inicial', 'pendente_validacao', 'pendente_pagamento']
    )
    
    if pendentes.exists():
        for modelo in pendentes:
            print(f"  - {modelo.nome_exibicao} (ID: {modelo.id}) - Status: {modelo.status}")
    else:
        print("  ❌ Nenhum modelo pendente encontrado!")
    
    # Mostrar todos os modelos
    print("\n📋 Todos os modelos:")
    for modelo in Modelo.objects.all()[:10]:  # Mostrar apenas os primeiros 10
        print(f"  - {modelo.nome_exibicao} (ID: {modelo.id}) - Status: {modelo.status}")
    
    if total > 10:
        print(f"  ... e mais {total - 10} modelos")

def test_filter():
    """Testa o filtro usado no dashboard"""
    print("\n🧪 TESTE DO FILTRO DO DASHBOARD")
    print("=" * 30)
    
    # Testar o filtro exato usado no AdminSite
    pendentes = Modelo.objects.filter(
        status__in=['cadastro_inicial', 'pendente_validacao', 'pendente_pagamento']
    )
    
    print(f"Modelos pendentes (filtro do dashboard): {pendentes.count()}")
    
    # Testar filtros individuais
    for status in ['cadastro_inicial', 'pendente_validacao', 'pendente_pagamento']:
        count = Modelo.objects.filter(status=status).count()
        print(f"  - {status}: {count}")
    
    # Testar o filtro usado no link
    pendentes_validacao = Modelo.objects.filter(status='pendente_validacao')
    print(f"\nModelos com status 'pendente_validacao': {pendentes_validacao.count()}")
    
    if pendentes_validacao.exists():
        for modelo in pendentes_validacao:
            print(f"  - {modelo.nome_exibicao} (ID: {modelo.id})")

def create_test_modelo():
    """Cria uma modelo de teste se não existir"""
    print("\n➕ CRIANDO MODELO DE TESTE")
    print("=" * 30)
    
    from django.contrib.auth.models import User
    
    # Verificar se já existe uma modelo de teste
    if Modelo.objects.filter(nome_exibicao='Modelo Teste').exists():
        print("✅ Modelo de teste já existe")
        return
    
    # Criar usuário de teste
    try:
        user = User.objects.create_user(
            username='teste_modelo',
            email='teste@modelo.com',
            password='teste123'
        )
        print("✅ Usuário de teste criado")
    except:
        user = User.objects.get(username='teste_modelo')
        print("✅ Usuário de teste já existe")
    
    # Criar modelo de teste
    from datetime import date
    modelo = Modelo.objects.create(
        user=user,
        nome='Teste',
        sobrenome='Modelo',
        cpf='12345678901',
        data_nascimento=date(1990, 1, 1),
        nome_exibicao='Modelo Teste',
        idade=30,
        estado='SP',
        cidade='São Paulo',
        etnia='branca',
        categoria_servico='JOB',
        tipo_ensaio='solo',
        local_atendimento='local_proprio',
        status='pendente_validacao'  # Status pendente para teste
    )
    
    print(f"✅ Modelo de teste criada: {modelo.nome_exibicao} (Status: {modelo.status})")

def main():
    """Função principal"""
    print("🎭 DEBUG DOS MODELOS - PAINEL ADMINISTRATIVO")
    print("=" * 60)
    
    # Debug dos modelos existentes
    debug_modelos()
    
    # Testar filtros
    test_filter()
    
    # Criar modelo de teste se necessário
    create_test_modelo()
    
    print("\n" + "=" * 60)
    print("🔧 PRÓXIMOS PASSOS:")
    print("1. Verifique se há modelos no banco de dados")
    print("2. Confirme se os status estão corretos")
    print("3. Teste o link 'Revisar Pendentes' novamente")
    print("4. Se não houver modelos, crie alguns para teste")

if __name__ == "__main__":
    main() 