#!/usr/bin/env python
"""
Exemplo de uso do Sistema de Testes Allure
Demonstra como executar diferentes tipos de testes
"""

import os
import sys
from pathlib import Path

# Adicionar o diretório do projeto ao path
BASE_DIR = Path(__file__).resolve().parent
sys.path.insert(0, str(BASE_DIR))

# Configurações do Django
os.environ.setdefault('DJANGO_SETTINGS_MODULE', 'allure_project.settings')

import django
django.setup()


def exemplo_testes_unitarios():
    """Exemplo de como executar testes unitários específicos"""
    print("🧪 EXEMPLO: Testes Unitários")
    print("=" * 50)
    
    print("\n1. Executar todos os testes unitários:")
    print("   python manage.py test")
    
    print("\n2. Executar testes de um app específico:")
    print("   python manage.py test core")
    print("   python manage.py test accounts")
    print("   python manage.py test models")
    
    print("\n3. Executar um teste específico:")
    print("   python manage.py test core.tests.CoreViewsTestCase.test_home_view")
    
    print("\n4. Executar com verbosidade:")
    print("   python manage.py test --verbosity=2")
    
    print("\n5. Executar e gerar relatório:")
    print("   python manage.py test --verbosity=2 > test_output.txt")


def exemplo_testes_funcionalidade():
    """Exemplo de como executar testes de funcionalidade"""
    print("\n🔍 EXEMPLO: Testes de Funcionalidade")
    print("=" * 50)
    
    print("\n1. Executar testes de funcionalidade:")
    print("   python test_functionality.py")
    
    print("\n2. Verificar relatórios gerados:")
    print("   - functionality_report.json")
    print("   - functionality_report.txt")


def exemplo_suite_completa():
    """Exemplo de como executar a suíte completa"""
    print("\n🚀 EXEMPLO: Suíte Completa de Testes")
    print("=" * 50)
    
    print("\n1. Executar todos os tipos de testes:")
    print("   python run_all_tests.py")
    
    print("\n2. Verificar relatórios consolidados:")
    print("   - consolidated_test_report.json")
    print("   - consolidated_test_report.txt")


def exemplo_testes_manuais():
    """Exemplo de testes manuais que podem ser executados"""
    print("\n👨‍💻 EXEMPLO: Testes Manuais")
    print("=" * 50)
    
    print("\n1. Testar configurações do projeto:")
    print("   python -c \"import django; django.setup(); from django.conf import settings; print('Apps:', settings.INSTALLED_APPS)\"")
    
    print("\n2. Testar conexão com banco:")
    print("   python manage.py dbshell")
    
    print("\n3. Testar URLs:")
    print("   python manage.py check --deploy")
    
    print("\n4. Verificar migrações:")
    print("   python manage.py showmigrations")


def exemplo_ci_cd():
    """Exemplo de configuração para CI/CD"""
    print("\n🔄 EXEMPLO: Configuração CI/CD")
    print("=" * 50)
    
    print("\n1. GitHub Actions (.github/workflows/tests.yml):")
    print("""
name: Testes Allure

on: [push, pull_request]

jobs:
  test:
    runs-on: ubuntu-latest
    
    steps:
    - uses: actions/checkout@v2
    
    - name: Configurar Python
      uses: actions/setup-python@v2
      with:
        python-version: 3.9
    
    - name: Instalar dependências
      run: |
        pip install -r requirements.txt
    
    - name: Executar testes
      run: |
        python run_all_tests.py
    
    - name: Upload relatórios
      uses: actions/upload-artifact@v2
      with:
        name: test-reports
        path: |
          consolidated_test_report.json
          consolidated_test_report.txt
    """)
    
    print("\n2. Execução local para desenvolvimento:")
    print("   # Adicionar ao .git/hooks/pre-commit:")
    print("   #!/bin/sh")
    print("   python run_all_tests.py")


def exemplo_interpretacao_resultados():
    """Exemplo de como interpretar os resultados"""
    print("\n📊 EXEMPLO: Interpretação de Resultados")
    print("=" * 50)
    
    print("\n1. Taxa de Sucesso:")
    print("   ✅ > 95%: Excelente")
    print("   ⚠️  80-95%: Bom, mas precisa melhorar")
    print("   ❌ < 80%: Precisa de atenção urgente")
    
    print("\n2. Tipos de Problemas:")
    print("   🔴 Erro: Problema crítico que impede execução")
    print("   🟡 Falha: Teste falhou, mas sistema funcionou")
    print("   🟢 Sucesso: Tudo funcionando corretamente")
    
    print("\n3. Ações Recomendadas:")
    print("   - Corrigir erros primeiro")
    print("   - Investigar falhas")
    print("   - Documentar mudanças")
    print("   - Executar testes novamente")


def main():
    """Função principal"""
    print("🎯 SISTEMA DE TESTES ALLURE - GUIA DE USO")
    print("=" * 60)
    
    exemplo_testes_unitarios()
    exemplo_testes_funcionalidade()
    exemplo_suite_completa()
    exemplo_testes_manuais()
    exemplo_ci_cd()
    exemplo_interpretacao_resultados()
    
    print("\n" + "=" * 60)
    print("📚 DOCUMENTAÇÃO ADICIONAL")
    print("=" * 60)
    print("\n📖 Arquivos importantes:")
    print("   - TESTES_README.md: Documentação completa")
    print("   - CORRECOES_TESTES.md: Problemas identificados")
    print("   - run_all_tests.py: Script principal")
    print("   - test_functionality.py: Testes de funcionalidade")
    
    print("\n🔗 Comandos úteis:")
    print("   python run_all_tests.py          # Suíte completa")
    print("   python test_functionality.py     # Apenas funcionalidade")
    print("   python manage.py test            # Apenas unitários")
    print("   python manage.py test --verbosity=2  # Com detalhes")
    
    print("\n💡 Dicas:")
    print("   - Execute testes antes de cada commit")
    print("   - Monitore a taxa de sucesso")
    print("   - Corrija problemas rapidamente")
    print("   - Documente mudanças nos testes")


if __name__ == '__main__':
    main() 