from django.contrib import admin
from django.utils.html import format_html
from django.urls import reverse
from django.utils.safestring import mark_safe
from django.db.models import Count, Q
from django.utils import timezone
from datetime import timedelta
from django.contrib import messages
from django import forms
from django.shortcuts import render
from django.utils.html import format_html
from .models import (
    Banner, Modelo, FotoModelo, FotoAprovacao, Servico, VisualizacaoModelo, 
    Favorito, Categoria, Avaliacao, Comentario, VideoModelo, PlanoAnuncio, PlanoAtivoLog
)


class StatusPendenteFilter(admin.SimpleListFilter):
    """Filtro personalizado para mostrar todos os modelos pendentes"""
    title = 'Status Pendente'
    parameter_name = 'status_pendente'

    def lookups(self, request, model_admin):
        return (
            ('todos_pendentes', 'Todos os Pendentes'),
            ('cadastro_inicial', 'Cadastro Inicial'),
            ('pendente_validacao', 'Pendente Validação'),
            ('pendente_pagamento', 'Pendente Pagamento'),
        )

    def queryset(self, request, queryset):
        if self.value() == 'todos_pendentes':
            return queryset.filter(
                status__in=['cadastro_inicial', 'pendente_validacao', 'pendente_pagamento']
            )
        elif self.value() in ['cadastro_inicial', 'pendente_validacao', 'pendente_pagamento']:
            return queryset.filter(status=self.value())
        return queryset


class PerfilCompletoFilter(admin.SimpleListFilter):
    """Filtro para modelos com perfil completo ou incompleto"""
    title = 'Perfil Completo'
    parameter_name = 'perfil_completo'

    def lookups(self, request, model_admin):
        return (
            ('completo', 'Perfil Completo'),
            ('incompleto', 'Perfil Incompleto'),
            ('sem_foto', 'Sem Foto de Perfil'),
            ('sem_contato', 'Sem Contato'),
        )

    def queryset(self, request, queryset):
        from django.db.models import Q
        
        if self.value() == 'completo':
            return queryset.exclude(
                Q(foto_perfil='') | Q(foto_perfil__isnull=True)
            ).exclude(
                Q(sobre_mim='') | Q(sobre_mim__isnull=True)
            ).exclude(
                Q(telefone='') & Q(whatsapp='') & Q(telegram='') & Q(link_contato='') & Q(email_contato='')
            )
        elif self.value() == 'incompleto':
            return queryset.filter(
                Q(foto_perfil='') | Q(foto_perfil__isnull=True) | 
                Q(sobre_mim='') | Q(sobre_mim__isnull=True) |
                (Q(telefone='') & Q(whatsapp='') & Q(telegram='') & Q(link_contato='') & Q(email_contato=''))
            )
        elif self.value() == 'sem_foto':
            return queryset.filter(
                Q(foto_perfil='') | Q(foto_perfil__isnull=True)
            )
        elif self.value() == 'sem_contato':
            return queryset.filter(
                Q(telefone='') & Q(whatsapp='') & Q(telegram='') & Q(link_contato='') & Q(email_contato='')
            )
        return queryset


class FotoModeloInline(admin.TabularInline):
    model = FotoModelo
    extra = 0
    fields = ['imagem', 'categoria', 'autorizar_redes_sociais', 'ordem', 'data_upload']
    readonly_fields = ['data_upload']


class PlanoAtivoLogInline(admin.TabularInline):
    model = PlanoAtivoLog
    extra = 0
    fields = ['plano', 'tipo_ativacao', 'data_inicio', 'data_fim', 'valor_pago', 'status_display', 'dias_restantes_display']
    readonly_fields = ['status_display', 'dias_restantes_display']
    
    def status_display(self, obj):
        """Exibe o status do plano com badge colorido"""
        if not obj.pk:
            return "-"
        
        status_colors = {
            'pendente': 'warning',
            'ativo': 'success',
            'expirado': 'danger',
        }
        color = status_colors.get(obj.status, 'secondary')
        return format_html(
            '<span class="badge bg-{}">{}</span>',
            color, obj.get_status_display()
        )
    status_display.short_description = 'Status'
    
    def dias_restantes_display(self, obj):
        """Exibe os dias restantes do plano"""
        if not obj.pk:
            return "-"
        
        dias = obj.dias_restantes
        if dias > 0:
            return format_html('<span class="text-success">{} dias</span>', dias)
        elif dias == 0:
            return format_html('<span class="text-warning">Expira hoje</span>')
        else:
            return format_html('<span class="text-danger">Expirado</span>')
    dias_restantes_display.short_description = 'Dias Restantes'
    can_delete = True
    
    def get_queryset(self, request):
        return super().get_queryset(request).order_by('-data_inicio')


class ModeloAdminForm(forms.ModelForm):
    """Formulário personalizado para o admin do Modelo"""
    
    categoria_servico = forms.MultipleChoiceField(
        choices=Modelo.CATEGORIA_SERVICO_CHOICES,
        widget=forms.CheckboxSelectMultiple,
        required=False,
        label='Tipos de Serviço'
    )
    
    local_atendimento = forms.MultipleChoiceField(
        choices=Modelo.LOCAL_ATENDIMENTO_CHOICES,
        widget=forms.CheckboxSelectMultiple,
        required=False,
        label='Locais de Atendimento'
    )
    
    dias_trabalho = forms.MultipleChoiceField(
        choices=Modelo.DIAS_SEMANA_CHOICES,
        widget=forms.CheckboxSelectMultiple,
        required=False,
        label='Dias de Trabalho'
    )
    
    class Meta:
        model = Modelo
        fields = '__all__'
    
    def __init__(self, *args, **kwargs):
        super().__init__(*args, **kwargs)
        if self.instance.pk:
            # Carregar as categorias existentes
            self.fields['categoria_servico'].initial = self.instance.get_categorias_servico_list()
            # Carregar os locais de atendimento existentes
            self.fields['local_atendimento'].initial = self.instance.get_locais_atendimento_list()
            # Carregar os dias de trabalho existentes
            self.fields['dias_trabalho'].initial = self.instance.get_dias_trabalho_list()
    
    def clean_categoria_servico(self):
        """Limpa e valida as categorias de serviço selecionadas"""
        categorias = self.cleaned_data.get('categoria_servico', [])
        return categorias if categorias else []
    
    def clean_local_atendimento(self):
        """Limpa e valida os locais de atendimento selecionados"""
        locais = self.cleaned_data.get('local_atendimento', [])
        return locais if locais else []
    
    def clean_dias_trabalho(self):
        """Limpa e valida os dias de trabalho selecionados"""
        dias = self.cleaned_data.get('dias_trabalho', [])
        return dias if dias else []
    
    def save(self, commit=True):
        """Salva o modelo com as categorias de serviço, locais de atendimento e dias de trabalho"""
        instance = super().save(commit=False)
        instance.set_categorias_servico(self.cleaned_data.get('categoria_servico', []))
        instance.set_locais_atendimento(self.cleaned_data.get('local_atendimento', []))
        instance.set_dias_trabalho(self.cleaned_data.get('dias_trabalho', []))
        if commit:
            instance.save()
        return instance


@admin.register(Categoria)
class CategoriaAdmin(admin.ModelAdmin):
    list_display = ['nome', 'ativo', 'ordem', 'data_criacao']
    list_filter = ['ativo', 'data_criacao']
    search_fields = ['nome', 'descricao']
    ordering = ['ordem', 'nome']
    list_editable = ['ativo', 'ordem']


@admin.register(Banner)
class BannerAdmin(admin.ModelAdmin):
    list_display = ['titulo', 'ativo', 'ordem', 'data_criacao']
    list_filter = ['ativo', 'data_criacao']
    search_fields = ['titulo']
    ordering = ['ordem', '-data_criacao']
    list_editable = ['ativo', 'ordem']
    
    # Ações personalizadas
    actions = ['ativar_banners', 'desativar_banners']
    
    def ativar_banners(self, request, queryset):
        """Ativa os banners selecionados"""
        updated = queryset.update(ativo=True)
        self.message_user(
            request, 
            f'{updated} banner(s) ativado(s) com sucesso!',
            messages.SUCCESS
        )
    ativar_banners.short_description = "✅ Ativar banners selecionados"
    
    def desativar_banners(self, request, queryset):
        """Desativa os banners selecionados"""
        updated = queryset.update(ativo=False)
        self.message_user(
            request, 
            f'{updated} banner(s) desativado(s)!',
            messages.WARNING
        )
    desativar_banners.short_description = "❌ Desativar banners selecionados"


@admin.register(Modelo)
class ModeloAdmin(admin.ModelAdmin):
    form = ModeloAdminForm
    list_display = [
        'nome_exibicao', 'nome', 'idade', 'cidade', 'estado', 
        'categorias_servico_display', 'status_badge', 'perfil_completo_badge', 'plano_ativo_display', 'anuncios_badge', 'data_cadastro'
    ]
    list_filter = [
        StatusPendenteFilter, PerfilCompletoFilter, 'status', 'etnia', 'categoria_servico', 
        'data_cadastro', 'planos_anuncio'
    ]
    search_fields = [
        'nome_exibicao', 'nome', 'sobrenome', 'cpf', 
        'cidade', 'estado', 'user__email'
    ]
    readonly_fields = ['data_cadastro', 'data_atualizacao', 'ultima_atividade']
    filter_horizontal = ['categorias', 'planos_anuncio']
    inlines = [FotoModeloInline, PlanoAtivoLogInline]
    
    # Ações personalizadas
    actions = [
        'aprovar_modelos', 'rejeitar_modelos', 'suspender_modelos', 'adicionar_plano_manual'
    ]
    
    fieldsets = (
        ('Informações Básicas', {
            'fields': ('user', 'nome', 'sobrenome', 'cpf', 'data_nascimento')
        }),
        ('Exibição', {
            'fields': ('nome_exibicao', 'idade')
        }),
        ('Localização', {
            'fields': ('estado', 'cidade')
        }),
        ('Características', {
            'fields': ('etnia', 'altura', 'peso', 'busto', 'cintura', 'quadril', 'calcado')
        }),
        ('Profissional', {
            'fields': ('categoria_servico', 'tipo_ensaio', 'local_atendimento', 'categorias', 'sobre_mim', 'biografia')
        }),
        ('Contatos', {
            'fields': ('telefone', 'whatsapp', 'telegram', 'link_contato', 'email_contato')
        }),
        ('Local de Trabalho', {
            'fields': ('estado_trabalho', 'cidade_trabalho', 'endereco_trabalho')
        }),
        ('Horário de Trabalho', {
            'fields': ('horario_inicio', 'horario_termino', 'dias_trabalho')
        }),
        ('Mídia', {
            'fields': ('foto_perfil', 'documento_rg', 'documento_selfie')
        }),
        ('Status', {
            'fields': ('status', 'planos_anuncio')
        }),
        ('Timestamps', {
            'fields': ('data_cadastro', 'data_atualizacao', 'ultima_atividade'),
            'classes': ('collapse',)
        }),
    )

    def status_badge(self, obj):
        """Exibe o status com badge colorido"""
        status_colors = {
            'cadastro_inicial': 'secondary',
            'pendente_validacao': 'warning',
            'validado': 'info',
            'pendente_pagamento': 'warning',
            'ativo': 'success',
            'rejeitado': 'danger',
            'suspenso': 'dark',
        }
        color = status_colors.get(obj.status, 'secondary')
        return format_html(
            '<span class="badge bg-{}">{}</span>',
            color, obj.get_status_display()
        )
    status_badge.short_description = 'Status'

    def anuncios_badge(self, obj):
        """Exibe o status dos planos de anúncio com badge"""
        planos_ativos = obj.planos_anuncio.filter(ativo=True)
        if planos_ativos.exists():
            nomes_planos = ', '.join([plano.nome for plano in planos_ativos])
            return format_html('<span class="badge bg-success">📢 {}</span>', nomes_planos)
        return format_html('<span class="badge bg-secondary">Sem Anúncios</span>')
    anuncios_badge.short_description = 'Anúncios'
    
    def plano_ativo_display(self, obj):
        """Exibe o plano ativo da modelo"""
        if obj.plano_ativo:
            # Verificar se ainda está ativo
            from django.utils import timezone
            agora = timezone.now()
            if obj.plano_data_fim and agora <= obj.plano_data_fim:
                return format_html(
                    '<span class="badge bg-success">{}<br><small>Até {}</small></span>',
                    obj.plano_ativo.nome,
                    obj.plano_data_fim.strftime('%d/%m/%Y')
                )
            else:
                return format_html(
                    '<span class="badge bg-warning">{}<br><small>Expirado</small></span>',
                    obj.plano_ativo.nome
                )
        return format_html('<span class="badge bg-secondary">Sem Plano</span>')
    plano_ativo_display.short_description = 'Plano Ativo'
    
    def categorias_servico_display(self, obj):
        """Exibe as categorias de serviço de forma formatada"""
        categorias = obj.get_categorias_servico_display()
        if not categorias:
            return format_html('<span class="text-muted">Nenhuma categoria selecionada</span>')
        
        badges = []
        for categoria in categorias:
            badges.append(f'<span class="badge bg-info me-1">{categoria}</span>')
        
        return format_html(' '.join(badges))
    categorias_servico_display.short_description = 'Tipos de Serviço'
    
    def perfil_completo_badge(self, obj):
        """Exibe o status do perfil completo"""
        if obj.tem_perfil_completo():
            return format_html('<span class="badge bg-success">✅ Completo</span>')
        else:
            campos_pendentes = obj.get_campos_pendentes()
            if len(campos_pendentes) == 1:
                return format_html('<span class="badge bg-warning">⚠️ {}</span>', campos_pendentes[0])
            else:
                return format_html('<span class="badge bg-danger">❌ {} pendentes</span>', len(campos_pendentes))
    perfil_completo_badge.short_description = 'Perfil'

    # Ações personalizadas
    def aprovar_modelos(self, request, queryset):
        """Aprova as modelos selecionadas"""
        updated = queryset.update(status='ativo')
        self.message_user(
            request, 
            f'{updated} modelo(s) aprovada(s) com sucesso!',
            messages.SUCCESS
        )
    aprovar_modelos.short_description = "✅ Aprovar modelos selecionadas"

    def rejeitar_modelos(self, request, queryset):
        """Rejeita as modelos selecionadas"""
        updated = queryset.update(status='rejeitado')
        self.message_user(
            request, 
            f'{updated} modelo(s) rejeitada(s)!',
            messages.WARNING
        )
    rejeitar_modelos.short_description = "❌ Rejeitar modelos selecionadas"

    def suspender_modelos(self, request, queryset):
        """Suspende as modelos selecionadas"""
        updated = queryset.update(status='suspenso')
        self.message_user(
            request, 
            f'{updated} modelo(s) suspensa(s)!',
            messages.WARNING
        )
    suspender_modelos.short_description = "⏸️ Suspender modelos selecionadas"
    
    def adicionar_plano_manual(self, request, queryset):
        """Adiciona um plano manualmente para as modelos selecionadas"""
        if request.POST.get('post'):
            # Processar o formulário
            plano_id = request.POST.get('plano_id')
            observacoes = request.POST.get('observacoes', '')
            
            if not plano_id:
                self.message_user(request, 'Por favor, selecione um plano.', messages.ERROR)
                return
            
            try:
                plano = PlanoAnuncio.objects.get(id=plano_id)
                agora = timezone.now()
                data_fim = agora + timedelta(days=plano.duracao_dias)
                
                for modelo in queryset:
                    # Atualizar modelo
                    modelo.status = 'ativo'
                    modelo.pagamento_aprovado = True
                    modelo.data_pagamento = agora
                    modelo.proximo_pagamento = data_fim
                    modelo.plano_ativo = plano
                    modelo.plano_data_inicio = agora
                    modelo.plano_data_fim = data_fim
                    modelo.save()
                    
                    # Criar log
                    PlanoAtivoLog.objects.create(
                        modelo=modelo,
                        plano=plano,
                        tipo_ativacao='admin',
                        data_inicio=agora,
                        data_fim=data_fim,
                        observacoes=observacoes or 'Plano adicionado manualmente pelo admin'
                    )
                
                self.message_user(
                    request,
                    f'Plano {plano.nome} adicionado para {queryset.count()} modelo(s)!',
                    messages.SUCCESS
                )
                
            except PlanoAnuncio.DoesNotExist:
                self.message_user(request, 'Plano não encontrado.', messages.ERROR)
        else:
            # Mostrar formulário
            planos = PlanoAnuncio.objects.filter(ativo=True).order_by('nome')
            context = {
                'queryset': queryset,
                'planos': planos,
                'action_name': 'adicionar_plano_manual',
                'title': 'Adicionar Plano Manualmente'
            }
            return render(request, 'admin/adicionar_plano_manual.html', context)
    
    adicionar_plano_manual.short_description = "📋 Adicionar Plano Manualmente"



@admin.register(FotoModelo)
class FotoModeloAdmin(admin.ModelAdmin):
    list_display = ['modelo', 'categoria', 'autorizar_redes_sociais', 'ordem', 'data_upload', 'imagem_preview']
    list_filter = ['categoria', 'autorizar_redes_sociais', 'data_upload']
    search_fields = ['modelo__nome_exibicao']
    ordering = ['modelo', 'ordem', '-data_upload']
    readonly_fields = ['data_upload', 'imagem_preview']
    
    def imagem_preview(self, obj):
        if obj.imagem:
            return format_html(
                '<img src="{}" style="max-height: 50px; max-width: 50px; object-fit: cover;" />',
                obj.imagem.url
            )
        return "Sem imagem"
    imagem_preview.short_description = 'Preview'


@admin.register(FotoAprovacao)
class FotoAprovacaoAdmin(admin.ModelAdmin):
    list_display = ['modelo', 'ordem', 'data_upload', 'imagem_preview']
    list_filter = ['data_upload']
    search_fields = ['modelo__nome_exibicao']
    ordering = ['modelo', 'ordem', '-data_upload']
    readonly_fields = ['data_upload', 'imagem_preview']
    
    def get_queryset(self, request):
        """Debug: verificar se o modelo está sendo carregado"""
        qs = super().get_queryset(request)
        print(f"DEBUG: FotoAprovacao queryset count: {qs.count()}")
        return qs
    
    def imagem_preview(self, obj):
        if obj.imagem:
            return format_html(
                '<img src="{}" style="max-height: 50px; max-width: 50px; object-fit: cover;" />',
                obj.imagem.url
            )
        return "Sem imagem"
    imagem_preview.short_description = 'Preview'


@admin.register(Servico)
class ServicoAdmin(admin.ModelAdmin):
    list_display = ['modelo', 'descricao', 'preco', 'duracao', 'ativo', 'data_criacao']
    list_filter = ['ativo', 'data_criacao']
    search_fields = ['modelo__nome_exibicao', 'descricao']
    ordering = ['-data_criacao']


@admin.register(VisualizacaoModelo)
class VisualizacaoModeloAdmin(admin.ModelAdmin):
    list_display = ['modelo', 'ip_address', 'data_visualizacao']
    list_filter = ['data_visualizacao']
    search_fields = ['modelo__nome_exibicao', 'ip_address']
    ordering = ['-data_visualizacao']
    readonly_fields = ['data_visualizacao']


@admin.register(Favorito)
class FavoritoAdmin(admin.ModelAdmin):
    list_display = ['modelo', 'user', 'data_favorito']
    list_filter = ['data_favorito']
    search_fields = ['modelo__nome_exibicao', 'user__username']
    ordering = ['-data_favorito']
    readonly_fields = ['data_favorito']


# @admin.register(Pagamento)  # DESABILITADO - Usando TransacaoPagamento do app payments
# class PagamentoAdmin(admin.ModelAdmin):
#     list_display = ['modelo', 'valor', 'status', 'data_criacao', 'data_pagamento']
#     list_filter = ['status', 'data_criacao', 'data_pagamento']
#     search_fields = ['modelo__nome_exibicao', 'mercadopago_id']
#     ordering = ['-data_criacao']
#     readonly_fields = ['data_criacao']


@admin.register(Avaliacao)
class AvaliacaoAdmin(admin.ModelAdmin):
    list_display = ['modelo', 'user', 'nota', 'aprovado', 'data_avaliacao']
    list_filter = ['nota', 'aprovado', 'data_avaliacao']
    search_fields = ['modelo__nome_exibicao', 'user__username', 'comentario']
    ordering = ['-data_avaliacao']
    readonly_fields = ['data_avaliacao']
    
    # Ações personalizadas
    actions = ['aprovar_avaliacoes', 'rejeitar_avaliacoes']
    
    def aprovar_avaliacoes(self, request, queryset):
        """Aprova as avaliações selecionadas"""
        updated = queryset.update(aprovado=True)
        self.message_user(
            request, 
            f'{updated} avaliação(ões) aprovada(s) com sucesso!',
            messages.SUCCESS
        )
    aprovar_avaliacoes.short_description = "✅ Aprovar avaliações selecionadas"
    
    def rejeitar_avaliacoes(self, request, queryset):
        """Rejeita as avaliações selecionadas"""
        updated = queryset.update(aprovado=False)
        self.message_user(
            request, 
            f'{updated} avaliação(ões) rejeitada(s)!',
            messages.WARNING
        )
    rejeitar_avaliacoes.short_description = "❌ Rejeitar avaliações selecionadas"


@admin.register(Comentario)
class ComentarioAdmin(admin.ModelAdmin):
    list_display = ['modelo', 'user', 'texto_curto', 'aprovado', 'data_comentario']
    list_filter = ['aprovado', 'data_comentario']
    search_fields = ['modelo__nome_exibicao', 'user__username', 'texto']
    ordering = ['-data_comentario']
    readonly_fields = ['data_comentario']
    
    # Ações personalizadas
    actions = ['aprovar_comentarios', 'rejeitar_comentarios']
    
    def texto_curto(self, obj):
        """Exibe apenas os primeiros 50 caracteres do texto"""
        return obj.texto[:50] + '...' if len(obj.texto) > 50 else obj.texto
    texto_curto.short_description = 'Texto'
    
    def aprovar_comentarios(self, request, queryset):
        """Aprova os comentários selecionados"""
        updated = queryset.update(aprovado=True)
        self.message_user(
            request, 
            f'{updated} comentário(s) aprovado(s) com sucesso!',
            messages.SUCCESS
        )
    aprovar_comentarios.short_description = "✅ Aprovar comentários selecionados"
    
    def rejeitar_comentarios(self, request, queryset):
        """Rejeita os comentários selecionados"""
        updated = queryset.update(aprovado=False)
        self.message_user(
            request, 
            f'{updated} comentário(s) rejeitado(s)!',
            messages.WARNING
        )
    rejeitar_comentarios.short_description = "❌ Rejeitar comentários selecionados"


@admin.register(VideoModelo)
class VideoModeloAdmin(admin.ModelAdmin):
    list_display = ['modelo', 'titulo', 'duracao_formatada', 'ativo', 'data_upload']
    list_filter = ['ativo', 'data_upload']
    search_fields = ['modelo__nome_exibicao', 'titulo', 'descricao']
    ordering = ['-data_upload']
    readonly_fields = ['data_upload']
    
    # Ações personalizadas
    actions = ['ativar_videos', 'desativar_videos']
    
    def duracao_formatada(self, obj):
        """Formata a duração em minutos:segundos"""
        if obj.duracao:
            minutos = obj.duracao // 60
            segundos = obj.duracao % 60
            return f"{minutos}:{segundos:02d}"
        return "N/A"
    duracao_formatada.short_description = 'Duração'
    
    def ativar_videos(self, request, queryset):
        """Ativa os vídeos selecionados"""
        updated = queryset.update(ativo=True)
        self.message_user(
            request, 
            f'{updated} vídeo(s) ativado(s) com sucesso!',
            messages.SUCCESS
        )
    ativar_videos.short_description = "✅ Ativar vídeos selecionados"
    
    def desativar_videos(self, request, queryset):
        """Desativa os vídeos selecionados"""
        updated = queryset.update(ativo=False)
        self.message_user(
            request, 
            f'{updated} vídeo(s) desativado(s)!',
            messages.WARNING
        )
    desativar_videos.short_description = "❌ Desativar vídeos selecionados"


@admin.register(PlanoAnuncio)
class PlanoAnuncioAdmin(admin.ModelAdmin):
    list_display = ['nome', 'tipo', 'duracao_dias', 'preco', 'boost', 'ativo', 'ordem']
    list_filter = ['tipo', 'ativo']
    search_fields = ['nome', 'tipo']
    list_editable = ['ativo', 'ordem', 'preco', 'boost']
    ordering = ['ordem', 'tipo', 'preco']
    
    fieldsets = (
        ('Informações Básicas', {
            'fields': ('nome', 'tipo', 'duracao_dias', 'preco', 'boost', 'ativo', 'ordem')
        }),
        ('Benefícios', {
            'fields': (
                'destaque_perfil', 'boost_visualizacoes', 'fotos_ilimitadas', 
                'videos_ilimitados', 'suporte_prioritario', 'relatorio_estatisticas'
            )
        }),
        ('Limites', {
            'fields': ('max_fotos', 'max_videos')
        }),
        ('Timestamps', {
            'fields': ('data_criacao', 'data_atualizacao'),
            'classes': ('collapse',)
        })
    )
    
    readonly_fields = ['data_criacao', 'data_atualizacao']
    
    def get_queryset(self, request):
        return super().get_queryset(request).select_related()


# Customização do site admin
admin.site.site_header = 'Painel Administrativo Allure'
admin.site.site_title = 'Allure Admin'
admin.site.index_title = 'Gerenciamento do Site Allure'
