from django.contrib.auth.models import User
from django.db import models
from django.core.validators import MinValueValidator, MaxValueValidator
from django.utils import timezone
from django.core.exceptions import ValidationError
import re


class Categoria(models.Model):
    """Model para as categorias de serviços das modelos"""
    
    nome = models.CharField(max_length=100, unique=True)
    descricao = models.TextField(blank=True)
    ativo = models.BooleanField(default=True)
    ordem = models.PositiveIntegerField(default=0)
    data_criacao = models.DateTimeField(auto_now_add=True)
    
    class Meta:
        ordering = ['ordem', 'nome']
        verbose_name = 'Categoria'
        verbose_name_plural = 'Categorias'
    
    def __str__(self):
        return self.nome


class Banner(models.Model):
    """Model para banners do site"""
    
    titulo = models.CharField(max_length=200)
    imagem = models.ImageField(upload_to='banners/')
    link = models.URLField(blank=True)
    ativo = models.BooleanField(default=True)
    ordem = models.PositiveIntegerField(default=0)
    data_criacao = models.DateTimeField(auto_now_add=True)
    
    class Meta:
        ordering = ['ordem', '-data_criacao']
        verbose_name = 'Banner'
        verbose_name_plural = 'Banners'
    
    def __str__(self):
        return self.titulo


class Modelo(models.Model):
    """Model principal para as modelos/acompanhantes"""
    
    # Choices para os campos
    TIPO_ENSAIO_CHOICES = [
        ('solo', 'Anúncio Só'),
        ('dupla', 'Anúncio com Dupla'),
        ('trio', 'Anúncio com Trio'),
        ('quarteto', 'Anúncio com Quarteto'),
    ]
    
    LOCAL_ATENDIMENTO_CHOICES = [
        ('local_proprio', 'Local Próprio'),
        ('motel', 'Motel'),
        ('hotel', 'Hotel'),
        ('residencia_cliente', 'Residência do Cliente'),
    ]
    
    ETNIA_CHOICES = [
        ('morena', 'Morena'),
        ('ruiva', 'Ruiva'),
        ('loira', 'Loira'),
        ('negra', 'Negra'),
        ('mulata', 'Mulata'),
        ('mestica', 'Mestiça'),
        ('oriental', 'Oriental'),
    ]
    
    DIAS_SEMANA_CHOICES = [
        ('segunda', 'Segunda-feira'),
        ('terca', 'Terça-feira'),
        ('quarta', 'Quarta-feira'),
        ('quinta', 'Quinta-feira'),
        ('sexta', 'Sexta-feira'),
        ('sabado', 'Sábado'),
        ('domingo', 'Domingo'),
    ]
    
    STATUS_CHOICES = [
        ('cadastro_inicial', 'Cadastro Inicial'),
        ('pendente_aprovacao', 'Pendente Aprovação'),
        ('aprovado', 'Aprovado'),
        ('cadastro_completo', 'Cadastro Completo'),
        ('ativo', 'Ativo'),
        ('rejeitado', 'Rejeitado'),
        ('suspenso', 'Suspenso'),
    ]
    
    # Campos básicos do usuário
    user = models.OneToOneField(User, on_delete=models.CASCADE, related_name='modelo_profile')
    nome = models.CharField(max_length=100, verbose_name='Nome')
    sobrenome = models.CharField(max_length=100, verbose_name='Sobrenome')
    cpf = models.CharField(max_length=14, unique=True, verbose_name='CPF')
    data_nascimento = models.DateField(verbose_name='Data de Nascimento')
    
    # Campos de exibição
    nome_exibicao = models.CharField(max_length=100, verbose_name='Nome de Exibição')
    slug = models.SlugField(max_length=120, unique=True, blank=True, verbose_name='Slug para URL')
    idade = models.PositiveIntegerField(validators=[MinValueValidator(18), MaxValueValidator(99)])
    
    # Localização
    estado = models.CharField(max_length=2)
    cidade = models.CharField(max_length=100)
    
    # Características físicas
    etnia = models.CharField(max_length=20, choices=ETNIA_CHOICES)
    altura = models.PositiveIntegerField(help_text='Altura em centímetros', blank=True, null=True)
    peso = models.PositiveIntegerField(help_text='Peso em quilos', blank=True, null=True)
    
    # Medidas adicionais
    busto = models.PositiveIntegerField(help_text='Busto em centímetros', blank=True, null=True)
    cintura = models.PositiveIntegerField(help_text='Cintura em centímetros', blank=True, null=True)
    quadril = models.PositiveIntegerField(help_text='Quadril em centímetros', blank=True, null=True)
    calcado = models.PositiveIntegerField(help_text='Calçado', blank=True, null=True)
    
    # Informações profissionais
    CATEGORIA_SERVICO_CHOICES = [
        ('JOB', 'Anúncios'),
        ('FAB', 'Venda de Conteúdos'),
        ('CAM', 'Câmera Privativa'),
    ]
    
    categoria_servico = models.TextField(default='[]', verbose_name='Tipos de Serviço')
    tipo_ensaio = models.CharField(max_length=20, choices=TIPO_ENSAIO_CHOICES)
    local_atendimento = models.TextField(default='[]', verbose_name='Locais de Atendimento')
    categorias = models.ManyToManyField(Categoria, blank=True, verbose_name='Categorias de Serviços')
    
    # Descrição
    sobre_mim = models.TextField(verbose_name='Sobre Mim', blank=True)
    biografia = models.TextField(verbose_name='Biografia', blank=True)
    
    # Contatos
    telefone = models.CharField(max_length=20, blank=True)
    whatsapp = models.CharField(max_length=20, blank=True)
    telegram = models.CharField(max_length=50, blank=True)
    link_contato = models.URLField(blank=True, verbose_name='Link para Contato')
    email_contato = models.EmailField(blank=True)
    
    # Local de Trabalho
    estado_trabalho = models.CharField(max_length=2, blank=True, verbose_name='Estado de Trabalho')
    cidade_trabalho = models.CharField(max_length=100, blank=True, verbose_name='Cidade de Trabalho')
    endereco_trabalho = models.TextField(blank=True, verbose_name='Endereço de Trabalho', help_text='Rua, Avenida, Bairro')
    
    # Horário de Trabalho
    horario_inicio = models.TimeField(blank=True, null=True, verbose_name='Horário de Início')
    horario_termino = models.TimeField(blank=True, null=True, verbose_name='Horário de Término')
    dias_trabalho = models.TextField(default='[]', verbose_name='Dias de Trabalho', help_text='Dias da semana que trabalha')
    
    # Fotos
    foto_perfil = models.ImageField(upload_to='modelos/perfil/', blank=True, null=True)
    
    # Documentos (apenas para verificação administrativa)
    documento_rg = models.ImageField(upload_to='modelos/documentos/', blank=True, null=True)
    documento_selfie = models.ImageField(upload_to='modelos/documentos/', blank=True, null=True)
    
    # Status e controle
    status = models.CharField(max_length=20, choices=STATUS_CHOICES, default='cadastro_inicial')

    # Planos de anúncio ativos
    planos_anuncio = models.ManyToManyField('PlanoAnuncio', blank=True, related_name='modelos_com_planos', verbose_name='Planos de Anúncio')
    
    # Pagamento
    pagamento_aprovado = models.BooleanField(default=False)
    data_pagamento = models.DateTimeField(blank=True, null=True)
    proximo_pagamento = models.DateTimeField(blank=True, null=True)
    
    # Plano ativo
    plano_ativo = models.ForeignKey('PlanoAnuncio', on_delete=models.SET_NULL, null=True, blank=True, related_name='modelos_ativas')
    plano_data_inicio = models.DateTimeField(blank=True, null=True)
    plano_data_fim = models.DateTimeField(blank=True, null=True)
    
    # Timestamps
    data_cadastro = models.DateTimeField(auto_now_add=True)
    data_atualizacao = models.DateTimeField(auto_now=True)
    ultima_atividade = models.DateTimeField(default=timezone.now)
    
    class Meta:
        verbose_name = 'Modelo'
        verbose_name_plural = 'Modelos'
        ordering = ['-data_cadastro']
    
    def __str__(self):
        return self.nome_exibicao
    
    def save(self, *args, **kwargs):
        """Override save para gerar slug automaticamente"""
        if not self.slug:
            self.slug = self.generate_slug()
        super().save(*args, **kwargs)
    
    def generate_slug(self):
        """Gera um slug único baseado no nome de exibição"""
        from django.utils.text import slugify
        from django.db import models
        
        base_slug = slugify(self.nome_exibicao)
        slug = base_slug
        counter = 1
        
        # Verificar se o slug já existe
        while Modelo.objects.filter(slug=slug).exclude(pk=self.pk).exists():
            slug = f"{base_slug}-{counter}"
            counter += 1
        
        return slug
    
    def clean(self):
        """Validação personalizada"""
        # Validar CPF
        if self.cpf:
            cpf = re.sub(r'[^0-9]', '', self.cpf)
            if len(cpf) != 11:
                raise ValidationError('CPF deve ter 11 dígitos.')
            
            # Verificar se todos os dígitos são iguais
            if cpf == cpf[0] * 11:
                raise ValidationError('CPF inválido.')
            
            # Validar dígitos verificadores
            soma = sum(int(cpf[i]) * (10 - i) for i in range(9))
            resto = soma % 11
            if resto < 2:
                digito1 = 0
            else:
                digito1 = 11 - resto
            
            soma = sum(int(cpf[i]) * (11 - i) for i in range(10))
            resto = soma % 11
            if resto < 2:
                digito2 = 0
            else:
                digito2 = 11 - resto
            
            if int(cpf[9]) != digito1 or int(cpf[10]) != digito2:
                raise ValidationError('CPF inválido.')
    
    @property
    def is_boost_ativo(self):
        """Verifica se o modelo tem um plano ativo (boost)"""
        if self.plano_ativo and self.plano_data_fim:
            return timezone.now() < self.plano_data_fim
        return False
    
    def get_total_views(self):
        """Retorna o total de visualizações"""
        return self.visualizacoes.count()
    
    def get_monthly_views(self):
        """Retorna as visualizações do último mês"""
        um_mes_atras = timezone.now() - timezone.timedelta(days=30)
        return self.visualizacoes.filter(data_visualizacao__gte=um_mes_atras).count()
    
    def get_categorias_servico_list(self):
        """Retorna a lista de categorias de serviço"""
        import json
        if not self.categoria_servico:
            return []
        try:
            if isinstance(self.categoria_servico, str):
                # Tentar fazer parse do JSON
                try:
                    return json.loads(self.categoria_servico)
                except json.JSONDecodeError:
                    # Se não for JSON válido, tratar como string simples
                    return [self.categoria_servico]
            return self.categoria_servico if isinstance(self.categoria_servico, list) else [self.categoria_servico]
        except (json.JSONDecodeError, TypeError):
            return []
    
    def get_categorias_servico_display(self):
        """Retorna as categorias de serviço formatadas para exibição"""
        if not self.categoria_servico:
            return []
        choices_dict = dict(self.CATEGORIA_SERVICO_CHOICES)
        categorias = self.get_categorias_servico_list()
        return [choices_dict.get(cat, cat) for cat in categorias]
    
    def set_categorias_servico(self, categorias_list):
        """Define as categorias de serviço a partir de uma lista"""
        import json
        if isinstance(categorias_list, list):
            self.categoria_servico = json.dumps(categorias_list)
        else:
            self.categoria_servico = json.dumps([categorias_list] if categorias_list else [])
    
    def get_locais_atendimento_list(self):
        """Retorna a lista de locais de atendimento"""
        import json
        if not self.local_atendimento:
            return []
        try:
            if isinstance(self.local_atendimento, str):
                # Tentar fazer parse do JSON
                try:
                    return json.loads(self.local_atendimento)
                except json.JSONDecodeError:
                    # Se não for JSON válido, tratar como string simples
                    return [self.local_atendimento]
            return self.local_atendimento if isinstance(self.local_atendimento, list) else [self.local_atendimento]
        except (json.JSONDecodeError, TypeError):
            return []
    
    def get_locais_atendimento_display(self):
        """Retorna os locais de atendimento formatados para exibição"""
        if not self.local_atendimento:
            return []
        choices_dict = dict(self.LOCAL_ATENDIMENTO_CHOICES)
        locais = self.get_locais_atendimento_list()
        return [choices_dict.get(local, local) for local in locais]
    
    def set_locais_atendimento(self, locais_list):
        """Define os locais de atendimento a partir de uma lista"""
        import json
        if isinstance(locais_list, list):
            self.local_atendimento = json.dumps(locais_list)
        else:
            self.local_atendimento = json.dumps([locais_list] if locais_list else [])
    
    def get_dias_trabalho_list(self):
        """Retorna a lista de dias de trabalho"""
        import json
        if not self.dias_trabalho:
            return []
        try:
            if isinstance(self.dias_trabalho, str):
                # Tentar fazer parse do JSON
                try:
                    return json.loads(self.dias_trabalho)
                except json.JSONDecodeError:
                    # Se não for JSON válido, tratar como string simples
                    return [self.dias_trabalho]
            return self.dias_trabalho if isinstance(self.dias_trabalho, list) else [self.dias_trabalho]
        except (json.JSONDecodeError, TypeError):
            return []
    
    def get_dias_trabalho_display(self):
        """Retorna os dias de trabalho formatados para exibição"""
        if not self.dias_trabalho:
            return []
        choices_dict = dict(self.DIAS_SEMANA_CHOICES)
        dias = self.get_dias_trabalho_list()
        return [choices_dict.get(dia, dia) for dia in dias]
    
    def set_dias_trabalho(self, dias_list):
        """Define os dias de trabalho a partir de uma lista"""
        import json
        if isinstance(dias_list, list):
            self.dias_trabalho = json.dumps(dias_list)
        else:
            self.dias_trabalho = json.dumps([dias_list] if dias_list else [])
    
    def tem_foto_perfil(self):
        """Verifica se a modelo tem foto de perfil"""
        return bool(
            self.foto_perfil and 
            self.foto_perfil.name and 
            self.foto_perfil.name.strip() != '' and
            self.foto_perfil.url
        )
    
    def tem_contato_valido(self):
        """Verifica se a modelo tem pelo menos um contato válido"""
        contatos = [
            self.telefone,
            self.whatsapp,
            self.telegram,
            self.link_contato,
            self.email_contato
        ]
        return any(contato and contato.strip() for contato in contatos)
    
    def tem_perfil_completo(self):
        """Verifica se o perfil da modelo está completo para aparecer publicamente"""
        return (
            self.tem_foto_perfil() and
            self.tem_contato_valido() and
            self.sobre_mim and self.sobre_mim.strip() and
            self.status in ['aprovado', 'ativo']
        )
    
    def get_campos_pendentes(self):
        """Retorna lista dos campos obrigatórios que estão pendentes"""
        pendentes = []
        
        if not self.tem_foto_perfil():
            pendentes.append('Foto de Perfil')
        
        if not self.tem_contato_valido():
            pendentes.append('Contato (telefone, whatsapp, telegram, link ou email)')
        
        if not self.sobre_mim or not self.sobre_mim.strip():
            pendentes.append('Descrição "Sobre Mim"')
        
        return pendentes
    
    def pode_aparecer_publicamente(self):
        """Verifica se a modelo pode aparecer em listagens públicas"""
        return (
            self.status in ['aprovado', 'ativo'] and
            self.tem_perfil_completo()
        )
    
    def tem_boost_ativo(self):
        """Verifica se a modelo tem um boost ativo no momento"""
        from django.utils import timezone
        return self.boost_usage.filter(
            data_expiracao__gt=timezone.now(),
            usado=True
        ).exists()
    
    def pode_usar_boost_hoje(self):
        """Verifica se a modelo pode usar boost hoje"""
        from django.utils import timezone
        from datetime import datetime, timedelta
        
        hoje = timezone.now().date()
        
        # Verificar se tem plano ativo
        if not (self.plano_ativo and self.plano_data_fim > timezone.now()):
            return False, "Você precisa ter um plano ativo para usar boosts"
        
        # Contar boosts usados hoje
        boosts_hoje = self.boost_usage.filter(
            data_boost__date=hoje,
            usado=True
        ).count()
        
        # Verificar limite do plano
        limite_diario = self.plano_ativo.get_boosts_diarios()
        
        if boosts_hoje >= limite_diario:
            return False, f"Você já usou todos os {limite_diario} boosts de hoje. Volte amanhã!"
        
        return True, f"Você pode usar {limite_diario - boosts_hoje} boost(s) hoje"
    
    def usar_boost(self):
        """Ativa um boost para a modelo"""
        from django.utils import timezone
        from datetime import timedelta
        
        # Verificar se pode usar
        pode_usar, mensagem = self.pode_usar_boost_hoje()
        if not pode_usar:
            return False, mensagem
        
        # Criar o boost (dura 1 hora)
        data_expiracao = timezone.now() + timedelta(hours=1)
        
        BoostUsage.objects.create(
            modelo=self,
            plano=self.plano_ativo,
            data_expiracao=data_expiracao
        )
        
        return True, f"Boost ativado! Você ficará em destaque por 1 hora até {data_expiracao.strftime('%H:%M')}"
    
    def get_boost_info(self):
        """Retorna informações sobre o boost atual"""
        from django.utils import timezone
        
        boost_ativo = self.boost_usage.filter(
            data_expiracao__gt=timezone.now(),
            usado=True
        ).first()
        
        if boost_ativo:
            tempo_restante = boost_ativo.data_expiracao - timezone.now()
            minutos_restantes = int(tempo_restante.total_seconds() / 60)
            return {
                'ativo': True,
                'minutos_restantes': minutos_restantes,
                'data_expiracao': boost_ativo.data_expiracao
            }
        
        return {'ativo': False}


class FotoModelo(models.Model):
    """Model para as fotos das modelos"""
    
    CATEGORIA_FOTO_CHOICES = [
        ('galeria', 'Galeria'),
        ('seios', 'Seios'),
        ('frente', 'Frente'),
        ('bumbum', 'Bumbum'),
        ('rosto', 'Rosto'),
    ]
    
    modelo = models.ForeignKey(Modelo, on_delete=models.CASCADE, related_name='fotos')
    imagem = models.ImageField(upload_to='modelos/fotos/')
    categoria = models.CharField(max_length=20, choices=CATEGORIA_FOTO_CHOICES, default='galeria')
    ordem = models.PositiveIntegerField(default=0)
    data_upload = models.DateTimeField(auto_now_add=True)
    autorizar_redes_sociais = models.BooleanField(
        default=False, 
        verbose_name='Autorizar uso em redes sociais',
        help_text='Marque se autoriza o uso desta foto em redes sociais da plataforma'
    )
    
    class Meta:
        ordering = ['ordem', 'data_upload']
        verbose_name = 'Foto da Modelo'
        verbose_name_plural = 'Fotos das Modelos'
    
    def __str__(self):
        return f'{self.modelo.nome_exibicao} - {self.categoria}'


class FotoAprovacao(models.Model):
    """Model para fotos enviadas para aprovação durante o cadastro inicial"""
    
    modelo = models.ForeignKey(Modelo, on_delete=models.CASCADE, related_name='fotos_aprovacao')
    imagem = models.ImageField(upload_to='modelos/aprovacao/')
    ordem = models.PositiveIntegerField(default=0)
    data_upload = models.DateTimeField(auto_now_add=True)
    
    class Meta:
        ordering = ['ordem', '-data_upload']
        verbose_name = 'Foto para Aprovação'
        verbose_name_plural = 'Fotos para Aprovação'
    
    def __str__(self):
        return f"Foto Aprovação {self.ordem} - {self.modelo.nome_exibicao}"


class PlanoAtivoLog(models.Model):
    """Model para registrar histórico de planos ativos das modelos"""
    
    TIPO_ATIVACAO_CHOICES = [
        ('pagamento', 'Pagamento Aprovado'),
        ('admin', 'Adicionado pelo Admin'),
        ('renovacao', 'Renovação'),
    ]
    
    modelo = models.ForeignKey(Modelo, on_delete=models.CASCADE, related_name='logs_planos')
    plano = models.ForeignKey('PlanoAnuncio', on_delete=models.CASCADE)
    tipo_ativacao = models.CharField(max_length=20, choices=TIPO_ATIVACAO_CHOICES)
    data_inicio = models.DateTimeField()
    data_fim = models.DateTimeField()
    valor_pago = models.DecimalField(max_digits=10, decimal_places=2, null=True, blank=True)
    transacao_pagamento = models.ForeignKey('payments.TransacaoPagamento', on_delete=models.SET_NULL, null=True, blank=True)
    observacoes = models.TextField(blank=True)
    data_criacao = models.DateTimeField(auto_now_add=True)
    
    class Meta:
        verbose_name = 'Log de Plano Ativo'
        verbose_name_plural = 'Logs de Planos Ativos'
        ordering = ['-data_inicio']
    
    def __str__(self):
        return f'{self.modelo.nome_exibicao} - {self.plano.nome} ({self.data_inicio.strftime("%d/%m/%Y")})'
    
    @property
    def dias_restantes(self):
        """Calcula quantos dias restam do plano"""
        from django.utils import timezone
        agora = timezone.now()
        if agora < self.data_fim:
            return (self.data_fim - agora).days
        return 0
    
    @property
    def status(self):
        """Retorna o status do plano"""
        from django.utils import timezone
        agora = timezone.now()
        if agora < self.data_inicio:
            return 'pendente'
        elif agora <= self.data_fim:
            return 'ativo'
        else:
            return 'expirado'


class Servico(models.Model):
    """Model para os serviços e preços das modelos"""
    
    modelo = models.ForeignKey(Modelo, on_delete=models.CASCADE, related_name='servicos')
    descricao = models.CharField(max_length=200)
    preco = models.DecimalField(max_digits=10, decimal_places=2)
    duracao = models.CharField(max_length=50, blank=True)
    ativo = models.BooleanField(default=True)
    data_criacao = models.DateTimeField(auto_now_add=True)
    
    class Meta:
        ordering = ['-data_criacao']
        verbose_name = 'Serviço'
        verbose_name_plural = 'Serviços'
    
    def __str__(self):
        return f'{self.modelo.nome_exibicao} - {self.descricao}'


class VisualizacaoModelo(models.Model):
    """Model para registrar visualizações dos perfis das modelos"""
    
    modelo = models.ForeignKey(Modelo, on_delete=models.CASCADE, related_name='visualizacoes')
    ip_address = models.GenericIPAddressField()
    user_agent = models.TextField(blank=True)
    data_visualizacao = models.DateTimeField(auto_now_add=True)
    
    class Meta:
        ordering = ['-data_visualizacao']
        verbose_name = 'Visualização'
        verbose_name_plural = 'Visualizações'
    
    def __str__(self):
        return f'{self.modelo.nome_exibicao} - {self.data_visualizacao}'


class Favorito(models.Model):
    """Model para favoritos dos usuários"""
    
    modelo = models.ForeignKey(Modelo, on_delete=models.CASCADE, related_name='favoritado_por')
    user = models.ForeignKey(User, on_delete=models.CASCADE, related_name='favoritos')
    data_favorito = models.DateTimeField(auto_now_add=True)
    
    class Meta:
        unique_together = ['modelo', 'user']
        ordering = ['-data_favorito']
        verbose_name = 'Favorito'
        verbose_name_plural = 'Favoritos'
    
    def __str__(self):
        return f'{self.user.username} favoritou {self.modelo.nome_exibicao}'


class Pagamento(models.Model):
    """Model para controle de pagamentos das modelos"""
    
    STATUS_CHOICES = [
        ('pendente', 'Pendente'),
        ('aprovado', 'Aprovado'),
        ('rejeitado', 'Rejeitado'),
        ('cancelado', 'Cancelado'),
    ]
    
    modelo = models.ForeignKey(Modelo, on_delete=models.CASCADE, related_name='pagamentos')
    valor = models.DecimalField(max_digits=10, decimal_places=2)
    status = models.CharField(max_length=20, choices=STATUS_CHOICES, default='pendente')
    mercadopago_id = models.CharField(max_length=100, blank=True)
    data_pagamento = models.DateTimeField(blank=True, null=True)
    data_criacao = models.DateTimeField(auto_now_add=True)
    
    class Meta:
        ordering = ['-data_criacao']
        verbose_name = 'Pagamento'
        verbose_name_plural = 'Pagamentos'
    
    def __str__(self):
        return f'{self.modelo.nome_exibicao} - R$ {self.valor}'


class Avaliacao(models.Model):
    """Model para avaliações das modelos"""
    
    modelo = models.ForeignKey(Modelo, on_delete=models.CASCADE, related_name='avaliacoes')
    user = models.ForeignKey(User, on_delete=models.CASCADE, related_name='avaliacoes_feitas')
    nota = models.PositiveIntegerField(validators=[MinValueValidator(1), MaxValueValidator(5)])
    comentario = models.TextField(blank=True)
    data_avaliacao = models.DateTimeField(auto_now_add=True)
    aprovado = models.BooleanField(default=False)
    
    class Meta:
        unique_together = ['modelo', 'user']
        ordering = ['-data_avaliacao']
        verbose_name = 'Avaliação'
        verbose_name_plural = 'Avaliações'
    
    def __str__(self):
        return f'{self.user.username} avaliou {self.modelo.nome_exibicao} com {self.nota} estrelas'


class Comentario(models.Model):
    """Model para comentários das modelos"""
    
    modelo = models.ForeignKey(Modelo, on_delete=models.CASCADE, related_name='comentarios')
    user = models.ForeignKey(User, on_delete=models.CASCADE, related_name='comentarios_feitos')
    texto = models.TextField()
    data_comentario = models.DateTimeField(auto_now_add=True)
    aprovado = models.BooleanField(default=False)
    
    class Meta:
        ordering = ['-data_comentario']
        verbose_name = 'Comentário'
        verbose_name_plural = 'Comentários'
    
    def __str__(self):
        return f'Comentário de {self.user.username} em {self.modelo.nome_exibicao}'


class VideoModelo(models.Model):
    """Model para vídeos das modelos"""
    
    modelo = models.ForeignKey(Modelo, on_delete=models.CASCADE, related_name='videos')
    titulo = models.CharField(max_length=200)
    descricao = models.TextField(blank=True)
    arquivo = models.FileField(upload_to='modelos/videos/')
    thumbnail = models.ImageField(upload_to='modelos/videos/thumbnails/', blank=True, null=True)
    duracao = models.PositiveIntegerField(help_text='Duração em segundos', blank=True, null=True)
    ativo = models.BooleanField(default=True)
    data_upload = models.DateTimeField(auto_now_add=True)
    
    class Meta:
        ordering = ['-data_upload']
        verbose_name = 'Vídeo da Modelo'
        verbose_name_plural = 'Vídeos das Modelos'
    
    def __str__(self):
        return f'{self.modelo.nome_exibicao} - {self.titulo}'


class PlanoAnuncio(models.Model):
    """Model para os planos de anúncios das modelos"""
    
    TIPO_PLANO_CHOICES = [
        ('basico', 'Básico'),
        ('premium', 'Premium'),
        ('vip', 'VIP'),
        ('exclusivo', 'Exclusivo'),
    ]
    
    nome = models.CharField(max_length=100)
    tipo = models.CharField(max_length=20, choices=TIPO_PLANO_CHOICES)
    duracao_dias = models.PositiveIntegerField(help_text='Duração do plano em dias')
    preco = models.DecimalField(max_digits=10, decimal_places=2)
    boost = models.PositiveIntegerField(help_text='Valor do boost', default=0)
    
    # Benefícios do plano
    destaque_perfil = models.BooleanField(default=False)
    boost_visualizacoes = models.BooleanField(default=False)
    fotos_ilimitadas = models.BooleanField(default=False)
    videos_ilimitados = models.BooleanField(default=False)
    suporte_prioritario = models.BooleanField(default=False)
    relatorio_estatisticas = models.BooleanField(default=False)
    
    # Limites
    max_fotos = models.PositiveIntegerField(default=10)
    max_videos = models.PositiveIntegerField(default=5)
    
    # Status
    ativo = models.BooleanField(default=True)
    ordem = models.PositiveIntegerField(default=0)
    
    # Timestamps
    data_criacao = models.DateTimeField(auto_now_add=True)
    data_atualizacao = models.DateTimeField(auto_now=True)
    
    class Meta:
        ordering = ['ordem', 'preco']
        verbose_name = 'Plano de Anúncio'
        verbose_name_plural = 'Planos de Anúncio'
    
    def __str__(self):
        return f"{self.nome} - {self.duracao_dias} dias - R$ {self.preco}"
    
    def get_preco_mensal(self):
        """Retorna o preço mensal do plano"""
        from decimal import Decimal
        return float(self.preco) / (self.duracao_dias / 30)
    
    def get_boosts_diarios(self):
        """Retorna quantos boosts por dia o plano oferece"""
        # Retorna o valor do campo boost do plano
        return self.boost if self.boost > 0 else 1


class BoostUsage(models.Model):
    """Model para controlar o uso de boosts das modelos"""
    
    modelo = models.ForeignKey(Modelo, on_delete=models.CASCADE, related_name='boost_usage')
    plano = models.ForeignKey(PlanoAnuncio, on_delete=models.CASCADE)
    data_boost = models.DateTimeField(auto_now_add=True)
    data_expiracao = models.DateTimeField()
    usado = models.BooleanField(default=True)
    
    class Meta:
        ordering = ['-data_boost']
        verbose_name = 'Uso de Boost'
        verbose_name_plural = 'Usos de Boost'
    
    def __str__(self):
        return f"{self.modelo.nome_exibicao} - Boost em {self.data_boost.strftime('%d/%m/%Y %H:%M')}"
    
    @property
    def is_ativo(self):
        """Verifica se o boost ainda está ativo"""
        from django.utils import timezone
        return timezone.now() < self.data_expiracao and self.usado

