from django.db.models.signals import post_save, post_delete
from django.dispatch import receiver
from django.utils import timezone


@receiver(post_save, sender='models.PlanoAtivoLog')
def ativar_plano_automaticamente(sender, instance, created, **kwargs):
    """
    Ativa automaticamente o plano_ativo da modelo quando um PlanoAtivoLog é criado
    """
    if created:
        from .models import Modelo
        
        modelo = instance.modelo
        agora = timezone.now()
        
        # Verificar se o plano deve ser ativado agora
        if instance.data_inicio <= agora <= instance.data_fim:
            # Ativar o plano na modelo
            modelo.plano_ativo = instance.plano
            modelo.plano_data_inicio = instance.data_inicio
            modelo.plano_data_fim = instance.data_fim
            modelo.save(update_fields=['plano_ativo', 'plano_data_inicio', 'plano_data_fim'])
            
            # Aplicar benefícios do plano se necessário
            # Nota: campo premium foi removido, benefícios são aplicados via plano_ativo


@receiver(post_delete, sender='models.PlanoAtivoLog')
def verificar_plano_apos_remocao(sender, instance, **kwargs):
    """
    Verifica se ainda há planos ativos após remoção de um log
    """
    from .models import Modelo, PlanoAtivoLog
    
    modelo = instance.modelo
    agora = timezone.now()
    
    # Buscar o log mais recente ainda ativo
    log_ativo = PlanoAtivoLog.objects.filter(
        modelo=modelo,
        data_inicio__lte=agora,
        data_fim__gt=agora
    ).order_by('-data_inicio').first()
    
    if log_ativo:
        # Manter o plano ativo
        modelo.plano_ativo = log_ativo.plano
        modelo.plano_data_inicio = log_ativo.data_inicio
        modelo.plano_data_fim = log_ativo.data_fim
        modelo.save(update_fields=['plano_ativo', 'plano_data_inicio', 'plano_data_fim'])
    else:
        # Desativar plano
        modelo.plano_ativo = None
        modelo.plano_data_inicio = None
        modelo.plano_data_fim = None
        modelo.save(update_fields=['plano_ativo', 'plano_data_inicio', 'plano_data_fim'])
