from django.contrib import admin
from django.contrib import messages
from django.shortcuts import render
from django.utils import timezone
from datetime import timedelta
from .models import TransacaoPagamento, WebhookLog, ConfiguracaoPagamento, PlanoAnuncioPagamento, Voucher, VoucherUso
from models.models import Modelo, PlanoAtivoLog
from .services import voucher_service


@admin.register(TransacaoPagamento)
class TransacaoPagamentoAdmin(admin.ModelAdmin):
    list_display = ['id', 'tipo', 'valor', 'status', 'modelo', 'usuario', 'data_criacao']
    list_filter = ['tipo', 'status', 'data_criacao']
    search_fields = ['external_reference', 'payment_id', 'email_pagador', 'nome_pagador']
    readonly_fields = ['id', 'external_reference', 'data_criacao', 'data_atualizacao']
    
    fieldsets = (
        ('Informações Básicas', {
            'fields': ('id', 'external_reference', 'tipo', 'valor', 'status')
        }),
        ('Dados do Mercado Pago', {
            'fields': ('payment_id', 'preference_id', 'merchant_order_id')
        }),
        ('Dados do Pagador', {
            'fields': ('email_pagador', 'nome_pagador')
        }),
        ('Relacionamentos', {
            'fields': ('modelo', 'usuario')
        }),
        ('Metadados', {
            'fields': ('descricao', 'dados_adicional')
        }),
        ('Timestamps', {
            'fields': ('data_criacao', 'data_atualizacao', 'data_aprovacao'),
            'classes': ('collapse',)
        }),
    )




@admin.register(WebhookLog)
class WebhookLogAdmin(admin.ModelAdmin):
    list_display = ['tipo', 'processado', 'data_recebimento', 'data_processamento']
    list_filter = ['tipo', 'processado', 'data_recebimento']
    search_fields = ['tipo']
    readonly_fields = ['tipo', 'dados_recebidos', 'data_recebimento']
    
    fieldsets = (
        ('Informações Básicas', {
            'fields': ('tipo', 'processado', 'erro')
        }),
        ('Dados Recebidos', {
            'fields': ('dados_recebidos',),
            'classes': ('collapse',)
        }),
        ('Timestamps', {
            'fields': ('data_recebimento', 'data_processamento')
        }),
    )


@admin.register(ConfiguracaoPagamento)
class ConfiguracaoPagamentoAdmin(admin.ModelAdmin):
    list_display = ['chave', 'valor', 'ativo', 'data_atualizacao']
    list_filter = ['ativo', 'data_criacao']
    search_fields = ['chave', 'descricao']
    readonly_fields = ['data_criacao', 'data_atualizacao']


@admin.register(PlanoAnuncioPagamento)
class PlanoAnuncioPagamentoAdmin(admin.ModelAdmin):
    list_display = ['pagamento', 'plano', 'data_inicio', 'data_fim', 'status']
    list_filter = ['plano__tipo', 'data_inicio', 'data_fim']
    search_fields = ['pagamento__external_reference', 'plano__nome']
    readonly_fields = ['pagamento', 'plano', 'data_inicio', 'data_fim']
    
    def status(self, obj):
        return obj.pagamento.status
    status.short_description = 'Status'
    
    fieldsets = (
        ('Informações do Pagamento', {
            'fields': ('pagamento', 'plano')
        }),
        ('Datas', {
            'fields': ('data_inicio', 'data_fim')
        })
    )


class VoucherUsoInline(admin.TabularInline):
    model = VoucherUso
    extra = 0
    fields = ['modelo', 'valor_original', 'valor_desconto', 'valor_final', 'data_uso']
    readonly_fields = ['data_uso']
    
    def has_add_permission(self, request, obj=None):
        return False


@admin.register(Voucher)
class VoucherAdmin(admin.ModelAdmin):
    list_display = ['codigo', 'nome', 'plano', 'tipo_desconto', 'status_badge', 'disponibilidade', 'usos_display', 'data_fim']
    list_filter = ['tipo_desconto', 'categoria', 'status', 'plano__tipo', 'data_criacao']
    search_fields = ['codigo', 'nome', 'descricao']
    readonly_fields = ['data_criacao', 'data_atualizacao', 'quantidade_usada']
    inlines = [VoucherUsoInline]
    actions = ['aplicar_voucher_modelo']
    
    fieldsets = (
        ('Identificação', {
            'fields': ('codigo', 'nome', 'descricao')
        }),
        ('Configuração do Desconto', {
            'fields': ('plano', 'tipo_desconto', 'valor_desconto', 'percentual_desconto')
        }),
        ('Limitações', {
            'fields': ('quantidade_maxima', 'quantidade_usada', 'modelo_especifico')
        }),
        ('Validade', {
            'fields': ('data_inicio', 'data_fim')
        }),
        ('Categorização', {
            'fields': ('categoria', 'status')
        }),
        ('Controle', {
            'fields': ('criado_por', 'observacoes')
        }),
        ('Timestamps', {
            'fields': ('data_criacao', 'data_atualizacao'),
            'classes': ('collapse',)
        }),
    )
    
    def status_badge(self, obj):
        """Exibe o status com badge colorido"""
        from django.utils.html import format_html
        status_colors = {
            'ativo': 'success',
            'inativo': 'secondary',
            'expirado': 'danger',
            'esgotado': 'warning',
        }
        color = status_colors.get(obj.status, 'secondary')
        return format_html(
            '<span class="badge bg-{}">{}</span>',
            color, obj.get_status_display()
        )
    status_badge.short_description = 'Status'
    
    def disponibilidade(self, obj):
        """Mostra se o voucher está disponível"""
        from django.utils.html import format_html
        if obj.disponivel:
            return format_html('<span class="text-success">✅ Disponível</span>')
        else:
            return format_html('<span class="text-danger">❌ Indisponível</span>')
    disponibilidade.short_description = 'Disponibilidade'
    
    def usos_display(self, obj):
        """Mostra quantidade de usos"""
        return f"{obj.quantidade_usada}/{obj.quantidade_maxima}"
    usos_display.short_description = 'Usos'
    
    def save_model(self, request, obj, form, change):
        if not change:  # Se é um novo voucher
            obj.criado_por = request.user
        super().save_model(request, obj, form, change)
    
    def aplicar_voucher_modelo(self, request, queryset):
        """Aplica vouchers selecionados a uma modelo específica"""
        if request.POST.get('post'):
            # Processar o formulário
            modelo_id = request.POST.get('modelo_id')
            observacoes = request.POST.get('observacoes', '')
            
            if not modelo_id:
                self.message_user(request, 'Por favor, selecione uma modelo.', messages.ERROR)
                return
            
            try:
                modelo = Modelo.objects.get(id=modelo_id)
                vouchers_aplicados = 0
                erros = []
                
                for voucher in queryset:
                    # Validar se o voucher pode ser usado
                    if not voucher.esta_disponivel():
                        erros.append(f"Voucher {voucher.codigo}: Não está disponível")
                        continue
                    
                    if not voucher.pode_ser_usado_por(modelo):
                        erros.append(f"Voucher {voucher.codigo}: Não pode ser usado por esta modelo")
                        continue
                    
                    # Calcular valores
                    valor_original = float(voucher.plano.preco)
                    resultado = voucher_service.aplicar_voucher(voucher, modelo, valor_original)
                    
                    if resultado['sucesso']:
                        valor_final = resultado['valor_final']
                        valor_desconto = resultado['valor_desconto']
                        
                        # Ativar o plano na modelo
                        agora = timezone.now()
                        data_fim = agora + timedelta(days=voucher.plano.duracao_dias)
                        
                        modelo.status = 'ativo'
                        modelo.pagamento_aprovado = True
                        modelo.data_pagamento = agora
                        modelo.proximo_pagamento = data_fim
                        modelo.plano_ativo = voucher.plano
                        modelo.plano_data_inicio = agora
                        modelo.plano_data_fim = data_fim
                        modelo.save()
                        
                        # Criar log de plano ativo
                        PlanoAtivoLog.objects.create(
                            modelo=modelo,
                            plano=voucher.plano,
                            tipo_ativacao='admin',
                            data_inicio=agora,
                            data_fim=data_fim,
                            observacoes=observacoes or f'Plano ativado via voucher {voucher.codigo}'
                        )
                        
                        # Registrar uso do voucher
                        voucher_service.registrar_uso_voucher(
                            voucher=voucher,
                            modelo=modelo,
                            transacao=None,  # Não há transação real
                            valor_original=valor_original,
                            valor_desconto=valor_desconto,
                            valor_final=valor_final,
                            observacoes=observacoes or 'Aplicado pelo admin'
                        )
                        
                        vouchers_aplicados += 1
                    else:
                        erros.append(f"Voucher {voucher.codigo}: {resultado.get('erro', 'Erro desconhecido')}")
                
                # Mostrar resultados
                if vouchers_aplicados > 0:
                    self.message_user(
                        request,
                        f'{vouchers_aplicados} voucher(s) aplicado(s) com sucesso para {modelo.nome_exibicao}!',
                        messages.SUCCESS
                    )
                
                if erros:
                    for erro in erros:
                        self.message_user(request, erro, messages.WARNING)
                
            except Modelo.DoesNotExist:
                self.message_user(request, 'Modelo não encontrada.', messages.ERROR)
        else:
            # Mostrar formulário
            modelos = Modelo.objects.filter(status__in=['aprovado', 'ativo']).order_by('nome_exibicao')
            context = {
                'queryset': queryset,
                'modelos': modelos,
                'action_name': 'aplicar_voucher_modelo',
                'title': 'Aplicar Voucher(s) a Modelo'
            }
            return render(request, 'admin/aplicar_voucher_modelo.html', context)
    
    aplicar_voucher_modelo.short_description = "🎫 Aplicar Voucher a Modelo"


@admin.register(VoucherUso)
class VoucherUsoAdmin(admin.ModelAdmin):
    list_display = ['voucher', 'modelo', 'valor_original', 'valor_desconto', 'valor_final', 'data_uso']
    list_filter = ['voucher__categoria', 'voucher__tipo_desconto', 'data_uso']
    search_fields = ['voucher__codigo', 'modelo__nome_exibicao', 'modelo__nome']
    readonly_fields = ['data_uso']
    
    fieldsets = (
        ('Informações do Voucher', {
            'fields': ('voucher', 'modelo', 'transacao')
        }),
        ('Valores', {
            'fields': ('valor_original', 'valor_desconto', 'valor_final')
        }),
        ('Detalhes', {
            'fields': ('observacoes', 'data_uso')
        }),
    )
    
    def has_add_permission(self, request):
        return False  # Vouchers são criados automaticamente quando usados
