from django.db import models
from django.contrib.auth.models import User
from models.models import Modelo
import uuid


class TransacaoPagamento(models.Model):
    """Modelo principal para pagamentos"""
    
    STATUS_CHOICES = [
        ('pendente', 'Pendente'),
        ('aprovado', 'Aprovado'),
        ('rejeitado', 'Rejeitado'),
        ('cancelado', 'Cancelado'),
        ('processando', 'Processando'),
        ('falhou', 'Falhou'),
    ]
    
    TIPO_CHOICES = [
        ('plano_anuncio', 'Plano de Anúncio'),
        ('servico', 'Serviço'),
        ('outro', 'Outro'),
    ]
    
    # Identificadores únicos
    id = models.UUIDField(primary_key=True, default=uuid.uuid4, editable=False)
    external_reference = models.CharField(max_length=100, unique=True, blank=True)
    
    # Dados do pagamento
    tipo = models.CharField(max_length=20, choices=TIPO_CHOICES)
    valor = models.DecimalField(max_digits=10, decimal_places=2)
    status = models.CharField(max_length=20, choices=STATUS_CHOICES, default='pendente')
    
    # Dados do Mercado Pago
    payment_id = models.CharField(max_length=100, blank=True, null=True)
    preference_id = models.CharField(max_length=100, blank=True, null=True)
    merchant_order_id = models.CharField(max_length=100, blank=True, null=True)
    
    # Dados do pagador
    email_pagador = models.EmailField(blank=True)
    nome_pagador = models.CharField(max_length=200, blank=True)
    
    # Relacionamentos
    modelo = models.ForeignKey(Modelo, on_delete=models.CASCADE, related_name='transacoes_pagamento', null=True, blank=True)
    usuario = models.ForeignKey(User, on_delete=models.CASCADE, related_name='transacoes_pagamento', null=True, blank=True)
    
    # Metadados
    descricao = models.TextField(blank=True)
    dados_adicional = models.JSONField(default=dict, blank=True)
    
    # Timestamps
    data_criacao = models.DateTimeField(auto_now_add=True)
    data_atualizacao = models.DateTimeField(auto_now=True)
    data_aprovacao = models.DateTimeField(null=True, blank=True)
    
    class Meta:
        ordering = ['-data_criacao']
        verbose_name = 'Pagamento'
        verbose_name_plural = 'Pagamentos'
    
    def __str__(self):
        return f'Transação {self.id} - {self.tipo} - R$ {self.valor}'
    
    def save(self, *args, **kwargs):
        if not self.external_reference:
            self.external_reference = f"allure_{self.id}"
        super().save(*args, **kwargs)




class WebhookLog(models.Model):
    """Log de webhooks recebidos do Mercado Pago"""
    
    TIPO_CHOICES = [
        ('payment', 'Payment'),
        ('preference', 'Preference'),
        ('order', 'Order'),
        ('other', 'Other'),
    ]
    
    tipo = models.CharField(max_length=20, choices=TIPO_CHOICES)
    dados_recebidos = models.JSONField()
    processado = models.BooleanField(default=False)
    erro = models.TextField(blank=True)
    data_recebimento = models.DateTimeField(auto_now_add=True)
    data_processamento = models.DateTimeField(null=True, blank=True)
    
    class Meta:
        ordering = ['-data_recebimento']
        verbose_name = 'Log de Webhook'
        verbose_name_plural = 'Logs de Webhook'
    
    def __str__(self):
        return f'Webhook {self.tipo} - {self.data_recebimento}'


class ConfiguracaoPagamento(models.Model):
    """Configurações do sistema de pagamento"""
    
    chave = models.CharField(max_length=100, unique=True)
    valor = models.TextField()
    descricao = models.TextField(blank=True)
    ativo = models.BooleanField(default=True)
    data_criacao = models.DateTimeField(auto_now_add=True)
    data_atualizacao = models.DateTimeField(auto_now=True)
    
    class Meta:
        verbose_name = 'Configuração de Pagamento'
        verbose_name_plural = 'Configurações de Pagamento'
    
    def __str__(self):
        return f'{self.chave}: {self.valor}'


class PlanoAnuncioPagamento(models.Model):
    """Modelo específico para pagamentos de planos de anúncio"""
    
    pagamento = models.OneToOneField(TransacaoPagamento, on_delete=models.CASCADE, related_name='plano_anuncio')
    plano = models.ForeignKey('models.PlanoAnuncio', on_delete=models.CASCADE)
    data_inicio = models.DateTimeField(null=True, blank=True)
    data_fim = models.DateTimeField(null=True, blank=True)
    
    class Meta:
        verbose_name = 'Pagamento de Plano de Anúncio'
        verbose_name_plural = 'Pagamentos de Planos de Anúncio'
    
    def __str__(self):
        return f'Plano {self.plano.nome} - {self.pagamento.id}'


class Voucher(models.Model):
    """Modelo para vouchers de desconto/cortesia"""
    
    TIPO_DESCONTO_CHOICES = [
        ('gratuito', 'Gratuito'),
        ('desconto_fixo', 'Desconto Valor Fixo'),
        ('desconto_percentual', 'Desconto Percentual'),
        ('extensao', 'Extensão de Plano'),
    ]
    
    CATEGORIA_CHOICES = [
        ('promocional', 'Promocional'),
        ('cortesia', 'Cortesia'),
        ('parceria', 'Parceria'),
        ('marketing', 'Marketing'),
        ('suporte', 'Suporte'),
    ]
    
    STATUS_CHOICES = [
        ('ativo', 'Ativo'),
        ('inativo', 'Inativo'),
        ('expirado', 'Expirado'),
        ('esgotado', 'Esgotado'),
    ]
    
    # Identificação
    codigo = models.CharField(max_length=50, unique=True, verbose_name='Código do Voucher')
    nome = models.CharField(max_length=100, verbose_name='Nome do Voucher')
    descricao = models.TextField(blank=True, verbose_name='Descrição')
    
    # Configuração
    plano = models.ForeignKey('models.PlanoAnuncio', on_delete=models.CASCADE, verbose_name='Plano de Anúncio')
    tipo_desconto = models.CharField(max_length=20, choices=TIPO_DESCONTO_CHOICES, verbose_name='Tipo de Desconto')
    valor_desconto = models.DecimalField(max_digits=10, decimal_places=2, null=True, blank=True, verbose_name='Valor do Desconto')
    percentual_desconto = models.PositiveIntegerField(null=True, blank=True, verbose_name='Percentual de Desconto (%)')
    
    # Limitações
    quantidade_maxima = models.PositiveIntegerField(default=1, verbose_name='Quantidade Máxima de Uso')
    quantidade_usada = models.PositiveIntegerField(default=0, verbose_name='Quantidade Usada')
    modelo_especifico = models.ForeignKey('models.Modelo', on_delete=models.CASCADE, null=True, blank=True, verbose_name='Modelo Específico')
    
    # Validade
    data_inicio = models.DateTimeField(verbose_name='Data de Início')
    data_fim = models.DateTimeField(verbose_name='Data de Fim')
    
    # Categorização
    categoria = models.CharField(max_length=20, choices=CATEGORIA_CHOICES, default='promocional', verbose_name='Categoria')
    status = models.CharField(max_length=20, choices=STATUS_CHOICES, default='ativo', verbose_name='Status')
    
    # Controle
    criado_por = models.ForeignKey('auth.User', on_delete=models.SET_NULL, null=True, blank=True, verbose_name='Criado por')
    observacoes = models.TextField(blank=True, verbose_name='Observações')
    
    # Timestamps
    data_criacao = models.DateTimeField(auto_now_add=True, verbose_name='Data de Criação')
    data_atualizacao = models.DateTimeField(auto_now=True, verbose_name='Data de Atualização')
    
    class Meta:
        verbose_name = 'Voucher'
        verbose_name_plural = 'Vouchers'
        ordering = ['-data_criacao']
    
    def __str__(self):
        return f'{self.codigo} - {self.nome}'
    
    @property
    def disponivel(self):
        """Verifica se o voucher está disponível para uso"""
        from django.utils import timezone
        agora = timezone.now()
        
        if self.status != 'ativo':
            return False
        
        if agora < self.data_inicio or agora > self.data_fim:
            return False
        
        if self.quantidade_usada >= self.quantidade_maxima:
            return False
        
        return True
    
    @property
    def dias_restantes(self):
        """Calcula quantos dias restam para expirar"""
        from django.utils import timezone
        agora = timezone.now()
        if agora < self.data_fim:
            return (self.data_fim - agora).days
        return 0
    
    def pode_ser_usado_por(self, modelo):
        """Verifica se o voucher pode ser usado por uma modelo específica"""
        if not self.disponivel:
            return False
        
        if self.modelo_especifico and self.modelo_especifico != modelo:
            return False
        
        return True
    
    def calcular_valor_final(self, valor_original):
        """Calcula o valor final após aplicar o desconto"""
        if self.tipo_desconto == 'gratuito':
            return 0
        
        elif self.tipo_desconto == 'desconto_fixo' and self.valor_desconto:
            return max(0, valor_original - self.valor_desconto)
        
        elif self.tipo_desconto == 'desconto_percentual' and self.percentual_desconto:
            desconto = (valor_original * self.percentual_desconto) / 100
            return max(0, valor_original - desconto)
        
        return valor_original


class VoucherUso(models.Model):
    """Modelo para registrar o uso de vouchers"""
    
    voucher = models.ForeignKey(Voucher, on_delete=models.CASCADE, related_name='usos')
    modelo = models.ForeignKey('models.Modelo', on_delete=models.CASCADE, related_name='vouchers_usados')
    transacao = models.ForeignKey(TransacaoPagamento, on_delete=models.CASCADE, related_name='voucher_usado', null=True, blank=True)
    
    valor_original = models.DecimalField(max_digits=10, decimal_places=2, verbose_name='Valor Original')
    valor_desconto = models.DecimalField(max_digits=10, decimal_places=2, verbose_name='Valor do Desconto')
    valor_final = models.DecimalField(max_digits=10, decimal_places=2, verbose_name='Valor Final')
    
    observacoes = models.TextField(blank=True, verbose_name='Observações')
    data_uso = models.DateTimeField(auto_now_add=True, verbose_name='Data de Uso')
    
    class Meta:
        verbose_name = 'Uso de Voucher'
        verbose_name_plural = 'Usos de Vouchers'
        ordering = ['-data_uso']
    
    def __str__(self):
        return f'{self.voucher.codigo} - {self.modelo.nome_exibicao} - {self.data_uso.strftime("%d/%m/%Y")}'
