from django.shortcuts import render, redirect, get_object_or_404
from django.contrib.auth.decorators import login_required
from django.contrib import messages
from django.http import JsonResponse, HttpResponse
from django.views.decorators.csrf import csrf_exempt
from django.views.decorators.http import require_http_methods
from django.utils import timezone
from django.conf import settings
import json
import logging

from .models import TransacaoPagamento, Voucher
from .services import mercado_pago_service, voucher_service
from models.models import Modelo

# Importar sistema de comunicação
from communication.services import trigger_pagamento_aprovado, trigger_plano_ativado

logger = logging.getLogger(__name__)




def sucesso_pagamento(request, pagamento_id):
    """Página de sucesso após pagamento"""
    try:
        pagamento = get_object_or_404(TransacaoPagamento, id=pagamento_id)
        
        # Verificar se o pagamento foi realmente aprovado
        if pagamento.payment_id:
            payment_data = mercado_pago_service.verificar_status_pagamento(pagamento.payment_id)
            if payment_data and payment_data.get('status') == 'approved':
                pagamento.status = 'aprovado'
                pagamento.save()
        
        context = {
            'pagamento': pagamento,
            'status': pagamento.status
        }
        
        return render(request, 'payments/success.html', context)
        
    except Exception as e:
        logger.error(f"Erro na página de sucesso: {str(e)}")
        messages.error(request, 'Erro ao processar pagamento.')
        return redirect('core:home')


def falha_pagamento(request, pagamento_id):
    """Página de falha após pagamento"""
    try:
        pagamento = get_object_or_404(TransacaoPagamento, id=pagamento_id)
        
        context = {
            'pagamento': pagamento,
            'status': pagamento.status
        }
        
        return render(request, 'payments/failure.html', context)
        
    except Exception as e:
        logger.error(f"Erro na página de falha: {str(e)}")
        messages.error(request, 'Erro ao processar pagamento.')
        return redirect('core:home')


def pendente_pagamento(request, pagamento_id):
    """Página de pagamento pendente"""
    try:
        pagamento = get_object_or_404(TransacaoPagamento, id=pagamento_id)
        
        context = {
            'pagamento': pagamento,
            'status': pagamento.status
        }
        
        return render(request, 'payments/pending.html', context)
        
    except Exception as e:
        logger.error(f"Erro na página de pendente: {str(e)}")
        messages.error(request, 'Erro ao processar pagamento.')
        return redirect('core:home')


@csrf_exempt
@require_http_methods(["POST"])
def webhook(request):
    """Endpoint para receber webhooks do Mercado Pago"""
    try:
        # Verificar se é um webhook válido
        if request.content_type != 'application/json':
            return HttpResponse(status=400)
        
        # Processar dados do webhook
        data = json.loads(request.body)
        
        # Processar webhook
        sucesso = mercado_pago_service.processar_webhook(data)
        
        if sucesso:
            return HttpResponse(status=200)
        else:
            return HttpResponse(status=500)
            
    except json.JSONDecodeError:
        logger.error("Webhook com JSON inválido")
        return HttpResponse(status=400)
    except Exception as e:
        logger.error(f"Erro ao processar webhook: {str(e)}")
        return HttpResponse(status=500)


@login_required
def historico_pagamentos(request):
    """Mostra histórico de pagamentos do usuário"""
    try:
        pagamentos = TransacaoPagamento.objects.filter(
            usuario=request.user
        ).order_by('-data_criacao')
        
        context = {
            'pagamentos': pagamentos
        }
        
        return render(request, 'payments/historico.html', context)
        
    except Exception as e:
        logger.error(f"Erro ao buscar histórico: {str(e)}")
        messages.error(request, 'Erro ao carregar histórico.')
        return redirect('core:home')


@login_required
def detalhes_pagamento(request, pagamento_id):
    """Mostra detalhes de um pagamento específico"""
    try:
        pagamento = get_object_or_404(TransacaoPagamento, id=pagamento_id, usuario=request.user)
        
        # Buscar dados específicos baseado no tipo
        plano_anuncio = None
        
        if pagamento.tipo == 'plano_anuncio':
            try:
                from .models import PlanoAnuncioPagamento
                plano_anuncio = PlanoAnuncioPagamento.objects.get(pagamento=pagamento)
            except:
                pass
        
        context = {
            'pagamento': pagamento,
            'plano_anuncio': plano_anuncio
        }
        
        return render(request, 'payments/detalhes.html', context)
        
    except Exception as e:
        logger.error(f"Erro ao buscar detalhes: {str(e)}")
        messages.error(request, 'Erro ao carregar detalhes.')
        return redirect('payments:historico')


@login_required
def cancelar_pagamento(request, pagamento_id):
    """Cancela um pagamento pendente"""
    if request.method == 'POST':
        try:
            pagamento = get_object_or_404(TransacaoPagamento, id=pagamento_id, usuario=request.user)
            
            if pagamento.status == 'pendente':
                sucesso = mercado_pago_service.cancelar_pagamento(pagamento)
                
                if sucesso:
                    messages.success(request, 'Pagamento cancelado com sucesso.')
                else:
                    messages.error(request, 'Erro ao cancelar pagamento.')
            else:
                messages.warning(request, 'Apenas pagamentos pendentes podem ser cancelados.')
            
            return redirect('payments:detalhes', pagamento_id=pagamento_id)
            
        except Exception as e:
            logger.error(f"Erro ao cancelar pagamento: {str(e)}")
            messages.error(request, 'Erro interno. Tente novamente.')
            return redirect('payments:historico')
    
    return redirect('payments:historico')




def selecionar_plano(request):
    """Página para seleção de planos de anúncio"""
    try:
        from models.models import PlanoAnuncio
        
        planos = PlanoAnuncio.objects.filter(ativo=True).order_by('ordem', 'preco')
        
        # Agrupar por tipo
        planos_por_tipo = {}
        for plano in planos:
            if plano.tipo not in planos_por_tipo:
                planos_por_tipo[plano.tipo] = []
            planos_por_tipo[plano.tipo].append(plano)
        
        # Verificar se é um usuário recém-cadastrado
        usuario_recem_cadastrado = False
        if request.user.is_authenticated:
            try:
                modelo = Modelo.objects.get(user=request.user)
                usuario_recem_cadastrado = modelo.status == 'cadastro_inicial'
            except Modelo.DoesNotExist:
                pass
        
        context = {
            'planos_por_tipo': planos_por_tipo,
            'tipos_planos': PlanoAnuncio.TIPO_PLANO_CHOICES,
            'usuario_recem_cadastrado': usuario_recem_cadastrado
        }
        
        return render(request, 'payments/selecionar_plano.html', context)
        
    except Exception as e:
        logger.error(f"Erro ao carregar planos: {str(e)}")
        messages.error(request, 'Erro ao carregar planos.')
        return redirect('core:home')

def criar_pagamento_plano(request, plano_id):
    """Cria um pagamento para um plano específico"""
    if request.method == 'POST':
        try:
            from models.models import PlanoAnuncio, Modelo
            
            plano = get_object_or_404(PlanoAnuncio, id=plano_id, ativo=True)
            
            # Verificar se usuário está logado e é modelo
            if not request.user.is_authenticated:
                messages.error(request, 'Você precisa estar logado para contratar um plano.')
                return redirect('accounts:login')
            
            try:
                modelo = Modelo.objects.get(user=request.user)
            except Modelo.DoesNotExist:
                messages.error(request, 'Você precisa ter um perfil de modelo ativo.')
                return redirect('core:home')
            
            # Criar pagamento
            from .services import PlanoAnuncioService
            transacao = PlanoAnuncioService.criar_pagamento_plano(
                modelo=modelo,
                plano_id=plano_id,
                email_pagador=request.user.email,
                nome_pagador=f"{request.user.first_name} {request.user.last_name}".strip()
            )
            
            if transacao:
                # Criar preferência no Mercado Pago
                preference = mercado_pago_service.criar_preferencia_pagamento(transacao)
                
                if preference:
                    # Redirecionar para checkout
                    checkout_url = preference.get('init_point')
                    return redirect(checkout_url)
                else:
                    messages.error(request, 'Erro ao criar pagamento. Tente novamente.')
                    transacao.delete()
                    return redirect('payments:selecionar_plano')
            else:
                messages.error(request, 'Erro ao criar pagamento.')
                return redirect('payments:selecionar_plano')
                
        except Exception as e:
            logger.error(f"Erro ao criar pagamento de plano: {str(e)}")
            messages.error(request, 'Erro interno. Tente novamente.')
            return redirect('payments:selecionar_plano')
    
    return redirect('payments:selecionar_plano')

def test_payment(request):
    """Página de teste de pagamento"""
    from django.conf import settings
    
    context = {
        'site_url': settings.SITE_URL,
        'sandbox': getattr(settings, 'MERCADO_PAGO_SANDBOX', False)
    }
    
    return render(request, 'payments/test_payment.html', context)

def teste_planos(request):
    """Página de teste para verificar se os planos estão funcionando"""
    try:
        from models.models import PlanoAnuncio
        
        planos = PlanoAnuncio.objects.all().order_by('ordem', 'tipo', 'preco')
        
        context = {
            'planos': planos
        }
        
        return render(request, 'payments/teste_planos.html', context)
        
    except Exception as e:
        logger.error(f"Erro ao carregar planos de teste: {str(e)}")
        messages.error(request, 'Erro ao carregar planos.')
        return redirect('core:home')

def teste_pagamento_simulado(request, pagamento_id):
    """Página para simular pagamento em modo de teste"""
    try:
        pagamento = get_object_or_404(TransacaoPagamento, id=pagamento_id)
        
        if request.method == 'POST':
            acao = request.POST.get('acao')
            
            if acao == 'aprovar':
                # Simular pagamento aprovado
                pagamento.status = 'aprovado'
                pagamento.payment_id = f'test_payment_{pagamento.id}'
                pagamento.data_aprovacao = timezone.now()
                pagamento.save()
                
                # Ativar serviço
                from .services import mercado_pago_service
                mercado_pago_service._ativar_servico(pagamento)
                
                messages.success(request, 'Pagamento simulado e aprovado com sucesso!')
                return redirect('payments:success', pagamento_id=pagamento.id)
                
            elif acao == 'rejeitar':
                # Simular pagamento rejeitado
                pagamento.status = 'rejeitado'
                pagamento.save()
                
                messages.error(request, 'Pagamento simulado e rejeitado.')
                return redirect('payments:failure', pagamento_id=pagamento.id)
        
        context = {
            'pagamento': pagamento,
            'modo_teste': True
        }
        
        return render(request, 'payments/teste_pagamento.html', context)
        
    except Exception as e:
        logger.error(f"Erro na simulação de pagamento: {str(e)}")
        messages.error(request, 'Erro ao simular pagamento.')
        return redirect('core:home')


def teste_fluxo_completo(request):
    """Página de teste para simular o fluxo completo de pagamento"""
    mensagens = []
    
    if request.method == 'POST':
        acao = request.POST.get('acao')
        
        if acao == 'criar_modelo_teste':
            try:
                from django.contrib.auth.models import User
                
                # Criar usuário de teste
                user, created = User.objects.get_or_create(
                    username='modelo_teste',
                    defaults={
                        'email': 'modelo.teste@allure.com',
                        'first_name': 'Modelo',
                        'last_name': 'Teste',
                        'is_active': True
                    }
                )
                
                if created:
                    user.set_password('teste123')
                    user.save()
                
                # Criar modelo de teste
                modelo, created = Modelo.objects.get_or_create(
                    user=user,
                    defaults={
                        'nome': 'Modelo',
                        'sobrenome': 'Teste',
                        'cpf': '12345678901',
                        'data_nascimento': '1990-01-01',
                        'nome_exibicao': 'Modelo Teste',
                        'idade': 30,
                        'estado': 'SP',
                        'cidade': 'São Paulo',
                        'etnia': 'branca',
                        'categoria_servico': 'JOB',
                        'tipo_ensaio': 'solo',
                        'local_atendimento': 'local_proprio',
                        'status': 'cadastro_inicial'
                    }
                )
                
                mensagens.append({
                    'tipo': 'success',
                    'icone': 'user-plus',
                    'texto': 'Modelo de teste criada com sucesso! Agora você pode selecionar um plano.'
                })
                
            except Exception as e:
                mensagens.append({
                    'tipo': 'error',
                    'icone': 'exclamation-triangle',
                    'texto': f'Erro ao criar modelo de teste: {str(e)}'
                })
            
        elif acao == 'selecionar_plano':
            try:
                # Buscar modelo de teste
                modelo = Modelo.objects.get(user__username='modelo_teste')
                
                # Buscar plano básico
                from models.models import PlanoAnuncio
                plano = PlanoAnuncio.objects.get(nome='Básico', ativo=True)
                
                # Criar pagamento
                from .services import PlanoAnuncioService
                transacao = PlanoAnuncioService.criar_pagamento_plano(
                    modelo=modelo,
                    plano_id=plano.id,
                    email_pagador=modelo.user.email,
                    nome_pagador=f"{modelo.user.first_name} {modelo.user.last_name}"
                )
                
                if transacao:
                    mensagens.append({
                        'tipo': 'success',
                        'icone': 'crown',
                        'texto': f'Plano selecionado e transação criada! ID: {transacao.id}'
                    })
                else:
                    mensagens.append({
                        'tipo': 'error',
                        'icone': 'exclamation-triangle',
                        'texto': 'Erro ao criar transação. Verifique os logs.'
                    })
                    
            except Exception as e:
                mensagens.append({
                    'tipo': 'error',
                    'icone': 'exclamation-triangle',
                    'texto': f'Erro ao selecionar plano: {str(e)}'
                })
                
        elif acao == 'simular_pagamento':
            try:
                # Buscar transação pendente
                transacao = TransacaoPagamento.objects.filter(
                    tipo='plano_anuncio',
                    status='pendente'
                ).first()
                
                if transacao:
                    # Simular aprovação
                    transacao.status = 'aprovado'
                    transacao.data_aprovacao = timezone.now()
                    transacao.save()
                    
                    # Ativar plano
                    from .services import PlanoAnuncioService
                    sucesso = PlanoAnuncioService.ativar_plano_anuncio(transacao)
                    
                    if sucesso:
                        mensagens.append({
                            'tipo': 'success',
                            'icone': 'check-circle',
                            'texto': 'Pagamento simulado e plano ativado com sucesso!'
                        })
                    else:
                        mensagens.append({
                            'tipo': 'error',
                            'icone': 'exclamation-triangle',
                            'texto': 'Erro ao ativar plano após pagamento.'
                        })
                else:
                    mensagens.append({
                        'tipo': 'warning',
                        'icone': 'info-circle',
                        'texto': 'Nenhuma transação pendente encontrada. Selecione um plano primeiro.'
                    })
                    
            except Exception as e:
                mensagens.append({
                    'tipo': 'error',
                    'icone': 'exclamation-triangle',
                    'texto': f'Erro ao simular pagamento: {str(e)}'
                })
                
        elif acao == 'limpar_teste':
            try:
                # Remover modelo de teste e transações relacionadas
                from django.contrib.auth.models import User
                
                user_teste = User.objects.filter(username='modelo_teste').first()
                if user_teste:
                    # Remover transações primeiro
                    transacoes = TransacaoPagamento.objects.filter(usuario=user_teste)
                    for transacao in transacoes:
                        # Remover plano de pagamento
                        try:
                            from .models import PlanoAnuncioPagamento
                            plano_pagamento = PlanoAnuncioPagamento.objects.get(pagamento=transacao)
                            plano_pagamento.delete()
                        except:
                            pass
                        transacao.delete()
                    
                    # Remover modelo
                    try:
                        modelo = Modelo.objects.get(user=user_teste)
                        modelo.delete()
                    except:
                        pass
                    
                    # Remover usuário
                    user_teste.delete()
                    
                    mensagens.append({
                        'tipo': 'success',
                        'icone': 'trash',
                        'texto': 'Dados de teste removidos com sucesso!'
                    })
                else:
                    mensagens.append({
                        'tipo': 'info',
                        'icone': 'info-circle',
                        'texto': 'Nenhum dado de teste encontrado para remover.'
                    })
                    
            except Exception as e:
                mensagens.append({
                    'tipo': 'error',
                    'icone': 'exclamation-triangle',
                    'texto': f'Erro ao limpar dados de teste: {str(e)}'
                })
    
    # Preparar contexto
    try:
        from models.models import PlanoAnuncio
        
        # Estatísticas gerais
        total_modelos = Modelo.objects.count()
        modelos_pendentes_pagamento = Modelo.objects.filter(status='pendente_pagamento').count()
        modelos_ativas = Modelo.objects.filter(status='ativo').count()
        total_transacoes = TransacaoPagamento.objects.count()
        
        # Modelo de teste
        modelo_teste = None
        try:
            modelo_teste = Modelo.objects.get(user__username='modelo_teste')
        except Modelo.DoesNotExist:
            pass
        
        # Transações de teste
        transacoes_teste = []
        if modelo_teste:
            transacoes_teste = TransacaoPagamento.objects.filter(
                modelo=modelo_teste
            ).order_by('-data_criacao')
        
        # Verificar se há transação pendente
        transacao_pendente = TransacaoPagamento.objects.filter(
            tipo='plano_anuncio',
            status='pendente'
        ).exists()
        
        context = {
            'total_modelos': total_modelos,
            'modelos_pendentes_pagamento': modelos_pendentes_pagamento,
            'modelos_ativas': modelos_ativas,
            'total_transacoes': total_transacoes,
            'modelo_teste': modelo_teste,
            'transacoes_teste': transacoes_teste,
            'transacao_pendente': transacao_pendente,
            'mensagens': mensagens
        }
        
        return render(request, 'payments/teste_fluxo_completo.html', context)
        
    except Exception as e:
        messages.error(request, f'Erro ao carregar página de teste: {str(e)}')
        return redirect('core:home')


def checkout_plano(request, plano_id):
    """View de checkout com aplicação de voucher"""
    from models.models import PlanoAnuncio
    
    plano = get_object_or_404(PlanoAnuncio, id=plano_id)
    modelo = get_object_or_404(Modelo, user=request.user)
    
    # Verificar se a modelo já tem um plano ativo
    if modelo.planos_anuncio.exists():
        messages.warning(request, 'Você já possui um plano ativo. Entre em contato com o suporte para mais informações.')
        return redirect('accounts:painel')
    
    valor_original = plano.preco
    voucher_aplicado = None
    valor_final = valor_original
    valor_desconto = 0
    erro_voucher = None
    
    if request.method == 'POST':
        codigo_voucher = request.POST.get('codigo_voucher', '').strip().upper()
        
        if codigo_voucher:
            # Validar voucher
            voucher, erro = voucher_service.validar_voucher(codigo_voucher, modelo)
            
            if voucher:
                # Aplicar voucher
                resultado = voucher_service.aplicar_voucher(voucher, modelo, valor_original)
                if resultado:
                    voucher_aplicado = voucher
                    valor_final = resultado['valor_final']
                    valor_desconto = resultado['valor_desconto']
                    messages.success(request, f'Voucher {voucher.codigo} aplicado com sucesso!')
                else:
                    erro_voucher = 'Erro ao aplicar voucher'
            else:
                erro_voucher = erro
                messages.error(request, erro)
        
        # Processar pagamento
        acao = request.POST.get('acao')
        if acao == 'finalizar_pagamento':
            return _processar_pagamento_checkout(request, modelo, plano, valor_final, voucher_aplicado, valor_desconto)
    
    context = {
        'plano': plano,
        'modelo': modelo,
        'valor_original': valor_original,
        'valor_final': valor_final,
        'valor_desconto': valor_desconto,
        'voucher_aplicado': voucher_aplicado,
        'erro_voucher': erro_voucher,
        'eh_gratuito': valor_final == 0,
    }
    
    return render(request, 'payments/checkout.html', context)


def _processar_pagamento_checkout(request, modelo, plano, valor_final, voucher_aplicado, valor_desconto):
    """Processa o pagamento após checkout"""
    try:
        # Criar transação
        transacao = TransacaoPagamento.objects.create(
            tipo='plano_anuncio',
            valor=valor_final,
            status='pending',
            modelo=modelo,
            usuario=request.user,
            external_reference=f"plano_{plano.id}_{modelo.id}_{timezone.now().strftime('%Y%m%d_%H%M%S')}"
        )
        
        # Criar PlanoAnuncioPagamento
        from .models import PlanoAnuncioPagamento
        PlanoAnuncioPagamento.objects.create(
            pagamento=transacao,
            plano=plano
        )
        
        # Se for gratuito (valor_final = 0), ativar diretamente
        if valor_final == 0:
            transacao.status = 'approved'
            transacao.data_aprovacao = timezone.now()
            transacao.save()
            
            # Ativar serviço
            mercado_pago_service._ativar_servico(transacao)
            
            # Registrar uso do voucher se aplicável
            if voucher_aplicado:
                voucher_service.registrar_uso_voucher(
                    voucher=voucher_aplicado,
                    modelo=modelo,
                    transacao=transacao,
                    valor_original=plano.preco,
                    valor_desconto=valor_desconto,
                    valor_final=valor_final,
                    observacoes='Pagamento gratuito via voucher'
                )
            
            messages.success(request, 'Plano ativado com sucesso! Seu voucher foi aplicado.')
            return redirect('payments:success', pagamento_id=transacao.id)
        
        # Se não for gratuito, ir para Mercado Pago
        else:
            # Registrar uso do voucher se aplicável (antes do pagamento)
            if voucher_aplicado:
                voucher_service.registrar_uso_voucher(
                    voucher=voucher_aplicado,
                    modelo=modelo,
                    transacao=transacao,
                    valor_original=plano.preco,
                    valor_desconto=valor_desconto,
                    valor_final=valor_final,
                    observacoes='Voucher aplicado no checkout'
                )
            
            # Criar preferência no Mercado Pago
            preferencia = mercado_pago_service.criar_preferencia_pagamento(transacao)
            
            if preferencia and 'init_point' in preferencia:
                return redirect(preferencia['init_point'])
            else:
                messages.error(request, 'Erro ao criar pagamento. Tente novamente.')
                return redirect('payments:selecionar_plano')
    
    except Exception as e:
        logger.error(f"Erro no checkout: {str(e)}")
        messages.error(request, 'Erro interno. Tente novamente.')
        return redirect('payments:selecionar_plano')


def validar_voucher_ajax(request):
    """View AJAX para validar voucher em tempo real"""
    if request.method == 'POST':
        codigo = request.POST.get('codigo', '').strip().upper()
        plano_id = request.POST.get('plano_id')
        
        if not codigo or not plano_id:
            return JsonResponse({'valido': False, 'erro': 'Dados inválidos'})
        
        try:
            from models.models import PlanoAnuncio
            plano = PlanoAnuncio.objects.get(id=plano_id)
            modelo = get_object_or_404(Modelo, user=request.user)
            
            voucher, erro = voucher_service.validar_voucher(codigo, modelo)
            
            if voucher:
                # Verificar se o voucher é para o plano correto
                if voucher.plano != plano:
                    return JsonResponse({
                        'valido': False, 
                        'erro': f'Voucher válido apenas para o plano: {voucher.plano.nome}'
                    })
                
                # Aplicar voucher
                resultado = voucher_service.aplicar_voucher(voucher, modelo, plano.preco)
                
                if resultado:
                    return JsonResponse({
                        'valido': True,
                        'voucher': {
                            'codigo': voucher.codigo,
                            'nome': voucher.nome,
                            'tipo': voucher.get_tipo_desconto_display(),
                            'valor_original': float(plano.preco),
                            'valor_desconto': float(resultado['valor_desconto']),
                            'valor_final': float(resultado['valor_final']),
                            'eh_gratuito': resultado['valor_final'] == 0
                        }
                    })
                else:
                    return JsonResponse({'valido': False, 'erro': 'Erro ao aplicar voucher'})
            else:
                return JsonResponse({'valido': False, 'erro': erro})
                
        except Exception as e:
            logger.error(f"Erro na validação AJAX: {str(e)}")
            return JsonResponse({'valido': False, 'erro': 'Erro interno'})
    
    return JsonResponse({'valido': False, 'erro': 'Método não permitido'})
