#!/bin/bash

# Script de inicialização para o projeto Allure com Docker

set -e

echo "🚀 Iniciando o projeto Allure..."

# Verificar se o Docker está instalado
if ! command -v docker &> /dev/null; then
    echo "❌ Docker não está instalado. Por favor, instale o Docker primeiro."
    exit 1
fi

# Verificar se o Docker Compose está instalado
if ! command -v docker-compose &> /dev/null; then
    echo "❌ Docker Compose não está instalado. Por favor, instale o Docker Compose primeiro."
    exit 1
fi

# Criar diretórios necessários
mkdir -p logs
mkdir -p media
mkdir -p staticfiles

# Verificar se existe arquivo .env
if [ ! -f .env ]; then
    echo "📝 Criando arquivo .env a partir do exemplo..."
    cp env.example .env
    echo "⚠️  Por favor, edite o arquivo .env com suas configurações antes de continuar."
    exit 1
fi

# Função para mostrar ajuda
show_help() {
    echo "Uso: $0 [comando]"
    echo ""
    echo "Comandos disponíveis:"
    echo "  dev     - Iniciar em modo desenvolvimento"
    echo "  prod    - Iniciar em modo produção"
    echo "  build   - Construir as imagens Docker"
    echo "  stop    - Parar todos os containers"
    echo "  clean   - Limpar containers e volumes"
    echo "  logs    - Mostrar logs dos containers"
    echo "  shell   - Acessar shell do container web"
    echo "  migrate - Executar migrações do banco"
    echo "  help    - Mostrar esta ajuda"
}

# Função para desenvolvimento
start_dev() {
    echo "🔧 Iniciando em modo desenvolvimento..."
    docker-compose -f docker-compose.dev.yml up --build
}

# Função para produção
start_prod() {
    echo "🏭 Iniciando em modo produção..."
    docker-compose up --build -d
}

# Função para construir imagens
build_images() {
    echo "🔨 Construindo imagens Docker..."
    docker-compose build
}

# Função para parar containers
stop_containers() {
    echo "🛑 Parando containers..."
    docker-compose down
    docker-compose -f docker-compose.dev.yml down
}

# Função para limpar
clean_all() {
    echo "🧹 Limpando containers e volumes..."
    docker-compose down -v
    docker-compose -f docker-compose.dev.yml down -v
    docker system prune -f
}

# Função para mostrar logs
show_logs() {
    echo "📋 Mostrando logs..."
    docker-compose logs -f
}

# Função para acessar shell
access_shell() {
    echo "🐚 Acessando shell do container web..."
    docker-compose exec web bash
}

# Função para executar migrações
run_migrations() {
    echo "🗄️ Executando migrações..."
    docker-compose exec web python manage.py migrate
}

# Processar argumentos
case "${1:-help}" in
    dev)
        start_dev
        ;;
    prod)
        start_prod
        ;;
    build)
        build_images
        ;;
    stop)
        stop_containers
        ;;
    clean)
        clean_all
        ;;
    logs)
        show_logs
        ;;
    shell)
        access_shell
        ;;
    migrate)
        run_migrations
        ;;
    help|*)
        show_help
        ;;
esac
