# Sistema de Busca Avançada - Allure

Este módulo implementa um sistema completo de busca avançada para modelos, com múltiplas estratégias de filtragem, sugestões automáticas e analytics.

## 🚀 Funcionalidades

### Motor de Busca Avançada
- **Busca por texto**: Busca em nomes, cidades, descrições e categorias
- **Filtros múltiplos**: Categoria, tipo de anúncio, localização, idade, preço, características físicas
- **Filtros de status**: Premium, boost ativo, online, disponibilidade
- **Ordenação flexível**: Por relevância, popularidade, avaliação, preço, data
- **Estatísticas em tempo real**: Métricas de busca e resultados

### Sistema de Sugestões
- **Autocomplete inteligente**: Sugestões baseadas em nomes, cidades e categorias
- **Buscas populares**: Termos mais utilizados pelos usuários
- **API REST**: Endpoints para integração com frontend

### Analytics e Métricas
- **Tracking de buscas**: Registro de filtros e resultados
- **Tendências**: Análise de padrões de busca
- **Performance**: Monitoramento de tempo de resposta

## 📁 Estrutura do Módulo

```
search/
├── __init__.py
├── apps.py
├── engine.py          # Motor de busca principal
├── forms.py           # Formulários de busca
├── views.py           # Views e APIs
├── urls.py            # Configuração de URLs
├── tests.py           # Testes automatizados
└── README.md          # Esta documentação
```

## 🔧 Componentes Principais

### AdvancedSearchEngine
Motor principal de busca que implementa:
- Filtros complexos com múltiplos critérios
- Anotações para métricas (visualizações, favoritos, avaliações)
- Ordenação inteligente com priorização de boost
- Cálculo de estatísticas em tempo real

### SearchFilters
Classe de dados para armazenar filtros:
```python
@dataclass
class SearchFilters:
    # Filtros básicos
    categoria_servico: List[str] = None
    tipo_ensaio: List[str] = None
    estado: str = None
    cidade: str = None
    
    # Filtros de idade e preço
    idade_min: int = None
    idade_max: int = None
    preco_min: float = None
    preco_max: float = None
    
    # Filtros de status
    premium: bool = None
    boost_ativo: bool = None
    online: bool = None
    
    # Busca por texto
    termo_busca: str = None
    
    # Ordenação e paginação
    ordenacao: str = 'recentes'
    pagina: int = 1
    por_pagina: int = 20
```

### SearchSuggestionEngine
Motor de sugestões que oferece:
- Busca por nomes de modelos
- Sugestões de cidades
- Categorias populares
- API para autocomplete

## 🎯 Uso Básico

### Busca Simples
```python
from search.engine import AdvancedSearchEngine, SearchFilters

# Inicializar motor
engine = AdvancedSearchEngine()

# Configurar filtros
filters = SearchFilters(
    termo_busca='São Paulo',
    estado='SP',
    idade_min=20,
    idade_max=30
)

# Executar busca
queryset, stats = engine.search(filters)
```

### Busca Avançada via View
```python
from search.views import advanced_search
from django.test import RequestFactory

# Simular requisição
factory = RequestFactory()
request = factory.get('/search/', {
    'estado': 'SP',
    'categoria_servico': ['JOB'],
    'idade_min': '20',
    'idade_max': '30'
})

# Executar view
response = advanced_search(request)
```

## 🌐 APIs Disponíveis

### Busca Avançada
- **URL**: `/search/`
- **Método**: GET
- **Parâmetros**: Todos os filtros via query string
- **Retorno**: Template com resultados paginados

### Busca Rápida
- **URL**: `/search/rapida/`
- **Método**: GET
- **Parâmetros**: `q`, `localidade`, `categoria`
- **Retorno**: Template simplificado

### Sugestões de Busca
- **URL**: `/search/api/sugestoes/`
- **Método**: POST
- **Body**: `{"term": "termo_busca"}`
- **Retorno**: `{"suggestions": ["sugestão1", "sugestão2"]}`

### Cidades por Estado
- **URL**: `/search/api/cidades/{estado}/`
- **Método**: GET
- **Retorno**: `{"cities": ["cidade1", "cidade2"]}`

### Analytics
- **URL**: `/search/api/analytics/`
- **Método**: POST
- **Body**: `{"filters": {...}, "results_count": 10}`

## 🎨 Templates

### advanced_search.html
Template principal com:
- Sidebar de filtros organizados por categoria
- Grid responsivo de resultados
- Paginação avançada
- Estatísticas em tempo real
- Interface moderna com animações

### quick_search.html
Template simplificado com:
- Formulário compacto
- Resultados em grid
- Paginação básica
- Link para busca avançada

## 🧪 Testes

O módulo inclui testes abrangentes:

```bash
# Executar todos os testes do módulo
python manage.py test search

# Executar testes específicos
python manage.py test search.tests.SearchEngineTestCase
python manage.py test search.tests.SearchViewsTestCase
```

### Cobertura de Testes
- ✅ Motor de busca (filtros, ordenação, estatísticas)
- ✅ Formulários (validação, conversão)
- ✅ Views (GET, POST, APIs)
- ✅ Sugestões (autocomplete, buscas populares)
- ✅ Integração (fluxo completo)
- ✅ Performance (tempo de resposta)

## ⚡ Performance

### Otimizações Implementadas
- **Queries otimizadas**: Uso de `select_related` e `prefetch_related`
- **Anotações eficientes**: Cálculo de métricas em uma query
- **Paginação**: Limitação de resultados por página
- **Cache de sugestões**: Redução de consultas ao banco
- **Índices**: Recomendação de índices para campos de busca

### Métricas de Performance
- **Tempo de resposta**: < 1 segundo para buscas complexas
- **Queries por busca**: < 5 queries otimizadas
- **Memória**: Uso eficiente com lazy loading

## 🔧 Configuração

### Instalação
1. Adicionar `'search'` ao `INSTALLED_APPS` em `settings.py`
2. Incluir URLs em `urls.py`:
```python
path('search/', include('search.urls')),
```

### Configurações Opcionais
```python
# settings.py

# Configurações de busca
SEARCH_SETTINGS = {
    'DEFAULT_RESULTS_PER_PAGE': 20,
    'MAX_RESULTS_PER_PAGE': 100,
    'SUGGESTION_LIMIT': 8,
    'POPULAR_SEARCHES_LIMIT': 10,
    'CACHE_TIMEOUT': 300,  # 5 minutos
}
```

## 🚀 Próximas Melhorias

### Funcionalidades Planejadas
- [ ] **Busca semântica**: Análise de contexto e sinônimos
- [ ] **Filtros geográficos**: Busca por proximidade
- [ ] **Histórico de busca**: Salvar buscas do usuário
- [ ] **Filtros salvos**: Favoritar combinações de filtros
- [ ] **Exportação**: CSV, Excel dos resultados
- [ ] **Cache Redis**: Melhorar performance com cache
- [ ] **Elasticsearch**: Motor de busca mais avançado

### Melhorias de UX
- [ ] **Filtros em cascata**: Dependências entre filtros
- [ ] **Busca por voz**: Interface de voz
- [ ] **Filtros visuais**: Sliders e range pickers
- [ ] **Resultados em tempo real**: AJAX sem reload
- [ ] **Mapa de resultados**: Visualização geográfica

## 📊 Analytics e Métricas

### Métricas Coletadas
- **Volume de busca**: Número de buscas por período
- **Filtros populares**: Combinações mais utilizadas
- **Taxa de conversão**: Busca → Visualização de perfil
- **Tempo de sessão**: Engajamento com resultados
- **Buscas sem resultado**: Oportunidades de melhoria

### Dashboard de Analytics
- Gráficos de tendências
- Análise de comportamento
- Relatórios de performance
- Insights para otimização

## 🤝 Contribuição

### Padrões de Código
- **PEP 8**: Formatação de código Python
- **Type hints**: Tipagem estática
- **Docstrings**: Documentação de funções
- **Testes**: Cobertura mínima de 90%

### Estrutura de Commits
```
feat: adicionar filtro por proximidade geográfica
fix: corrigir ordenação por avaliação
docs: atualizar documentação da API
test: adicionar testes para busca semântica
```

## 📞 Suporte

Para dúvidas ou sugestões sobre o sistema de busca:
- **Issues**: Criar issue no repositório
- **Documentação**: Consultar este README
- **Código**: Analisar exemplos em `tests.py`

---

**Desenvolvido para o projeto Allure** 🎭 