"""
Testes para o sistema de busca avançada
"""

from django.test import TestCase, Client
from django.contrib.auth.models import User
from django.urls import reverse
from django.core.files.uploadedfile import SimpleUploadedFile
from models.models import Modelo, Categoria
from .engine import AdvancedSearchEngine, SearchFilters, SearchSuggestionEngine
from .forms import AdvancedSearchForm, QuickSearchForm


class SearchEngineTestCase(TestCase):
    """Testes para o motor de busca"""
    
    def setUp(self):
        """Configuração inicial"""
        self.user = User.objects.create_user(
            username='testuser',
            email='test@example.com',
            password='testpass123'
        )
        
        self.categoria = Categoria.objects.create(
            nome='Teste',
            descricao='Categoria de teste',
            ativo=True
        )
        
        # Criar modelos de teste
        self.modelo1 = Modelo.objects.create(
            user=self.user,
            nome_exibicao='Modelo Teste 1',
            nome='Teste',
            sobrenome='Um',
            cpf='12345678901',
            data_nascimento='1995-01-01',
            idade=25,
            estado='SP',
            cidade='São Paulo',
            etnia='branca',
            categoria_servico='JOB',
            tipo_ensaio='solo',
            local_atendimento='local_proprio',
            status='ativo'
        )
        
        self.modelo2 = Modelo.objects.create(
            user=User.objects.create_user(username='testuser2', password='testpass123'),
            nome_exibicao='Modelo Teste 2',
            nome='Teste',
            sobrenome='Dois',
            cpf='12345678902',
            data_nascimento='1990-01-01',
            idade=30,
            estado='RJ',
            cidade='Rio de Janeiro',
            etnia='parda',
            categoria_servico='FAB',
            tipo_ensaio='dupla',
            local_atendimento='motel',
            status='ativo'
        )
        
        self.search_engine = AdvancedSearchEngine()
    
    def test_basic_search(self):
        """Testa busca básica"""
        filters = SearchFilters(
            termo_busca='São Paulo'
        )
        
        queryset, stats = self.search_engine.search(filters)
        
        self.assertEqual(queryset.count(), 1)
        self.assertEqual(queryset.first(), self.modelo1)
        self.assertEqual(stats['total_results'], 1)
    
    def test_filter_by_state(self):
        """Testa filtro por estado"""
        filters = SearchFilters(estado='SP')
        
        queryset, stats = self.search_engine.search(filters)
        
        self.assertEqual(queryset.count(), 1)
        self.assertEqual(queryset.first().estado, 'SP')
    
    def test_filter_by_age_range(self):
        """Testa filtro por faixa de idade"""
        filters = SearchFilters(
            idade_min=20,
            idade_max=30
        )
        
        queryset, stats = self.search_engine.search(filters)
        
        self.assertEqual(queryset.count(), 2)
        
        # Testar apenas idade mínima
        filters = SearchFilters(idade_min=28)
        queryset, stats = self.search_engine.search(filters)
        self.assertEqual(queryset.count(), 1)
        self.assertEqual(queryset.first(), self.modelo2)
    
    def test_filter_by_service_category(self):
        """Testa filtro por categoria de serviço"""
        filters = SearchFilters(categoria_servico=['JOB'])
        
        queryset, stats = self.search_engine.search(filters)
        
        self.assertEqual(queryset.count(), 1)
        self.assertEqual(queryset.first().categoria_servico, 'JOB')
    
    def test_filter_by_ethnicity(self):
        """Testa filtro por etnia"""
        filters = SearchFilters(etnia=['parda'])
        
        queryset, stats = self.search_engine.search(filters)
        
        self.assertEqual(queryset.count(), 1)
        self.assertEqual(queryset.first().etnia, 'parda')
    
    def test_ordering(self):
        """Testa ordenação"""
        # Ordenação por mais recentes (padrão)
        filters = SearchFilters(ordenacao='recentes')
        queryset, stats = self.search_engine.search(filters)
        
        # Verificar se está ordenado por data de cadastro (mais recente primeiro)
        self.assertEqual(queryset.first(), self.modelo2)
    
    def test_search_stats(self):
        """Testa cálculo de estatísticas"""
        filters = SearchFilters()
        queryset, stats = self.search_engine.search(filters)
        
        self.assertIn('total_results', stats)
        self.assertIn('age_stats', stats)
        self.assertIn('filters_applied', stats)
        self.assertEqual(stats['total_results'], 2)


class SearchFormsTestCase(TestCase):
    """Testes para formulários de busca"""
    
    def test_advanced_search_form_valid(self):
        """Testa formulário de busca avançada válido"""
        form_data = {
            'termo_busca': 'São Paulo',
            'estado': 'SP',
            'idade_min': 20,
            'idade_max': 30,
            'categoria_servico': ['JOB'],
            'ordenacao': 'recentes',
            'por_pagina': '20'
        }
        
        form = AdvancedSearchForm(data=form_data)
        self.assertTrue(form.is_valid())
        
        # Testar conversão para SearchFilters
        search_filters = form.get_search_filters()
        self.assertEqual(search_filters.termo_busca, 'São Paulo')
        self.assertEqual(search_filters.estado, 'SP')
        self.assertEqual(search_filters.idade_min, 20)
        self.assertEqual(search_filters.idade_max, 30)
    
    def test_advanced_search_form_invalid_age(self):
        """Testa validação de idade inválida"""
        form_data = {
            'idade_min': 30,
            'idade_max': 20  # Idade mínima maior que máxima
        }
        
        form = AdvancedSearchForm(data=form_data)
        self.assertFalse(form.is_valid())
        self.assertIn('Idade mínima não pode ser maior que a máxima', str(form.errors))
    
    def test_quick_search_form_valid(self):
        """Testa formulário de busca rápida válido"""
        form_data = {
            'q': 'São Paulo',
            'localidade': 'SP',
            'categoria': 'JOB'
        }
        
        form = QuickSearchForm(data=form_data)
        self.assertTrue(form.is_valid())


class SearchViewsTestCase(TestCase):
    """Testes para views de busca"""
    
    def setUp(self):
        """Configuração inicial"""
        self.client = Client()
        self.user = User.objects.create_user(
            username='testuser',
            password='testpass123'
        )
        
        self.modelo = Modelo.objects.create(
            user=self.user,
            nome_exibicao='Modelo Teste',
            nome='Teste',
            sobrenome='Um',
            cpf='12345678901',
            data_nascimento='1995-01-01',
            idade=25,
            estado='SP',
            cidade='São Paulo',
            etnia='branca',
            categoria_servico='JOB',
            tipo_ensaio='solo',
            local_atendimento='local_proprio',
            status='ativo'
        )
    
    def test_advanced_search_view_get(self):
        """Testa view de busca avançada GET"""
        response = self.client.get(reverse('search:advanced_search'))
        
        self.assertEqual(response.status_code, 200)
        self.assertTemplateUsed(response, 'search/advanced_search.html')
        self.assertIn('form', response.context)
    
    def test_advanced_search_view_with_filters(self):
        """Testa view de busca avançada com filtros"""
        response = self.client.get(reverse('search:advanced_search'), {
            'estado': 'SP',
            'idade_min': '20',
            'idade_max': '30'
        })
        
        self.assertEqual(response.status_code, 200)
        self.assertIn('page_obj', response.context)
        self.assertIn('total_results', response.context)
    
    def test_quick_search_view_get(self):
        """Testa view de busca rápida GET"""
        response = self.client.get(reverse('search:quick_search'))
        
        self.assertEqual(response.status_code, 200)
        self.assertTemplateUsed(response, 'search/quick_search.html')
        self.assertIn('form', response.context)
    
    def test_quick_search_view_with_query(self):
        """Testa view de busca rápida com query"""
        response = self.client.get(reverse('search:quick_search'), {
            'q': 'São Paulo',
            'localidade': 'SP'
        })
        
        self.assertEqual(response.status_code, 200)
        self.assertIn('page_obj', response.context)
    
    def test_search_suggestions_api(self):
        """Testa API de sugestões de busca"""
        response = self.client.post(
            reverse('search:search_suggestions'),
            data={'term': 'São Paulo'},
            content_type='application/json'
        )
        
        self.assertEqual(response.status_code, 200)
        data = response.json()
        self.assertIn('suggestions', data)
    
    def test_get_cities_by_state_api(self):
        """Testa API para obter cidades por estado"""
        response = self.client.get(
            reverse('search:get_cities_by_state', kwargs={'estado': 'SP'})
        )
        
        self.assertEqual(response.status_code, 200)
        data = response.json()
        self.assertIn('cities', data)
        self.assertIn('São Paulo', data['cities'])


class SearchSuggestionEngineTestCase(TestCase):
    """Testes para o motor de sugestões"""
    
    def setUp(self):
        """Configuração inicial"""
        self.user = User.objects.create_user(
            username='testuser',
            password='testpass123'
        )
        
        self.modelo = Modelo.objects.create(
            user=self.user,
            nome_exibicao='Modelo São Paulo',
            nome='São',
            sobrenome='Paulo',
            cpf='12345678901',
            data_nascimento='1995-01-01',
            idade=25,
            estado='SP',
            cidade='São Paulo',
            etnia='branca',
            categoria_servico='JOB',
            tipo_ensaio='solo',
            local_atendimento='local_proprio',
            status='ativo'
        )
        
        self.suggestion_engine = SearchSuggestionEngine()
    
    def test_get_suggestions(self):
        """Testa obtenção de sugestões"""
        suggestions = self.suggestion_engine.get_suggestions('São', limit=5)
        
        self.assertIsInstance(suggestions, list)
        self.assertIn('Modelo São Paulo', suggestions)
        self.assertIn('São Paulo', suggestions)
    
    def test_get_suggestions_empty_term(self):
        """Testa sugestões com termo vazio"""
        suggestions = self.suggestion_engine.get_suggestions('', limit=5)
        self.assertEqual(suggestions, [])
    
    def test_get_suggestions_short_term(self):
        """Testa sugestões com termo muito curto"""
        suggestions = self.suggestion_engine.get_suggestions('a', limit=5)
        self.assertEqual(suggestions, [])
    
    def test_get_popular_searches(self):
        """Testa buscas populares"""
        popular = self.suggestion_engine.get_popular_searches(limit=5)
        
        self.assertIsInstance(popular, list)
        self.assertLessEqual(len(popular), 5)


class SearchIntegrationTestCase(TestCase):
    """Testes de integração do sistema de busca"""
    
    def setUp(self):
        """Configuração inicial"""
        self.client = Client()
        self.user = User.objects.create_user(
            username='testuser',
            password='testpass123'
        )
        
        # Criar múltiplos modelos para testar
        for i in range(5):
            modelo = Modelo.objects.create(
                user=User.objects.create_user(
                    username=f'testuser{i}',
                    password='testpass123'
                ),
                nome_exibicao=f'Modelo {i}',
                nome=f'Teste{i}',
                sobrenome=f'Numero{i}',
                cpf=f'1234567890{i}',
                data_nascimento='1995-01-01',
                idade=20 + i,
                estado='SP' if i % 2 == 0 else 'RJ',
                cidade=f'Cidade {i}',
                etnia='branca' if i % 2 == 0 else 'parda',
                categoria_servico='JOB' if i % 2 == 0 else 'FAB',
                tipo_ensaio='solo',
                local_atendimento='local_proprio',
                status='ativo'
            )
    
    def test_complete_search_workflow(self):
        """Testa fluxo completo de busca"""
        # 1. Acessar página de busca avançada
        response = self.client.get(reverse('search:advanced_search'))
        self.assertEqual(response.status_code, 200)
        
        # 2. Fazer busca com filtros
        response = self.client.get(reverse('search:advanced_search'), {
            'estado': 'SP',
            'categoria_servico': ['JOB'],
            'idade_min': '20',
            'idade_max': '25'
        })
        self.assertEqual(response.status_code, 200)
        
        # 3. Verificar resultados
        self.assertIn('page_obj', response.context)
        self.assertIn('total_results', response.context)
        
        # 4. Testar paginação
        if response.context['page_obj'].has_other_pages():
            response = self.client.get(reverse('search:advanced_search'), {
                'estado': 'SP',
                'page': '2'
            })
            self.assertEqual(response.status_code, 200)
    
    def test_search_performance(self):
        """Testa performance da busca"""
        import time
        
        start_time = time.time()
        
        # Fazer busca complexa
        response = self.client.get(reverse('search:advanced_search'), {
            'estado': 'SP',
            'categoria_servico': ['JOB', 'FAB'],
            'etnia': ['branca', 'parda'],
            'idade_min': '18',
            'idade_max': '99'
        })
        
        end_time = time.time()
        execution_time = end_time - start_time
        
        self.assertEqual(response.status_code, 200)
        # Verificar se a busca não demora mais que 1 segundo
        self.assertLess(execution_time, 1.0) 