#!/usr/bin/env python
"""
Script de setup completo para o painel administrativo Allure
"""
import os
import sys
import subprocess
import django

def run_command(command, description):
    """Executa um comando e mostra o resultado"""
    print(f"\n🔄 {description}...")
    try:
        result = subprocess.run(command, shell=True, check=True, capture_output=True, text=True)
        print(f"✅ {description} concluído com sucesso!")
        return True
    except subprocess.CalledProcessError as e:
        print(f"❌ Erro ao {description.lower()}: {e}")
        print(f"Erro: {e.stderr}")
        return False

def setup_django():
    """Configura o ambiente Django"""
    os.environ.setdefault('DJANGO_SETTINGS_MODULE', 'allure_project.settings')
    django.setup()

def check_dependencies():
    """Verifica se as dependências estão instaladas"""
    print("🔍 Verificando dependências...")
    
    try:
        import django
        print(f"✅ Django {django.get_version()}")
    except ImportError:
        print("❌ Django não encontrado. Instalando...")
        if not run_command("pip install -r requirements.txt", "Instalar dependências"):
            return False
    
    try:
        from PIL import Image
        print("✅ Pillow (PIL) encontrado")
    except ImportError:
        print("❌ Pillow não encontrado. Instalando...")
        if not run_command("pip install Pillow", "Instalar Pillow"):
            return False
    
    return True

def setup_database():
    """Configura o banco de dados"""
    print("\n🗄️ Configurando banco de dados...")
    
    # Criar migrações
    if not run_command("python manage.py makemigrations", "Criar migrações"):
        return False
    
    # Aplicar migrações
    if not run_command("python manage.py migrate", "Aplicar migrações"):
        return False
    
    return True

def create_superuser():
    """Cria um superusuário"""
    print("\n👤 Criando superusuário...")
    
    from django.contrib.auth.models import User
    
    if User.objects.filter(is_superuser=True).exists():
        print("✅ Superusuário já existe!")
        return True
    
    print("Digite as informações do superusuário:")
    username = input("Nome de usuário: ")
    email = input("Email: ")
    password = input("Senha: ")
    confirm_password = input("Confirme a senha: ")
    
    if password != confirm_password:
        print("❌ As senhas não coincidem!")
        return False
    
    try:
        user = User.objects.create_superuser(
            username=username,
            email=email,
            password=password
        )
        print(f"✅ Superusuário '{username}' criado com sucesso!")
        return True
    except Exception as e:
        print(f"❌ Erro ao criar superusuário: {e}")
        return False

def collect_static():
    """Coleta arquivos estáticos"""
    print("\n📁 Coletando arquivos estáticos...")
    return run_command("python manage.py collectstatic --noinput", "Coletar arquivos estáticos")

def show_next_steps():
    """Mostra os próximos passos"""
    print("\n" + "="*60)
    print("🎉 SETUP CONCLUÍDO COM SUCESSO!")
    print("="*60)
    print("\n📋 Próximos passos:")
    print("1. Inicie o servidor: python manage.py runserver")
    print("2. Acesse o painel admin: http://localhost:8000/admin/")
    print("3. Faça login com suas credenciais")
    print("\n📚 Documentação:")
    print("- README_ADMIN.md - Guia completo do painel")
    print("- create_admin.py - Script para gerenciar usuários")
    print("\n🔧 Comandos úteis:")
    print("- python manage.py runserver - Iniciar servidor")
    print("- python create_admin.py create - Criar novo admin")
    print("- python create_admin.py list - Listar usuários")
    print("\n🎭 Painel Administrativo Allure está pronto!")

def main():
    """Função principal"""
    print("🎭 SETUP DO PAINEL ADMINISTRATIVO ALLURE")
    print("="*50)
    
    # Verificar se estamos no diretório correto
    if not os.path.exists("manage.py"):
        print("❌ Erro: Execute este script no diretório allure_project/")
        print("   cd allure_project")
        print("   python setup_admin.py")
        return
    
    # Verificar dependências
    if not check_dependencies():
        print("❌ Falha na verificação de dependências")
        return
    
    # Configurar Django
    setup_django()
    
    # Setup do banco de dados
    if not setup_database():
        print("❌ Falha na configuração do banco de dados")
        return
    
    # Criar superusuário
    if not create_superuser():
        print("❌ Falha na criação do superusuário")
        return
    
    # Coletar arquivos estáticos
    collect_static()
    
    # Mostrar próximos passos
    show_next_steps()

if __name__ == "__main__":
    main() 