#!/usr/bin/env python
"""
Script para testar funcionalidades específicas do projeto Allure
Testa configurações, URLs, templates, banco de dados, autenticação, views, formulários e modelos
"""

import os
import sys
import json
from datetime import datetime
from pathlib import Path

# Adicionar o diretório do projeto ao path
BASE_DIR = Path(__file__).resolve().parent
sys.path.insert(0, str(BASE_DIR))

# Configurações do Django
os.environ.setdefault('DJANGO_SETTINGS_MODULE', 'allure_project.settings')

import django
django.setup()

from django.test import Client
from django.urls import reverse
from django.contrib.auth.models import User
from django.core.files.uploadedfile import SimpleUploadedFile
from django.db import connection
from django.conf import settings
from models.models import Modelo, FotoModelo, Banner
from core.forms import ModeloRegistrationForm


class FunctionalityTester:
    def __init__(self):
        self.results = {
            'timestamp': datetime.now().isoformat(),
            'test_suite': 'Functionality Tests',
            'total_tests': 0,
            'passed': 0,
            'failed': 0,
            'errors': 0,
            'issues_found': [],
            'recommendations': [],
            'execution_time': 0
        }
        self.client = Client()
        self.start_time = datetime.now()
    
    def log_success(self, test_name):
        """Registra teste bem-sucedido"""
        self.results['total_tests'] += 1
        self.results['passed'] += 1
        print(f"   ✅ {test_name}")
    
    def log_failure(self, test_name, error):
        """Registra teste que falhou"""
        self.results['total_tests'] += 1
        self.results['failed'] += 1
        self.results['issues_found'].append({
            'test': test_name,
            'error': str(error),
            'type': 'failure'
        })
        print(f"   ❌ {test_name}: {error}")
    
    def log_error(self, test_name, error):
        """Registra erro no teste"""
        self.results['total_tests'] += 1
        self.results['errors'] += 1
        self.results['issues_found'].append({
            'test': test_name,
            'error': str(error),
            'type': 'error'
        })
        print(f"   💥 {test_name}: {error}")
    
    def test_project_configuration(self):
        """Testa configurações básicas do projeto"""
        print("📋 Testando configurações do projeto...")
        
        try:
            # Verificar se as apps estão instaladas
            required_apps = ['core', 'models', 'accounts']
            for app in required_apps:
                if app in settings.INSTALLED_APPS:
                    self.log_success(f"App '{app}' instalada")
                else:
                    self.log_failure(f"App '{app}' não encontrada", f"App '{app}' não está em INSTALLED_APPS")
            
            # Verificar configurações básicas
            if settings.SECRET_KEY:
                self.log_success("SECRET_KEY configurada")
            else:
                self.log_failure("SECRET_KEY não configurada", "SECRET_KEY está vazia")
            
            if settings.DEBUG is not None:
                self.log_success("DEBUG configurado")
            else:
                self.log_failure("DEBUG não configurado", "DEBUG não está definido")
            
            if settings.ALLOWED_HOSTS:
                self.log_success("ALLOWED_HOSTS configurado")
            else:
                self.log_failure("ALLOWED_HOSTS não configurado", "ALLOWED_HOSTS está vazio")
            
            # Verificar configurações de templates
            if settings.TEMPLATES:
                self.log_success("TEMPLATES configurado")
            else:
                self.log_failure("TEMPLATES não configurado", "TEMPLATES não está definido")
                
        except Exception as e:
            self.log_error("Configurações do projeto", str(e))
    
    def test_urls(self):
        """Testa configuração de URLs"""
        print("🔗 Testando configuração de URLs...")
        
        try:
            # Testar URLs principais
            urls_to_test = [
                ('core:home', 'Página inicial'),
                ('core:modelos_list', 'Lista de modelos'),
                ('core:seja_modelo', 'Cadastro de modelo'),
                ('accounts:login', 'Login'),
                ('accounts:cadastro_modelo', 'Cadastro de modelo'),
            ]
            
            for url_name, description in urls_to_test:
                try:
                    url = reverse(url_name)
                    self.log_success(f"URL {description}")
                except Exception as e:
                    self.log_failure(f"URL {description}", str(e))
                    
        except Exception as e:
            self.log_error("Configuração de URLs", str(e))
    
    def test_templates(self):
        """Testa existência de templates"""
        print("📄 Testando templates...")
        
        try:
            templates_to_test = [
                'base.html',
                'core/home.html',
                'core/modelos_list.html',
                'core/modelo_detail.html',
                'core/seja_modelo.html',
                'accounts/login.html',
                'accounts/cadastro_modelo.html',
                'accounts/editar_perfil.html',
                'accounts/painel_modelo.html',
            ]
            
            for template in templates_to_test:
                template_path = BASE_DIR / 'templates' / template
                if template_path.exists():
                    self.log_success(f"Template {template}")
                else:
                    self.log_failure(f"Template {template}", f"Arquivo não encontrado: {template_path}")
                    
        except Exception as e:
            self.log_error("Templates", str(e))
    
    def test_database(self):
        """Testa operações de banco de dados"""
        print("🗄️ Testando banco de dados...")
        
        try:
            # Testar conexão
            with connection.cursor() as cursor:
                cursor.execute("SELECT 1")
                self.log_success("Conexão com banco de dados")
            
            # Testar criação de usuário
            user = User.objects.create_user(
                username='test_functionality',
                email='test@functionality.com',
                password='testpass123'
            )
            self.log_success("Criação de usuário")
            
            # Testar criação de modelo
            modelo = Modelo.objects.create(
                user=user,
                nome_exibicao='Modelo Funcionalidade',
                idade=25,
                altura=170,
                peso=60,
                cidade='São Paulo',
                estado='SP',
                sobre_mim='Descrição de teste',
                categoria='JOB',
                etnia='branca',
                tipo_ensaio='solo',
                local_atendimento='local_proprio',
            )
            self.log_success("Criação de modelo")
            
            # Testar criação de foto
            image = SimpleUploadedFile(
                "test_image.jpg",
                b"fake-image-content",
                content_type="image/jpeg"
            )
            foto = FotoModelo.objects.create(
                modelo=modelo,
                imagem=image,
                categoria='galeria'
            )
            self.log_success("Criação de foto")
            
            # Testar criação de banner
            banner = Banner.objects.create(
                titulo='Banner Teste',
                imagem=image,
                link='https://example.com',
                ordem=1
            )
            self.log_success("Criação de banner")
            
            # Limpar dados de teste
            modelo.delete()
            user.delete()
            banner.delete()
            
        except Exception as e:
            self.log_error("Banco de dados", str(e))
    
    def test_authentication(self):
        """Testa funcionalidades de autenticação"""
        print("🔐 Testando autenticação...")
        
        try:
            # Criar usuário de teste
            user = User.objects.create_user(
                username='auth_test',
                email='auth@test.com',
                password='testpass123'
            )
            
            # Testar login
            login_success = self.client.login(username='auth_test', password='testpass123')
            if login_success:
                self.log_success("Login de usuário")
            else:
                self.log_failure("Login de usuário", "Falha no login")
            
            # Testar logout
            self.client.logout()
            self.log_success("Logout de usuário")
            
            # Limpar
            user.delete()
            
        except Exception as e:
            self.log_error("Autenticação", str(e))
    
    def test_views(self):
        """Testa views principais"""
        print("👁️ Testando views...")
        
        try:
            # Testar view da página inicial
            response = self.client.get(reverse('core:home'))
            if response.status_code == 200:
                self.log_success("View da página inicial")
            else:
                self.log_failure("View da página inicial", f"Status code: {response.status_code}")
            
            # Testar view de listagem de modelos
            response = self.client.get(reverse('core:modelos_list'))
            if response.status_code == 200:
                self.log_success("View de listagem de modelos")
            else:
                self.log_failure("View de listagem de modelos", f"Status code: {response.status_code}")
            
            # Testar view de cadastro de modelo
            response = self.client.get(reverse('core:seja_modelo'))
            if response.status_code == 200:
                self.log_success("View de cadastro de modelo")
            else:
                self.log_failure("View de cadastro de modelo", f"Status code: {response.status_code}")
                
        except Exception as e:
            self.log_error("Views", str(e))
    
    def test_forms(self):
        """Testa formulários"""
        print("📝 Testando formulários...")
        
        try:
            # Testar formulário de cadastro de modelo
            form_data = {
                'username': 'form_test',
                'email': 'form@test.com',
                'password1': 'testpass123',
                'password2': 'testpass123',
                'nome_exibicao': 'Modelo Form Test',
                'idade': 25,
                'estado': 'SP',
                'cidade': 'São Paulo',
                'etnia': 'branca',
                'categoria': 'JOB',
                'tipo_ensaio': 'solo',
                'local_atendimento': 'local_proprio',
                'aceito_termos': True,
            }
            
            image = SimpleUploadedFile(
                "test_image.jpg",
                b"fake-image-content",
                content_type="image/jpeg"
            )
            
            files = {
                'foto_perfil': image,
                'documento_rg': image,
                'documento_selfie': image,
            }
            
            form = ModeloRegistrationForm(data=form_data, files=files)
            if form.is_valid():
                self.log_success("Formulário de cadastro de modelo")
            else:
                self.log_failure("Formulário de cadastro de modelo", f"Erros: {form.errors}")
                
        except Exception as e:
            self.log_error("Formulários", str(e))
    
    def test_models(self):
        """Testa modelos"""
        print("🏗️ Testando modelos...")
        
        try:
            # Testar modelo Modelo
            user = User.objects.create_user(username='model_test', password='testpass123')
            modelo = Modelo.objects.create(
                user=user,
                nome_exibicao='Modelo Model Test',
                idade=25,
                estado='SP',
                cidade='São Paulo',
                categoria='JOB',
                etnia='branca',
                tipo_ensaio='solo',
                local_atendimento='local_proprio',
            )
            self.log_success("Modelo Modelo")
            
            # Testar modelo FotoModelo
            image = SimpleUploadedFile(
                "test_image.jpg",
                b"fake-image-content",
                content_type="image/jpeg"
            )
            foto = FotoModelo.objects.create(
                modelo=modelo,
                imagem=image,
                categoria='galeria'
            )
            self.log_success("Modelo FotoModelo")
            
            # Testar modelo Banner
            banner = Banner.objects.create(
                titulo='Banner Model Test',
                imagem=image,
                link='https://example.com',
                ordem=1
            )
            self.log_success("Modelo Banner")
            
            # Limpar
            modelo.delete()
            user.delete()
            banner.delete()
            
        except Exception as e:
            self.log_error("Modelos", str(e))
    
    def test_security(self):
        """Testa aspectos de segurança"""
        print("🛡️ Testando segurança...")
        
        try:
            # Testar proteção CSRF
            data = {
                'username': 'security_test',
                'password': 'testpass123',
            }
            
            # Usar client sem CSRF para testar
            client = Client(enforce_csrf_checks=True)
            response = client.post(reverse('accounts:login'), data)
            if response.status_code == 403:  # CSRF error
                self.log_success("Proteção CSRF")
            else:
                self.log_failure("Proteção CSRF", f"Status code: {response.status_code}")
            
            # Testar proteção XSS (Django auto-escape)
            malicious_content = '<script>alert("xss")</script>'
            # Django automaticamente escapa conteúdo no template
            self.log_success("Proteção XSS (Django auto-escape)")
            
        except Exception as e:
            self.log_error("Segurança", str(e))
    
    def run_all_tests(self):
        """Executa todos os testes"""
        print("🔍 TESTANDO FUNCIONALIDADES DO PROJETO ALLURE")
        print("=" * 60)
        
        self.test_project_configuration()
        self.test_urls()
        self.test_templates()
        self.test_database()
        self.test_authentication()
        self.test_views()
        self.test_forms()
        self.test_models()
        self.test_security()
        
        # Calcular tempo de execução
        end_time = datetime.now()
        self.results['execution_time'] = (end_time - self.start_time).total_seconds()
        
        # Gerar recomendações
        self.generate_recommendations()
        
        # Exibir resultados
        self.display_results()
        
        # Salvar relatório
        self.save_report()
    
    def generate_recommendations(self):
        """Gera recomendações baseadas nos problemas encontrados"""
        if self.results['failed'] > 0 or self.results['errors'] > 0:
            self.results['recommendations'].append("🔧 Corrija os problemas identificados acima")
            self.results['recommendations'].append("🧪 Execute os testes novamente após as correções")
            self.results['recommendations'].append("📝 Documente as mudanças realizadas")
        
        self.results['recommendations'].append("🔄 Execute este teste regularmente durante o desenvolvimento")
        self.results['recommendations'].append("📊 Monitore a qualidade do código")
    
    def display_results(self):
        """Exibe os resultados dos testes"""
        print("\n" + "=" * 60)
        print("📊 RELATÓRIO DE FUNCIONALIDADE")
        print("=" * 60)
        
        # Estatísticas
        total = self.results['total_tests']
        passed = self.results['passed']
        failed = self.results['failed']
        errors = self.results['errors']
        success_rate = (passed / total * 100) if total > 0 else 0
        
        print(f"\n📈 ESTATÍSTICAS:")
        print(f"   Total de testes: {total}")
        print(f"   ✅ Passaram: {passed}")
        print(f"   ❌ Falharam: {failed}")
        print(f"   💥 Erros: {errors}")
        print(f"   📊 Taxa de sucesso: {success_rate:.1f}%")
        
        # Problemas encontrados
        if self.results['issues_found']:
            print(f"\n🚨 PROBLEMAS ENCONTRADOS:")
            for issue in self.results['issues_found']:
                print(f"   {issue['test']}: {issue['error']}")
                print(f"      💡 Recomendação: Revisar o código e logs para identificar o problema")
        
        # Recomendações
        if self.results['recommendations']:
            print(f"\n💡 RECOMENDAÇÕES GERAIS:")
            for i, rec in enumerate(self.results['recommendations'], 1):
                print(f"   {i}. {rec}")
        
        # Resumo final
        if failed == 0 and errors == 0:
            print(f"\n🎯 RESUMO FINAL:")
            print(f"   ✅ Todos os testes passaram! Sistema funcionando corretamente.")
        else:
            print(f"\n🎯 RESUMO FINAL:")
            print(f"   ⚠️  {failed + errors} problemas encontrados. Revise as recomendações acima.")
    
    def save_report(self):
        """Salva o relatório em arquivo"""
        try:
            # Salvar em JSON
            json_path = BASE_DIR / 'functionality_report.json'
            with open(json_path, 'w', encoding='utf-8') as f:
                json.dump(self.results, f, indent=2, ensure_ascii=False)
            
            # Salvar em texto
            txt_path = BASE_DIR / 'functionality_report.txt'
            with open(txt_path, 'w', encoding='utf-8') as f:
                f.write("RELATÓRIO DE FUNCIONALIDADE - PROJETO ALLURE\n")
                f.write("=" * 50 + "\n\n")
                f.write(f"Data: {self.results['timestamp']}\n")
                f.write(f"Tempo de execução: {self.results['execution_time']:.2f}s\n\n")
                
                f.write(f"ESTATÍSTICAS:\n")
                f.write(f"Total de testes: {self.results['total_tests']}\n")
                f.write(f"Passaram: {self.results['passed']}\n")
                f.write(f"Falharam: {self.results['failed']}\n")
                f.write(f"Erros: {self.results['errors']}\n\n")
                
                if self.results['issues_found']:
                    f.write("PROBLEMAS ENCONTRADOS:\n")
                    for issue in self.results['issues_found']:
                        f.write(f"- {issue['test']}: {issue['error']}\n")
                    f.write("\n")
                
                if self.results['recommendations']:
                    f.write("RECOMENDAÇÕES:\n")
                    for rec in self.results['recommendations']:
                        f.write(f"- {rec}\n")
            
            print(f"\n📄 Relatório salvo em: {json_path}")
            
        except Exception as e:
            print(f"❌ Erro ao salvar relatório: {e}")


def main():
    """Função principal"""
    tester = FunctionalityTester()
    tester.run_all_tests()


if __name__ == '__main__':
    main() 