#!/usr/bin/env python3
"""
Script para monitorar e compilar SASS automaticamente
"""
import os
import sys
import time
import subprocess
from pathlib import Path
from watchdog.observers import Observer
from watchdog.events import FileSystemEventHandler

class SassHandler(FileSystemEventHandler):
    def __init__(self, compile_func):
        self.compile_func = compile_func
        self.last_compile = 0
        
    def on_modified(self, event):
        if event.is_directory:
            return
            
        # Monitora tanto arquivos .scss quanto .css
        if event.src_path.endswith(('.scss', '.css')):
            # Evita múltiplas compilações em sequência
            current_time = time.time()
            if current_time - self.last_compile > 1:
                file_type = "SCSS" if event.src_path.endswith('.scss') else "CSS"
                print(f"🔄 Arquivo {file_type} alterado: {event.src_path}")
                self.compile_func()
                self.last_compile = current_time

def watch_sass():
    """Monitora mudanças nos arquivos SASS e CSS"""
    project_root = Path(__file__).parent
    scss_dir = project_root / 'static' / 'scss'
    css_dir = project_root / 'static' / 'css'
    
    if not scss_dir.exists():
        print(f"❌ Diretório SASS não encontrado: {scss_dir}")
        return False
    
    def compile_sass():
        try:
            cmd = [
                'sass',
                str(scss_dir / 'main.scss'),
                str(project_root / 'static' / 'css' / 'main.css'),
                '--style=compressed'
            ]
            
            result = subprocess.run(cmd, capture_output=True, text=True)
            
            if result.returncode == 0:
                print("✅ SASS recompilado com sucesso!")
            else:
                print(f"❌ Erro ao compilar: {result.stderr}")
                
        except Exception as e:
            print(f"❌ Erro: {e}")
    
    # Compila uma vez no início
    print("🔨 Compilando SASS inicial...")
    compile_sass()
    
    # Inicia o monitoramento
    print(f"👀 Monitorando mudanças em: {scss_dir}")
    if css_dir.exists():
        print(f"👀 Monitorando mudanças em: {css_dir}")
    print("Pressione Ctrl+C para parar")
    
    event_handler = SassHandler(compile_sass)
    observer = Observer()
    observer.schedule(event_handler, str(scss_dir), recursive=True)
    
    # Monitora também o diretório CSS se existir
    if css_dir.exists():
        observer.schedule(event_handler, str(css_dir), recursive=True)
    
    observer.start()
    
    try:
        while True:
            time.sleep(1)
    except KeyboardInterrupt:
        observer.stop()
        print("\n🛑 Monitoramento parado")
    
    observer.join()
    return True

if __name__ == '__main__':
    success = watch_sass()
    sys.exit(0 if success else 1) 