# Sistema de Controle de Visão Hierárquico

## Visão Geral

O sistema de controle de visão hierárquico permite gerenciar permissões de acesso aos elementos da aplicação de forma granular e hierárquica. O sistema segue a estrutura:

```
Workspace
├── Portfolio
│   ├── Project
│   │   └── Board
│   └── Project
└── Portfolio
```

## Características Principais

### 1. Herança Automática
- Quando um usuário tem acesso ao **Workspace**, automaticamente tem acesso a todos os elementos abaixo (Portfolios, Projetos, Quadros)
- Permite criar exceções para restringir acesso específico a níveis inferiores

### 2. Controle por Colaborador e Área
- **Colaborador**: Permissão específica para um colaborador individual
- **Área**: Permissão aplicada a todos os colaboradores de uma área/departamento

### 3. Tipos de Acesso
- **Permitir**: Concede acesso ao elemento
- **Negar**: Remove acesso ao elemento (tem precedência sobre permissões de permissão)

### 4. Hierarquia de Permissões
- Permissões mais específicas têm precedência sobre permissões genéricas
- Permissões de negação têm precedência sobre permissões de permissão
- Herança automática do nível superior se não houver permissões específicas

## Modelos

### VisibilityPermission
Modelo principal para controlar permissões de visão.

```python
class VisibilityPermission(models.Model):
    PERMISSION_TYPES = [
        ('workspace', 'Workspace'),
        ('portfolio', 'Portfolio'),
        ('project', 'Projeto'),
        ('board', 'Quadro'),
    ]
    
    ACCESS_TYPES = [
        ('allow', 'Permitir'),
        ('deny', 'Negar'),
    ]
    
    SCOPE_TYPES = [
        ('collaborator', 'Colaborador'),
        ('area', 'Área'),
    ]
    
    permission_type = models.CharField(max_length=20, choices=PERMISSION_TYPES)
    access_type = models.CharField(max_length=10, choices=ACCESS_TYPES, default='allow')
    scope_type = models.CharField(max_length=20, choices=SCOPE_TYPES)
    
    # Escopo da permissão
    collaborator = models.ForeignKey(Collaborator, on_delete=models.CASCADE, blank=True, null=True)
    area = models.ForeignKey(Area, on_delete=models.CASCADE, blank=True, null=True)
    
    # Referência genérica ao objeto
    content_type = models.ForeignKey(ContentType, on_delete=models.CASCADE)
    object_id = models.PositiveIntegerField()
    content_object = GenericForeignKey('content_type', 'object_id')
    
    # Controle de herança
    inherit_from_parent = models.BooleanField(default=True)
    override_parent = models.BooleanField(default=False)
    
    ativo = models.BooleanField(default=True)
```

### WorkspaceAccess
Modelo simplificado para controle de acesso a workspaces.

```python
class WorkspaceAccess(models.Model):
    workspace = models.ForeignKey(Workspace, on_delete=models.CASCADE)
    collaborator = models.ForeignKey(Collaborator, on_delete=models.CASCADE, blank=True, null=True)
    area = models.ForeignKey(Area, on_delete=models.CASCADE, blank=True, null=True)
    can_manage = models.BooleanField(default=False)
    ativo = models.BooleanField(default=True)
```

## Mixins para Views

### WorkspacePermissionMixin
Filtra automaticamente workspaces baseado nas permissões do usuário.

```python
class WorkspaceListView(WorkspacePermissionMixin, ListView):
    model = Workspace
    template_name = 'workspace/list.html'
```

### PortfolioPermissionMixin
Filtra automaticamente portfolios baseado nas permissões do usuário.

```python
class PortfolioListView(PortfolioPermissionMixin, ListView):
    model = Portfolio
    template_name = 'portfolio/list.html'
```

### ProjectPermissionMixin
Filtra automaticamente projetos baseado nas permissões do usuário.

```python
class ProjectListView(ProjectPermissionMixin, ListView):
    model = Project
    template_name = 'project/list.html'
```

### BoardPermissionMixin
Filtra automaticamente boards baseado nas permissões do usuário.

```python
class BoardListView(BoardPermissionMixin, ListView):
    model = Board
    template_name = 'board/list.html'
```

### ManagePermissionMixin
Verifica se o usuário pode gerenciar um objeto específico.

```python
class WorkspaceUpdateView(ManagePermissionMixin, UpdateView):
    model = Workspace
    template_name = 'workspace/form.html'
```

### PermissionContextMixin
Adiciona informações de permissões ao contexto da view.

```python
class WorkspaceDetailView(PermissionContextMixin, DetailView):
    model = Workspace
    template_name = 'workspace/detail.html'
```

## Métodos Principais

### VisibilityPermission.has_access(user, obj, permission_type=None)
Verifica se um usuário tem acesso a um objeto específico.

```python
# Verifica acesso a um workspace
workspace = Workspace.objects.get(id=1)
if VisibilityPermission.has_access(request.user, workspace):
    # Usuário tem acesso
    pass
```

### VisibilityPermission.get_accessible_objects(user, model_class, permission_type=None)
Retorna todos os objetos de um tipo específico que o usuário pode acessar.

```python
# Obtém todos os workspaces acessíveis
accessible_workspaces = VisibilityPermission.get_accessible_objects(
    request.user, Workspace
)
```

## Uso em Views Function-Based

```python
@login_required
def workspace_list(request):
    # Obtém apenas workspaces acessíveis
    workspaces = VisibilityPermission.get_accessible_objects(
        request.user, Workspace
    )
    
    # Aplica filtros adicionais
    search = request.GET.get('search', '')
    if search:
        workspaces = workspaces.filter(nome__icontains=search)
    
    context = {'workspaces': workspaces}
    return render(request, 'workspace/list.html', context)

@login_required
def workspace_detail(request, pk):
    workspace = get_object_or_404(Workspace, pk=pk)
    
    # Verifica permissão
    if not VisibilityPermission.has_access(request.user, workspace):
        raise PermissionDenied("Acesso negado")
    
    context = {'workspace': workspace}
    return render(request, 'workspace/detail.html', context)
```

## Uso em Templates

### Verificar permissões de gerenciamento
```html
{% if can_manage %}
    <a href="{% url 'workspace:edit' workspace.id %}" class="btn btn-primary">
        <i class="fas fa-edit"></i> Editar
    </a>
    <a href="{% url 'workspace:delete' workspace.id %}" class="btn btn-danger">
        <i class="fas fa-trash"></i> Excluir
    </a>
    <a href="{% url 'permissions:workspace_permissions' workspace.id %}" class="btn btn-info">
        <i class="fas fa-shield-alt"></i> Gerenciar Permissões
    </a>
{% endif %}
```

### Mostrar informações de permissões
```html
{% if user_permissions.workspaces %}
    <div class="alert alert-info">
        <i class="fas fa-info-circle"></i>
        Você tem permissões específicas para este workspace.
    </div>
{% endif %}
```

## URLs do Sistema

### Dashboard de Permissões
```
/permissions/dashboard/
```

### Gerenciar Permissões de Workspace
```
/permissions/workspace/<workspace_id>/
```

### Gerenciamento em Lote
```
/permissions/bulk/
```

### Relatórios
```
/permissions/report/
```

## Admin Django

O sistema inclui interfaces administrativas para:

- **VisibilityPermission**: Gerenciar permissões de visão
- **WorkspaceAccess**: Gerenciar acesso simplificado a workspaces

### Campos do Admin
- Tipo de permissão (workspace, portfolio, project, board)
- Tipo de acesso (permitir, negar)
- Tipo de escopo (colaborador, área)
- Colaborador ou área específica
- Objeto de destino
- Controle de herança

## Exemplos de Uso

### 1. Adicionar Colaborador ao Workspace
```python
from permissions.models import VisibilityPermission
from django.contrib.contenttypes.models import ContentType

# Adicionar colaborador ao workspace
permission = VisibilityPermission.objects.create(
    permission_type='workspace',
    scope_type='collaborator',
    collaborator=collaborator,
    content_type=ContentType.objects.get_for_model(workspace),
    object_id=workspace.id,
    access_type='allow',
    ativo=True
)
```

### 2. Adicionar Área Inteira ao Workspace
```python
# Adicionar área inteira ao workspace
permission = VisibilityPermission.objects.create(
    permission_type='workspace',
    scope_type='area',
    area=area,
    content_type=ContentType.objects.get_for_model(workspace),
    object_id=workspace.id,
    access_type='allow',
    ativo=True
)
```

### 3. Negar Acesso Específico
```python
# Negar acesso de um colaborador a um projeto específico
permission = VisibilityPermission.objects.create(
    permission_type='project',
    scope_type='collaborator',
    collaborator=collaborator,
    content_type=ContentType.objects.get_for_model(project),
    object_id=project.id,
    access_type='deny',
    ativo=True
)
```

### 4. Verificar Acesso em View
```python
def project_detail(request, pk):
    project = get_object_or_404(Project, pk=pk)
    
    # Verifica se o usuário tem acesso
    if not VisibilityPermission.has_access(request.user, project):
        messages.error(request, "Você não tem permissão para acessar este projeto.")
        return redirect('project:list')
    
    context = {'project': project}
    return render(request, 'project/detail.html', context)
```

## Fluxo de Verificação de Permissões

1. **Verificação de Permissões Específicas**
   - Busca permissões específicas do colaborador para o objeto
   - Busca permissões específicas da área do colaborador para o objeto

2. **Aplicação de Regras**
   - Se há permissões de negação → Acesso negado
   - Se há permissões de permissão → Acesso concedido
   - Se não há permissões específicas → Verifica herança hierárquica

3. **Herança Hierárquica**
   - Board → Project → Portfolio → Workspace
   - Se o usuário tem acesso ao workspace, tem acesso a todos os elementos abaixo
   - Permissões específicas podem sobrescrever a herança

## Casos de Uso Comuns

### 1. Gestor de Projeto
- Acesso total ao workspace e todos os projetos
- Pode gerenciar permissões de outros colaboradores

### 2. Desenvolvedor
- Acesso apenas aos projetos específicos
- Não pode ver outros projetos do workspace

### 3. Cliente
- Acesso apenas aos projetos do seu workspace
- Visualização limitada (sem acesso a detalhes internos)

### 4. Área de Design
- Acesso a todos os projetos da área de design
- Pode ser restrito a projetos específicos

## Boas Práticas

### 1. Configuração Inicial
- Configure permissões por área primeiro (mais fácil de gerenciar)
- Use permissões por colaborador apenas para casos específicos

### 2. Manutenção
- Revise permissões regularmente
- Remova permissões de colaboradores inativos
- Mantenha um registro de mudanças

### 3. Testes
- Teste regularmente o acesso dos usuários
- Verifique se as permissões estão funcionando corretamente
- Teste cenários de herança hierárquica

### 4. Performance
- Use `select_related()` e `prefetch_related()` em consultas
- Considere cache para permissões frequentemente verificadas
- Monitore o desempenho das consultas de permissão

## Troubleshooting

### Problema: Usuário não consegue acessar workspace
**Solução:**
1. Verifique se o colaborador existe e está ativo
2. Verifique se há permissões específicas para o workspace
3. Verifique se há permissões por área
4. Verifique se não há permissões de negação

### Problema: Herança hierárquica não está funcionando
**Solução:**
1. Verifique se o usuário tem acesso ao workspace pai
2. Verifique se não há permissões específicas negando acesso
3. Verifique se `inherit_from_parent` está habilitado

### Problema: Performance lenta
**Solução:**
1. Otimize consultas com `select_related()`
2. Considere usar cache para permissões
3. Verifique se há muitas permissões desnecessárias

## Migração de Sistema Existente

Para migrar um sistema existente:

1. **Backup dos dados**
2. **Criar permissões padrão**
3. **Testar em ambiente de desenvolvimento**
4. **Migrar gradualmente**

```python
# Exemplo de migração
from permissions.models import VisibilityPermission
from django.contrib.contenttypes.models import ContentType

def migrate_existing_permissions():
    # Migra permissões existentes para o novo sistema
    for workspace in Workspace.objects.all():
        # Cria permissão padrão para o criador do workspace
        if workspace.created_by:
            VisibilityPermission.objects.create(
                permission_type='workspace',
                scope_type='collaborator',
                collaborator=workspace.created_by.collaborator,
                content_type=ContentType.objects.get_for_model(workspace),
                object_id=workspace.id,
                access_type='allow',
                ativo=True
            )
```

## Conclusão

O sistema de controle de visão hierárquico oferece:

- **Flexibilidade**: Controle granular por colaborador ou área
- **Simplicidade**: Herança automática reduz configuração
- **Segurança**: Controle de acesso baseado em permissões
- **Escalabilidade**: Suporte a grandes volumes de dados
- **Manutenibilidade**: Interface administrativa intuitiva

Este sistema permite que a agência gerencie eficientemente o acesso aos projetos, garantindo que cada colaborador veja apenas o que é relevante para seu trabalho, mantendo a segurança e organização dos dados. 