# 📚 Documentação Completa - Sistema de Gestão de Agência

## 📋 Índice

1. [Visão Geral](#visão-geral)
2. [Instalação e Configuração](#instalação-e-configuração)
3. [Manual do Usuário](#manual-do-usuário)
4. [Guia de Administração](#guia-de-administração)
5. [API REST](#api-rest)
6. [Estrutura Técnica](#estrutura-técnica)
7. [Solução de Problemas](#solução-de-problemas)
8. [FAQ](#faq)

---

## 🎯 Visão Geral

O **Sistema de Gestão de Agência** é uma plataforma completa para gestão de projetos, equipes e clientes, inspirada no Monday.com. O sistema oferece uma hierarquia organizacional clara e funcionalidades avançadas para controle de projetos.

### 🏗️ Hierarquia do Sistema

```
Workspace (Cliente)
├── Portfolio (Agrupamento de Projetos)
│   └── Project (Projeto)
│       └── Board (Quadro Kanban)
│           └── Group (Grupo de Tarefas)
│               └── Task (Tarefa)
```

### 🎨 Características Principais

- **Interface Moderna**: Bootstrap 5 com design responsivo
- **Gestão Hierárquica**: Workspace → Portfolio → Project → Board → Group → Task
- **Controle de Tempo**: Cronômetro integrado para tarefas
- **Notificações**: Sistema de alertas em tempo real
- **API REST**: Endpoints completos para integração
- **Relatórios**: Dashboards e estatísticas detalhadas
- **Permissões Granulares**: Controle de acesso por grupos

---

## 🚀 Instalação e Configuração

### Pré-requisitos

- Python 3.11+
- pip
- Redis (opcional, para Celery)

### Instalação Rápida

```bash
# 1. Clone o repositório
git clone <repository-url>
cd gestao_agencia

# 2. Crie ambiente virtual
python3 -m venv venv
source venv/bin/activate

# 3. Instale dependências
pip install -r requirements.txt

# 4. Execute configuração
python setup.py

# 5. Inicie o sistema
./start.sh
```

### Configuração Manual

```bash
# Migrações
python manage.py makemigrations
python manage.py migrate

# Superusuário
python manage.py createsuperuser

# Grupos e permissões
python manage.py setup_groups

# Dados de exemplo
python manage.py load_sample_data

# Arquivos estáticos
python manage.py collectstatic
```

---

## 👥 Manual do Usuário

### 🔐 Acesso ao Sistema

- **URL Principal**: http://localhost:8000
- **Admin**: http://localhost:8000/admin

### 👤 Tipos de Usuário

#### 1. **Administrador**
- **Acesso**: Completo ao sistema
- **Funções**: Gestão de usuários, configurações, relatórios
- **Usuário**: `admin` / `admin123`

#### 2. **Gerente**
- **Acesso**: Gestão de projetos e equipes
- **Funções**: Criação de projetos, atribuição de tarefas, relatórios
- **Usuário**: `gerente` / `123456`

#### 3. **Colaborador**
- **Acesso**: Visualização de projetos, edição de tarefas
- **Funções**: Execução de tarefas, controle de tempo
- **Usuários**: `ana.dev`, `carlos.design`, etc. / `123456`

#### 4. **Cliente**
- **Acesso**: Visualização de seus projetos
- **Funções**: Acompanhamento de progresso

### 📊 Dashboard Principal

O dashboard exibe:
- **Estatísticas Gerais**: Projetos ativos, tarefas pendentes
- **Projetos Recentes**: Últimos projetos criados
- **Tarefas Pendentes**: Tarefas atribuídas ao usuário
- **Ações Rápidas**: Links para funcionalidades principais

### 🏢 Gestão de Clientes

#### Criar Cliente
1. Acesse **Clientes** → **Novo Cliente**
2. Preencha:
   - **Nome**: Nome da empresa
   - **CNPJ**: CNPJ da empresa
   - **Email**: Email de contato
   - **Telefone**: Telefone principal
   - **Endereço**: Endereço completo
3. Clique em **Salvar**

#### Editar Cliente
1. Acesse a lista de clientes
2. Clique no cliente desejado
3. Clique em **Editar**
4. Faça as alterações necessárias
5. Clique em **Salvar**

### 👥 Gestão de Colaboradores

#### Criar Colaborador
1. Acesse **Colaboradores** → **Novo Colaborador**
2. Preencha:
   - **Usuário**: Usuário Django associado
   - **Nome**: Nome completo
   - **Email**: Email profissional
   - **Área**: Área de trabalho
   - **Cargo**: Cargo na empresa
   - **Telefone**: Telefone de contato
3. Clique em **Salvar**

#### Áreas de Trabalho
- **Desenvolvimento**: Programadores e desenvolvedores
- **Design**: Designers e UX/UI
- **Marketing**: Analistas de marketing
- **Vendas**: Equipe comercial
- **Suporte**: Suporte técnico
- **Gestão de Projetos**: Gerentes e coordenadores
- **Recursos Humanos**: RH e recrutamento
- **Financeiro**: Contabilidade e finanças
- **Qualidade**: QA e controle de qualidade
- **Infraestrutura**: TI e infraestrutura

### 🏠 Gestão de Workspaces

#### Criar Workspace
1. Acesse **Workspaces** → **Novo Workspace**
2. Preencha:
   - **Nome**: Nome do workspace
   - **Cliente**: Cliente associado
   - **Descrição**: Descrição do workspace
   - **Cor**: Cor de identificação
3. Clique em **Salvar**

### 📁 Gestão de Portfolios

#### Criar Portfolio
1. Acesse **Portfolios** → **Novo Portfolio**
2. Preencha:
   - **Nome**: Nome do portfolio
   - **Workspace**: Workspace associado
   - **Cliente**: Cliente do portfolio
   - **Descrição**: Descrição do portfolio
3. Clique em **Salvar**

### 📋 Gestão de Projetos

#### Criar Projeto
1. Acesse **Projetos** → **Novo Projeto**
2. Preencha:
   - **Nome**: Nome do projeto
   - **Portfolio**: Portfolio associado
   - **Workspace**: Workspace do projeto
   - **Cliente**: Cliente do projeto
   - **Cronograma**: Data de início e fim
   - **Status**: Status atual do projeto
   - **Prioridade**: Prioridade do projeto
   - **Orçamento**: Orçamento previsto
3. Clique em **Salvar**

#### Status de Projetos
- **Planejamento**: Projeto em fase de planejamento
- **Em Andamento**: Projeto em execução
- **Concluído**: Projeto finalizado
- **Cancelado**: Projeto cancelado
- **Pausado**: Projeto temporariamente pausado

### 📊 Gestão de Boards

#### Criar Board
1. Acesse **Boards** → **Novo Board**
2. Preencha:
   - **Nome**: Nome do board
   - **Projeto**: Projeto associado
   - **Workspace**: Workspace do board
   - **Cliente**: Cliente do board
   - **Descrição**: Descrição do board
   - **Ordem**: Ordem de exibição
3. Clique em **Salvar**

#### Tipos de Board Comuns
- **Planejamento**: Atividades de planejamento
- **Desenvolvimento**: Atividades de desenvolvimento
- **Testes**: Atividades de testes
- **Revisão**: Atividades de revisão
- **Deploy**: Atividades de deploy

### 📝 Gestão de Tarefas

#### Criar Tarefa
1. Acesse **Tarefas** → **Nova Tarefa**
2. Preencha:
   - **Nome**: Nome da tarefa
   - **Grupo**: Grupo da tarefa
   - **Descrição**: Descrição detalhada
   - **Responsável**: Colaborador responsável
   - **Cronograma**: Data de início e fim
   - **Status**: Status atual
   - **Prioridade**: Prioridade da tarefa
   - **Esforço**: Esforço estimado em horas
3. Clique em **Salvar**

#### Status de Tarefas
- **Não Iniciada**: Tarefa ainda não iniciada
- **Em Andamento**: Tarefa em execução
- **Pausada**: Tarefa temporariamente pausada
- **Concluída**: Tarefa finalizada
- **Cancelada**: Tarefa cancelada

#### Prioridades
- **Baixa**: Prioridade baixa
- **Média**: Prioridade média
- **Alta**: Prioridade alta

#### Controle de Tempo
- **Iniciar Trabalho**: Inicia o cronômetro
- **Pausar Trabalho**: Pausa o cronômetro
- **Concluir Tarefa**: Finaliza a tarefa

### 🔔 Sistema de Notificações

#### Tipos de Notificação
- **Tarefa Atribuída**: Nova tarefa atribuída
- **Prazo Próximo**: Tarefa com prazo próximo
- **Tarefa Atrasada**: Tarefa fora do prazo
- **Tarefa Concluída**: Tarefa finalizada
- **Comentário**: Novo comentário em tarefa

#### Gerenciar Notificações
1. Acesse o ícone de notificações no topo
2. Visualize notificações não lidas
3. Marque como lida clicando na notificação
4. Acesse configurações para personalizar alertas

---

## 🔧 Guia de Administração

### 👤 Gestão de Usuários

#### Criar Superusuário
```bash
python manage.py createsuperuser
```

#### Grupos de Usuários

##### Gerentes
- **Permissões**: 40 permissões
- **Acesso**: Completo a projetos e equipes
- **Funções**: Gestão de projetos, relatórios gerenciais

##### Colaboradores
- **Permissões**: 12 permissões
- **Acesso**: Visualização de projetos, edição de tarefas
- **Funções**: Execução de tarefas, controle de tempo

##### Clientes
- **Permissões**: 6 permissões
- **Acesso**: Visualização de seus projetos
- **Funções**: Acompanhamento de progresso

### ⚙️ Configurações do Sistema

#### Configurações de Email
```python
# settings.py
EMAIL_BACKEND = 'django.core.mail.backends.smtp.EmailBackend'
EMAIL_HOST = 'smtp.gmail.com'
EMAIL_PORT = 587
EMAIL_USE_TLS = True
EMAIL_HOST_USER = 'seu-email@gmail.com'
EMAIL_HOST_PASSWORD = 'sua-senha'
```

#### Configurações de Banco de Dados
```python
# SQLite (padrão)
DATABASES = {
    'default': {
        'ENGINE': 'django.db.backends.sqlite3',
        'NAME': BASE_DIR / 'db.sqlite3',
    }
}

# PostgreSQL (produção)
DATABASES = {
    'default': {
        'ENGINE': 'django.db.backends.postgresql',
        'NAME': 'gestao_agencia',
        'USER': 'postgres',
        'PASSWORD': 'password',
        'HOST': 'localhost',
        'PORT': '5432',
    }
}
```

#### Configurações de Celery
```python
CELERY_BROKER_URL = 'redis://localhost:6379/0'
CELERY_RESULT_BACKEND = 'redis://localhost:6379/0'
```

### 📊 Relatórios e Estatísticas

#### Relatórios Disponíveis
- **Progresso de Projetos**: Status e progresso dos projetos
- **Produtividade por Colaborador**: Horas trabalhadas por colaborador
- **Análise de Prazos**: Tarefas atrasadas e no prazo
- **Custos por Projeto**: Análise de custos e orçamentos
- **Utilização de Recursos**: Uso de recursos por área

#### Como Gerar Relatórios
1. Acesse **Relatórios** no menu principal
2. Selecione o tipo de relatório
3. Configure filtros (período, projeto, colaborador)
4. Clique em **Gerar Relatório**
5. Exporte em PDF ou Excel

### 🔒 Backup e Restauração

#### Backup do Banco de Dados
```bash
# SQLite
cp db.sqlite3 backup_$(date +%Y%m%d_%H%M%S).sqlite3

# PostgreSQL
pg_dump gestao_agencia > backup_$(date +%Y%m%d_%H%M%S).sql
```

#### Backup de Arquivos
```bash
# Arquivos de mídia
tar -czf media_backup_$(date +%Y%m%d_%H%M%S).tar.gz media/

# Arquivos estáticos
tar -czf static_backup_$(date +%Y%m%d_%H%M%S).tar.gz staticfiles/
```

---

## 🔌 API REST

### 🔑 Autenticação

#### Login
```bash
POST /api/auth/login/
{
    "username": "admin",
    "password": "admin123"
}
```

#### Usar Token
```bash
Authorization: Token <token>
```

### 📡 Endpoints Principais

#### Clientes
```bash
# Listar clientes
GET /api/clients/

# Criar cliente
POST /api/clients/
{
    "nome": "Empresa XYZ",
    "cnpj": "12.345.678/0001-90",
    "email": "contato@empresa.com",
    "telefone": "(11) 99999-9999"
}

# Atualizar cliente
PUT /api/clients/1/
{
    "nome": "Empresa XYZ Atualizada"
}

# Deletar cliente
DELETE /api/clients/1/
```

#### Colaboradores
```bash
# Listar colaboradores
GET /api/collaborators/

# Criar colaborador
POST /api/collaborators/
{
    "user": 1,
    "nome": "João Silva",
    "email": "joao@agencia.com",
    "area": 1,
    "cargo": "Desenvolvedor"
}
```

#### Projetos
```bash
# Listar projetos
GET /api/projects/

# Criar projeto
POST /api/projects/
{
    "nome": "Novo Projeto",
    "portfolio": 1,
    "workspace": 1,
    "cliente": 1,
    "cronograma_inicio": "2024-01-15",
    "cronograma_fim": "2024-03-15",
    "status": "em_andamento",
    "prioridade": "alta"
}
```

#### Tarefas
```bash
# Listar tarefas
GET /api/tasks/

# Criar tarefa
POST /api/tasks/
{
    "nome": "Nova Tarefa",
    "group": 1,
    "responsavel": 1,
    "cronograma_inicio": "2024-01-15",
    "cronograma_fim": "2024-01-20",
    "status": "nao_iniciada",
    "prioridade": "media"
}

# Atualizar status
PATCH /api/tasks/1/update_status/
{
    "status": "em_andamento"
}
```

#### Notificações
```bash
# Listar notificações
GET /api/notifications/

# Marcar como lida
PATCH /api/notifications/1/mark_read/
```

### 📊 Filtros e Paginação

#### Filtros
```bash
# Filtrar por status
GET /api/tasks/?status=em_andamento

# Filtrar por responsável
GET /api/tasks/?responsavel=1

# Filtrar por projeto
GET /api/tasks/?group__board__project=1

# Buscar por nome
GET /api/tasks/?search=nova tarefa
```

#### Paginação
```bash
# Primeira página
GET /api/tasks/?page=1

# Segunda página
GET /api/tasks/?page=2

# Tamanho da página
GET /api/tasks/?page_size=50
```

---

## 🏗️ Estrutura Técnica

### 📁 Arquitetura do Projeto

```
gestao_agencia/
├── gestao_agencia/          # Configurações principais
│   ├── settings.py          # Configurações Django
│   ├── urls.py             # URLs principais
│   ├── permissions.py      # Permissões customizadas
│   ├── middleware.py       # Middleware customizado
│   └── celery.py          # Configuração Celery
├── client/                 # App de clientes
├── collaborator/          # App de colaboradores
├── workspace/             # App de workspaces
├── portfolio/             # App de portfolios
├── project/               # App de projetos
├── board/                 # App de boards
├── task/                  # App de tarefas
├── notification/          # App de notificações
├── templates/             # Templates HTML
├── static/                # Arquivos estáticos
└── requirements.txt       # Dependências Python
```

### 🗄️ Modelos de Dados

#### Cliente
```python
class Client(models.Model):
    nome = models.CharField(max_length=200)
    cnpj = models.CharField(max_length=18, unique=True)
    email = models.EmailField()
    telefone = models.CharField(max_length=20)
    endereco = models.TextField()
    ativo = models.BooleanField(default=True)
```

#### Colaborador
```python
class Collaborator(models.Model):
    user = models.OneToOneField(User, on_delete=models.CASCADE)
    nome = models.CharField(max_length=200)
    email = models.EmailField(unique=True)
    area = models.ForeignKey(Area, on_delete=models.PROTECT)
    cargo = models.CharField(max_length=100)
    ativo = models.BooleanField(default=True)
```

#### Projeto
```python
class Project(models.Model):
    nome = models.CharField(max_length=200)
    portfolio = models.ForeignKey(Portfolio, on_delete=models.CASCADE)
    workspace = models.ForeignKey(Workspace, on_delete=models.CASCADE)
    cliente = models.ForeignKey(Client, on_delete=models.CASCADE)
    cronograma_inicio = models.DateField()
    cronograma_fim = models.DateField()
    status = models.CharField(max_length=20, choices=STATUS_CHOICES)
    prioridade = models.CharField(max_length=10, choices=PRIORIDADE_CHOICES)
    orcamento = models.DecimalField(max_digits=12, decimal_places=2)
```

#### Tarefa
```python
class Task(models.Model):
    nome = models.CharField(max_length=200)
    group = models.ForeignKey(Group, on_delete=models.CASCADE)
    responsavel = models.ForeignKey(Collaborator, on_delete=models.SET_NULL)
    cronograma_inicio = models.DateField()
    cronograma_fim = models.DateField()
    status = models.CharField(max_length=20, choices=STATUS_CHOICES)
    prioridade = models.CharField(max_length=10, choices=PRIORIDADE_CHOICES)
    esforco_estimado = models.DecimalField(max_digits=6, decimal_places=2)
    esforco_utilizado = models.DecimalField(max_digits=6, decimal_places=2)
```

### 🔄 Tarefas Assíncronas (Celery)

#### Tarefas Disponíveis
```python
# Processar notificações pendentes
@shared_task
def processar_notificacoes_pendentes():
    # Processa notificações não enviadas

# Verificar tarefas atrasadas
@shared_task
def verificar_tarefas_atrasadas():
    # Identifica tarefas fora do prazo

# Verificar prazos próximos
@shared_task
def verificar_prazos_proximos():
    # Alerta sobre prazos próximos
```

#### Configuração do Celery
```bash
# Worker
celery -A gestao_agencia worker --loglevel=info

# Beat (tarefas periódicas)
celery -A gestao_agencia beat --loglevel=info
```

### 🎨 Frontend

#### Tecnologias Utilizadas
- **Bootstrap 5**: Framework CSS
- **jQuery**: Manipulação DOM
- **HTML5/CSS3**: Estrutura e estilo
- **JavaScript**: Interatividade

#### Componentes Principais
- **Dashboard**: Estatísticas e visão geral
- **Kanban Board**: Organização visual de tarefas
- **Cronômetro**: Controle de tempo integrado
- **Notificações**: Sistema de alertas
- **Formulários**: Interface de criação/edição

---

## 🐛 Solução de Problemas

### ❌ Erros Comuns

#### 1. Erro de Migração
```bash
# Solução
python manage.py migrate --fake-initial
```

#### 2. Arquivos Estáticos Não Carregam
```bash
# Solução
python manage.py collectstatic --clear
```

#### 3. Celery Não Funciona
```bash
# Verificar Redis
redis-cli ping

# Reiniciar Celery
pkill -f celery
celery -A gestao_agencia worker --loglevel=info
```

#### 4. Erro de Permissão
```bash
# Verificar permissões
chmod +x start.sh
chmod -R 755 staticfiles/
```

#### 5. Banco de Dados Corrompido
```bash
# Backup e restauração
cp db.sqlite3 backup.sqlite3
python manage.py migrate --fake-initial
```

### 📝 Logs do Sistema

#### Logs do Django
```bash
# Visualizar logs
tail -f django.log

# Logs de erro
grep "ERROR" django.log
```

#### Logs do Celery
```bash
# Logs do worker
celery -A gestao_agencia worker --loglevel=debug

# Logs do beat
celery -A gestao_agencia beat --loglevel=debug
```

### 🔧 Manutenção

#### Limpeza de Dados
```bash
# Limpar notificações antigas
python manage.py shell
>>> from notification.models import Notification
>>> Notification.objects.filter(created_at__lt=timezone.now()-timedelta(days=30)).delete()
```

#### Otimização de Performance
```bash
# Rebuild do banco
python manage.py dbshell
VACUUM;
```

#### Backup Automático
```bash
# Script de backup
#!/bin/bash
DATE=$(date +%Y%m%d_%H%M%S)
cp db.sqlite3 backup_$DATE.sqlite3
tar -czf media_backup_$DATE.tar.gz media/
```

---

## ❓ FAQ

### 🤔 Perguntas Frequentes

#### Q: Como adicionar um novo tipo de área?
**A**: Acesse o admin Django → Colaboradores → Áreas → Adicionar área

#### Q: Como configurar notificações por email?
**A**: Configure SMTP em `settings.py` e ative o Celery

#### Q: Como exportar dados do sistema?
**A**: Use a API REST ou acesse o admin Django para exportar

#### Q: Como fazer backup automático?
**A**: Configure um cron job para executar o script de backup

#### Q: Como personalizar o tema visual?
**A**: Edite os arquivos CSS em `static/css/custom.css`

#### Q: Como adicionar novos campos aos modelos?
**A**: Crie uma migração: `python manage.py makemigrations`

#### Q: Como configurar SSL/HTTPS?
**A**: Configure um proxy reverso (nginx) com certificado SSL

#### Q: Como monitorar o sistema?
**A**: Use ferramentas como Sentry para monitoramento de erros

### 📞 Suporte

#### Contatos
- **Email**: suporte@gestaoagencia.com
- **Issues**: GitHub Issues
- **Documentação**: Wiki do projeto

#### Recursos Adicionais
- **Tutorial em Vídeo**: Canal do YouTube
- **Comunidade**: Fórum de discussão
- **Treinamento**: Cursos online

---

## 📄 Licença

Este projeto está sob a licença MIT. Veja o arquivo LICENSE para detalhes.

---

**🎉 Obrigado por usar o Sistema de Gestão de Agência!**

*Esta documentação é atualizada regularmente. Para a versão mais recente, consulte o repositório do projeto.* 