# Documentação do Módulo de Logs do Sistema

## Visão Geral

O módulo `system_logs` foi criado para registrar todas as ações do sistema, fornecendo um histórico completo de atividades para auditoria e monitoramento. O acesso é restrito apenas a superusuários.

## Características Principais

- **Registro Automático**: Captura automaticamente informações da requisição (IP, User Agent, URL, etc.)
- **Acesso Restrito**: Apenas superusuários podem visualizar os logs
- **Filtros Avançados**: Busca por data, usuário, ação, severidade, etc.
- **Dashboard Interativo**: Visualização de estatísticas em tempo real
- **Exportação**: Suporte para exportação em CSV e JSON
- **Limpeza Automática**: Comandos para limpar logs antigos

## Modelo SystemLog

### Campos Principais

- **timestamp**: Data e hora do log
- **user**: Usuário que executou a ação
- **action**: Tipo de ação (CREATE, UPDATE, DELETE, LOGIN, etc.)
- **severity**: Severidade (LOW, MEDIUM, HIGH, CRITICAL)
- **description**: Descrição da ação
- **details**: Detalhes adicionais em JSON
- **content_type/object_id**: Objeto relacionado à ação
- **ip_address**: Endereço IP do usuário
- **user_agent**: User Agent do navegador
- **url**: URL acessada
- **method**: Método HTTP

### Tipos de Ação Disponíveis

- `CREATE`: Criação de objetos
- `UPDATE`: Atualização de objetos
- `DELETE`: Exclusão de objetos
- `LOGIN`: Login de usuário
- `LOGOUT`: Logout de usuário
- `VIEW`: Visualização de páginas/objetos
- `DOWNLOAD`: Download de arquivos
- `UPLOAD`: Upload de arquivos
- `EXPORT`: Exportação de dados
- `IMPORT`: Importação de dados
- `PERMISSION_CHANGE`: Alteração de permissões
- `SYSTEM_CONFIG`: Configuração do sistema
- `ERROR`: Erros do sistema
- `WARNING`: Avisos do sistema
- `INFO`: Informações gerais

### Níveis de Severidade

- `LOW`: Baixa importância
- `MEDIUM`: Importância média
- `HIGH`: Alta importância
- `CRITICAL`: Crítica

## Como Usar

### 1. Registro Manual de Logs

```python
from system_logs.utils import log_system_action

# Log básico
log_system_action(
    request=request,
    action='CREATE',
    description='Cliente criado com sucesso',
    severity='MEDIUM'
)

# Log com objeto relacionado
log_system_action(
    request=request,
    action='UPDATE',
    description='Projeto atualizado',
    severity='MEDIUM',
    obj=project_instance,
    details={'changes': {'status': 'old_status -> new_status'}}
)
```

### 2. Utilitários Específicos

```python
from system_logs.utils import (
    log_user_login, log_user_logout,
    log_object_created, log_object_updated, log_object_deleted,
    log_permission_change, log_error, log_warning, log_info
)

# Login/Logout
log_user_login(request, user, success=True)
log_user_logout(request, user)

# Operações CRUD
log_object_created(request, client_instance)
log_object_updated(request, project_instance, changes={'status': 'new_status'})
log_object_deleted(request, task_instance)

# Permissões
log_permission_change(request, user, 'can_edit_projects', granted=True)

# Logs de sistema
log_error(request, 'Erro ao processar pagamento')
log_warning(request, 'Usuário tentou acessar área restrita')
log_info(request, 'Backup realizado com sucesso')
```

### 3. Decorator para Views

```python
from system_logs.utils import log_view_action

@log_view_action('VIEW', 'LOW')
def my_view(request):
    # código da view
    pass
```

### 4. Middleware Automático

O middleware `SystemLogMiddleware` captura automaticamente:
- Erros HTTP (4xx, 5xx)
- Exceções não tratadas
- Informações da requisição (IP, User Agent, etc.)

## URLs Disponíveis

- `/system-logs/` - Dashboard principal
- `/system-logs/list/` - Lista de todos os logs
- `/system-logs/detail/<id>/` - Detalhes de um log específico

## Comandos de Gerenciamento

### Limpar Logs Antigos

```bash
# Limpar logs com mais de 90 dias (padrão)
python manage.py cleanup_logs

# Limpar logs com mais de 30 dias
python manage.py cleanup_logs --days 30

# Limpar apenas logs de baixa severidade
python manage.py cleanup_logs --severity LOW

# Simular limpeza (não remove nada)
python manage.py cleanup_logs --dry-run

# Forçar limpeza sem confirmação
python manage.py cleanup_logs --force
```

### Exportar Logs

```bash
# Exportar todos os logs em CSV
python manage.py export_logs

# Exportar logs dos últimos 7 dias em JSON
python manage.py export_logs --format json --days 7

# Exportar logs de um usuário específico
python manage.py export_logs --user admin

# Exportar logs críticos
python manage.py export_logs --severity CRITICAL

# Especificar arquivo de saída
python manage.py export_logs --output meus_logs.csv
```

## Admin Django

O módulo inclui uma interface administrativa completa com:
- Lista filtrada de logs
- Detalhes completos de cada log
- Estatísticas em tempo real
- Acesso restrito a superusuários
- Campos somente leitura para integridade

## Integração com Outros Módulos

### Exemplo: Logging em Views de Cliente

```python
from system_logs.utils import log_object_created, log_object_updated, log_object_deleted

class ClientCreateView(CreateView):
    model = Client
    # ... configurações da view
    
    def form_valid(self, form):
        response = super().form_valid(form)
        log_object_created(self.request, self.object)
        return response

class ClientUpdateView(UpdateView):
    model = Client
    # ... configurações da view
    
    def form_valid(self, form):
        changes = self.get_changes(form)
        response = super().form_valid(form)
        log_object_updated(self.request, self.object, changes=changes)
        return response
    
    def get_changes(self, form):
        # Lógica para detectar mudanças
        return {'field': 'old_value -> new_value'}
```

### Exemplo: Logging em Signals

```python
from django.db.models.signals import post_save, post_delete
from django.dispatch import receiver
from system_logs.utils import log_object_created, log_object_updated, log_object_deleted

@receiver(post_save, sender=Project)
def log_project_changes(sender, instance, created, **kwargs):
    if created:
        log_object_created(None, instance, description=f"Projeto '{instance.nome}' criado")
    else:
        log_object_updated(None, instance, description=f"Projeto '{instance.nome}' atualizado")

@receiver(post_delete, sender=Project)
def log_project_deletion(sender, instance, **kwargs):
    log_object_deleted(None, instance, description=f"Projeto '{instance.nome}' excluído")
```

## Configuração

### Settings.py

```python
INSTALLED_APPS = [
    # ... outros apps
    'system_logs',
]

MIDDLEWARE = [
    # ... outros middlewares
    'system_logs.middleware.SystemLogMiddleware',
]
```

### URLs

```python
urlpatterns = [
    # ... outras URLs
    path('system-logs/', include('system_logs.urls')),
]
```

## Monitoramento e Alertas

### Logs Críticos

O sistema identifica automaticamente logs críticos:
- Erros HTTP 5xx
- Exceções não tratadas
- Tentativas de acesso não autorizado
- Alterações de permissões

### Dashboard em Tempo Real

O dashboard mostra:
- Total de logs
- Logs de hoje
- Logs críticos
- Logs por severidade
- Logs por ação
- Logs recentes

## Segurança

- Acesso restrito apenas a superusuários
- Logs não podem ser editados (somente leitura)
- Informações sensíveis são mascaradas
- Auditoria completa de todas as ações

## Performance

- Índices otimizados no banco de dados
- Paginação de resultados
- Limpeza automática de logs antigos
- Cache de estatísticas

## Manutenção

### Backup de Logs

```bash
# Exportar logs antes da limpeza
python manage.py export_logs --output backup_logs.json

# Limpar logs antigos
python manage.py cleanup_logs --days 90
```

### Monitoramento de Espaço

```bash
# Verificar tamanho da tabela de logs
python manage.py dbshell
SELECT pg_size_pretty(pg_total_relation_size('system_logs_systemlog'));
```

## Troubleshooting

### Problemas Comuns

1. **Logs não aparecem**: Verificar se o middleware está ativo
2. **Erro de permissão**: Verificar se o usuário é superusuário
3. **Performance lenta**: Verificar índices e considerar limpeza de logs antigos

### Logs de Debug

```python
# Ativar logs de debug
import logging
logging.getLogger('system_logs').setLevel(logging.DEBUG)
```

## Contribuição

Para adicionar novos tipos de ação ou funcionalidades:

1. Adicionar nova ação em `ACTION_CHOICES` no modelo
2. Criar utilitário específico em `utils.py`
3. Atualizar documentação
4. Adicionar testes se necessário 