# 🚀 **GUIA COMPLETO DE DEPLOY - Sistema de Gestão de Agência**

## 📋 **ÍNDICE**

1. [O que é Deploy?](#o-que-é-deploy)
2. [Por que usar Docker?](#por-que-usar-docker)
3. [Tipos de Deploy](#tipos-de-deploy)
4. [Deploy com Docker](#deploy-com-docker)
5. [Deploy Automatizado (CI/CD)](#deploy-automatizado)
6. [Deploy Manual](#deploy-manual)
7. [Configuração de Produção](#configuração-de-produção)
8. [Monitoramento](#monitoramento)

---

## 🎯 **O QUE É DEPLOY?**

**Deploy** é o processo de colocar sua aplicação em um servidor de produção para que os usuários finais possam acessá-la pela internet.

### 🔄 **Fluxo de Deploy**
```
Desenvolvimento → Testes → Staging → Produção
     (local)        (QA)     (homolog)   (usuários)
```

### 🎯 **Objetivos do Deploy**
- ✅ Disponibilizar a aplicação 24/7
- ✅ Garantir performance e segurança
- ✅ Facilitar atualizações
- ✅ Manter backup e recuperação
- ✅ Monitorar e alertar problemas

---

## 🐳 **POR QUE USAR DOCKER?**

### ✅ **Vantagens**

1. **Consistência**: "Funciona na minha máquina" não é mais problema
2. **Isolamento**: Cada aplicação roda em seu próprio ambiente
3. **Portabilidade**: Roda em qualquer lugar (Linux, Windows, Cloud)
4. **Escalabilidade**: Fácil de replicar e escalar
5. **Versionamento**: Controle de versões da aplicação
6. **Facilidade de Deploy**: Um comando para subir tudo

### 🔧 **Como Funciona**

```dockerfile
# Dockerfile - "Receita" da aplicação
FROM python:3.11-slim
WORKDIR /app
COPY requirements.txt .
RUN pip install -r requirements.txt
COPY . .
EXPOSE 8000
CMD ["gunicorn", "--bind", "0.0.0.0:8000", "gestao_agencia.wsgi:application"]
```

### 🏗️ **Arquitetura Docker**

```
┌─────────────────┐
│   Nginx (80)    │ ← Proxy reverso
├─────────────────┤
│   Django App    │ ← Aplicação principal
├─────────────────┤
│   PostgreSQL    │ ← Banco de dados
├─────────────────┤
│   Redis         │ ← Cache/Celery
└─────────────────┘
```

---

## 🛠️ **TIPOS DE DEPLOY**

### 1. **Deploy Manual** (Tradicional)
```bash
# No servidor
git pull
pip install -r requirements.txt
python manage.py migrate
python manage.py collectstatic
sudo systemctl restart gunicorn
```

**Prós**: Simples, controle total
**Contras**: Propenso a erros, difícil de replicar

### 2. **Deploy com Docker**
```bash
# Construir e executar
docker-compose up -d
```

**Prós**: Consistente, isolado, fácil
**Contras**: Curva de aprendizado

### 3. **Deploy Automatizado (CI/CD)**
```yaml
# GitHub Actions
on:
  push:
    branches: [main]
jobs:
  deploy:
    runs-on: ubuntu-latest
    steps:
      - name: Deploy
        run: |
          ssh server "cd /app && git pull && docker-compose up -d"
```

**Prós**: Automático, confiável, rápido
**Contras**: Configuração inicial complexa

---

## 🐳 **DEPLOY COM DOCKER**

### 📦 **Arquivos Criados**

1. **Dockerfile** - Receita da aplicação
2. **docker-compose.yml** - Orquestração dos serviços
3. **nginx.conf** - Configuração do proxy reverso
4. **.dockerignore** - Arquivos ignorados no build

### 🚀 **Como Usar**

#### **Desenvolvimento Local**
```bash
# Construir e iniciar
docker-compose up -d

# Ver logs
docker-compose logs -f

# Parar
docker-compose down
```

#### **Produção**
```bash
# Deploy completo
./deploy.sh production

# Ou manualmente
docker-compose -f docker-compose.prod.yml up -d
```

### 🔧 **Comandos Úteis**

```bash
# Ver status dos serviços
docker-compose ps

# Executar comando no container
docker-compose exec web python manage.py migrate

# Ver logs de um serviço
docker-compose logs web

# Backup do banco
docker-compose exec db pg_dump -U postgres gestao_agencia > backup.sql

# Restaurar backup
docker-compose exec -T db psql -U postgres gestao_agencia < backup.sql
```

---

## 🤖 **DEPLOY AUTOMATIZADO (CI/CD)**

### 🔄 **Fluxo Automatizado**

```
Push no Git → Testes → Build → Deploy → Monitoramento
```

### 📋 **GitHub Actions**

O arquivo `.github/workflows/deploy.yml` configura:

1. **Trigger**: Push na branch main
2. **Testes**: Executa todos os testes
3. **Deploy**: Conecta no servidor e atualiza

### 🔐 **Secrets Necessários**

No GitHub → Settings → Secrets:

```bash
HOST=seu-servidor.com
USERNAME=usuario-ssh
SSH_KEY=chave-privada-ssh
```

### 🚀 **Como Configurar**

1. **Preparar servidor**:
```bash
# Instalar Docker
curl -fsSL https://get.docker.com | sh
sudo usermod -aG docker $USER

# Instalar Docker Compose
sudo curl -L "https://github.com/docker/compose/releases/download/v2.20.0/docker-compose-$(uname -s)-$(uname -m)" -o /usr/local/bin/docker-compose
sudo chmod +x /usr/local/bin/docker-compose
```

2. **Configurar SSH**:
```bash
# Gerar chave SSH
ssh-keygen -t rsa -b 4096 -C "deploy@gestaoagencia.com"

# Adicionar ao servidor
ssh-copy-id usuario@seu-servidor.com
```

3. **Configurar GitHub**:
   - Vá em Settings → Secrets
   - Adicione HOST, USERNAME, SSH_KEY

---

## 🛠️ **DEPLOY MANUAL**

### 📝 **Script de Deploy**

O arquivo `deploy.sh` automatiza:

```bash
#!/bin/bash
# 1. Backup do banco
# 2. Parar serviços
# 3. Atualizar código
# 4. Construir imagens
# 5. Iniciar serviços
# 6. Executar migrações
# 7. Verificar saúde
# 8. Rollback se necessário
```

### 🚀 **Como Usar**

```bash
# Deploy em produção
./deploy.sh production

# Deploy em staging
./deploy.sh staging

# Ver logs
docker-compose logs -f web
```

---

## ⚙️ **CONFIGURAÇÃO DE PRODUÇÃO**

### 🔐 **Variáveis de Ambiente**

Criar arquivo `.env`:

```bash
# Django
SECRET_KEY=sua-chave-secreta-muito-segura
DEBUG=False
ALLOWED_HOSTS=seu-dominio.com,www.seu-dominio.com

# Banco de dados
DATABASE_URL=postgresql://user:pass@localhost:5432/gestao_agencia

# Redis
REDIS_URL=redis://localhost:6379/0

# Email
EMAIL_HOST=smtp.gmail.com
EMAIL_PORT=587
EMAIL_HOST_USER=seu-email@gmail.com
EMAIL_HOST_PASSWORD=sua-senha-app
```

### 🗄️ **Banco de Dados**

```bash
# PostgreSQL
sudo apt install postgresql postgresql-contrib
sudo -u postgres createdb gestao_agencia
sudo -u postgres createuser gestao_user
sudo -u postgres psql -c "GRANT ALL PRIVILEGES ON DATABASE gestao_agencia TO gestao_user;"
```

### 🔒 **SSL/HTTPS**

```bash
# Certbot para Let's Encrypt
sudo apt install certbot python3-certbot-nginx
sudo certbot --nginx -d seu-dominio.com
```

---

## 📊 **MONITORAMENTO**

### 🏥 **Health Checks**

```bash
# Verificar se a aplicação está funcionando
curl http://localhost/health/

# Verificar logs
docker-compose logs -f web

# Verificar recursos
docker stats
```

### 📈 **Métricas**

```python
# settings.py
LOGGING = {
    'version': 1,
    'disable_existing_loggers': False,
    'handlers': {
        'file': {
            'level': 'INFO',
            'class': 'logging.FileHandler',
            'filename': 'logs/app.log',
        },
    },
    'loggers': {
        'django': {
            'handlers': ['file'],
            'level': 'INFO',
        },
    },
}
```

### 🚨 **Alertas**

```bash
# Script de monitoramento
#!/bin/bash
if ! curl -f http://localhost/health/ > /dev/null 2>&1; then
    echo "ALERTA: Aplicação não está respondendo!" | mail -s "Alerta Gestão Agência" admin@empresa.com
fi
```

---

## 🎯 **CHECKLIST DE DEPLOY**

### ✅ **Pré-Deploy**
- [ ] Testes passando
- [ ] Código revisado
- [ ] Documentação atualizada
- [ ] Backup do banco
- [ ] Variáveis de ambiente configuradas

### ✅ **Deploy**
- [ ] Servidor preparado
- [ ] Docker instalado
- [ ] SSL configurado
- [ ] Domínio apontando
- [ ] Monitoramento ativo

### ✅ **Pós-Deploy**
- [ ] Aplicação respondendo
- [ ] Logs sem erros
- [ ] Performance OK
- [ ] Backup funcionando
- [ ] Alertas configurados

---

## 🚀 **PRÓXIMOS PASSOS**

1. **Escolher estratégia**: Docker + CI/CD recomendado
2. **Preparar servidor**: Instalar Docker e dependências
3. **Configurar domínio**: DNS e SSL
4. **Testar deploy**: Ambiente de staging
5. **Monitorar**: Logs e métricas
6. **Documentar**: Processos e procedimentos

---

## 📞 **SUPORTE**

- **Documentação**: Este guia
- **Issues**: GitHub Issues
- **Logs**: `docker-compose logs`
- **Backup**: `/opt/backups/`

**🎉 Com Docker e CI/CD, seu deploy será consistente, confiável e automatizado!** 