# Documentação Técnica - Gestão de Agência

## Visão Geral da Arquitetura

O sistema Gestão de Agência foi desenvolvido seguindo os princípios de arquitetura limpa e padrões do Django, com foco em escalabilidade, manutenibilidade e extensibilidade. A aplicação implementa uma hierarquia completa de gestão de projetos inspirada no Monday.com, com módulos bem definidos e responsabilidades claras.

### Arquitetura de Alto Nível

```
┌─────────────────┐    ┌─────────────────┐    ┌─────────────────┐
│   Frontend      │    │   Backend       │    │   Database      │
│   Bootstrap 5   │◄──►│   Django 5.2    │◄──►│   SQLite/       │
│   jQuery        │    │   DRF 3.16      │    │   PostgreSQL    │
└─────────────────┘    └─────────────────┘    └─────────────────┘
                              │
                              ▼
                       ┌─────────────────┐
                       │   Background    │
                       │   Celery +      │
                       │   Redis         │
                       └─────────────────┘
```

## Estrutura de Módulos

### 1. Core Modules

#### Client (client/)
Gerencia informações de clientes da agência.

**Modelo Principal:**
- `Client`: Dados básicos, CNPJ, contatos, endereço
- Relacionamentos: OneToMany com Workspace e Project

**Funcionalidades:**
- CRUD completo via admin e API
- Validação de CNPJ
- Histórico de alterações
- Filtros por status ativo

#### Collaborator (collaborator/)
Gestão de colaboradores e áreas organizacionais.

**Modelos:**
- `Area`: Departamentos/áreas da empresa com cores
- `Collaborator`: Colaboradores vinculados a usuários Django

**Funcionalidades:**
- Integração com sistema de autenticação Django
- Controle de valor/hora por colaborador
- Organização por áreas com cores personalizadas
- Perfis de usuário estendidos

### 2. Project Management Modules

#### Workspace (workspace/)
Espaços de trabalho organizados por cliente.

**Características:**
- Agrupamento lógico de projetos por cliente
- Cores personalizadas para identificação visual
- Controle de acesso baseado em permissões

#### Portfolio (portfolio/)
Agrupamento de projetos relacionados dentro de um workspace.

**Funcionalidades:**
- Organização temática de projetos
- Estatísticas agregadas
- Relatórios consolidados

#### Project (project/)
Núcleo da gestão de projetos.

**Modelo Principal:**
```python
class Project(models.Model):
    nome = models.CharField(max_length=200)
    portfolio = models.ForeignKey(Portfolio)
    workspace = models.ForeignKey(Workspace)
    cliente = models.ForeignKey(Client)
    cronograma_inicio = models.DateField()
    cronograma_fim = models.DateField()
    status = models.CharField(choices=STATUS_CHOICES)
    orcamento = models.DecimalField()
    progresso = models.IntegerField(default=0)
```

**Funcionalidades:**
- Cálculo automático de progresso baseado em tarefas
- Controle de orçamento e esforço
- Timeline visual
- Relatórios de performance

#### Board (board/)
Quadros Kanban para organização visual.

**Características:**
- Múltiplos boards por projeto
- Ordenação customizável
- Visualização estilo Kanban

#### Task (task/)
Sistema completo de gestão de tarefas.

**Modelos:**
- `Group`: Colunas do Kanban (A Fazer, Em Andamento, Concluído)
- `Task`: Tarefas individuais com todas as funcionalidades

**Funcionalidades Avançadas:**
- Sistema de dependências entre tarefas
- Controle de tempo integrado (cronômetro)
- Cálculo automático de progresso
- Notificações automáticas
- Histórico de alterações

### 3. System Modules

#### Notification (notification/)
Sistema completo de notificações.

**Tipos de Notificação:**
- Tarefa criada
- Responsável adicionado
- Tarefa concluída
- Tarefa atrasada
- Prazo próximo

**Funcionalidades:**
- Notificações em tempo real
- Envio por email assíncrono
- Marcação de leitura
- Filtros e busca

## Tecnologias e Dependências

### Backend Core
- **Django 5.2**: Framework web principal
- **Django REST Framework 3.16**: API REST
- **django-cors-headers**: CORS para frontend
- **Celery 5.3**: Tarefas assíncronas
- **Redis**: Broker para Celery e cache

### Frontend
- **Bootstrap 5**: Framework CSS responsivo
- **jQuery**: Manipulação DOM e AJAX
- **Font Awesome**: Ícones
- **Chart.js**: Gráficos e dashboards

### Desenvolvimento
- **django-extensions**: Ferramentas de desenvolvimento
- **django-debug-toolbar**: Debug em desenvolvimento

## Sistema de Permissões

### Grupos de Usuários

1. **Gerentes**
   - Acesso completo a todos os módulos
   - Gestão de equipes e projetos
   - Relatórios gerenciais
   - Configurações do sistema

2. **Colaboradores**
   - Visualização de projetos onde participam
   - Edição de suas próprias tarefas
   - Controle de tempo pessoal
   - Notificações relevantes

3. **Clientes**
   - Visualização apenas de seus projetos
   - Acompanhamento de progresso
   - Relatórios de seus projetos

### Middleware Customizado

#### CollaboratorRequiredMiddleware
Verifica se usuários autenticados possuem perfil de colaborador ativo.

#### ProjectAccessMiddleware
Controla acesso a projetos baseado nas permissões do usuário.

#### AuditMiddleware
Registra ações dos usuários para auditoria.

## API REST

### Estrutura de Endpoints

```
/api/
├── clients/                 # CRUD de clientes
├── areas/                   # Áreas organizacionais
├── collaborators/           # Colaboradores
├── workspaces/             # Workspaces
├── portfolios/             # Portfolios
├── projects/               # Projetos
├── boards/                 # Boards
├── groups/                 # Grupos de tarefas
├── tasks/                  # Tarefas
└── notifications/          # Notificações
```

### Autenticação
- Token-based authentication
- Permissões granulares por endpoint
- Rate limiting configurável

### Funcionalidades Especiais

#### Task API
- `POST /api/tasks/{id}/start_timer/`: Iniciar cronômetro
- `POST /api/tasks/{id}/stop_timer/`: Parar cronômetro
- `PATCH /api/tasks/{id}/update_status/`: Atualizar status
- `GET /api/tasks/my_tasks/`: Tarefas do usuário logado

#### Project API
- `GET /api/projects/{id}/statistics/`: Estatísticas do projeto
- `GET /api/projects/{id}/timeline/`: Timeline do projeto
- `POST /api/projects/{id}/duplicate/`: Duplicar projeto

## Sistema de Notificações

### Arquitetura
```
Django Signal → Celery Task → Email/Database → Frontend Update
```

### Tipos de Eventos
1. **Criação de Tarefa**: Notifica responsável
2. **Mudança de Status**: Notifica stakeholders
3. **Prazo Próximo**: Alerta 2 dias antes
4. **Tarefa Atrasada**: Verificação diária
5. **Conclusão**: Notifica gerentes

### Processamento Assíncrono
- **Celery Worker**: Processa envio de emails
- **Celery Beat**: Tarefas periódicas (verificação de prazos)
- **Redis**: Queue e cache de notificações

## Controle de Tempo

### Funcionalidades
- Cronômetro integrado nas tarefas
- Registro automático de tempo trabalhado
- Relatórios de produtividade
- Cálculo de custos baseado em valor/hora

### Implementação
```python
class Task(models.Model):
    tempo_inicio = models.DateTimeField(null=True)
    tempo_fim = models.DateTimeField(null=True)
    esforco_utilizado = models.DecimalField()
    
    def iniciar_cronometro(self):
        self.tempo_inicio = timezone.now()
        self.save()
    
    def parar_cronometro(self):
        if self.tempo_inicio:
            self.tempo_fim = timezone.now()
            delta = self.tempo_fim - self.tempo_inicio
            self.esforco_utilizado += delta.total_seconds() / 3600
            self.save()
```

## Signals e Automações

### Task Signals
- **pre_save**: Validações e cálculos antes de salvar
- **post_save**: Recálculo de progresso em cascata
- **m2m_changed**: Atualização de dependências

### Cascata de Cálculos
```
Task → Group → Board → Project → Portfolio
```

Quando uma tarefa é atualizada, o progresso é recalculado automaticamente em toda a hierarquia.

## Testes

### Estrutura de Testes
```python
# Exemplo de teste de modelo
class TaskModelTest(TestCase):
    def setUp(self):
        self.client = Client.objects.create(nome="Test Client")
        self.collaborator = Collaborator.objects.create(...)
    
    def test_task_progress_calculation(self):
        task = Task.objects.create(...)
        self.assertEqual(task.progresso_percentual, 0)
```

### Cobertura
- Modelos: 95%+
- Views: 85%+
- APIs: 90%+
- Signals: 80%+

## Performance e Otimização

### Database Optimization
- Índices em campos de busca frequente
- Select_related e prefetch_related em queries
- Paginação em listagens grandes

### Caching Strategy
- Redis para cache de sessões
- Cache de queries pesadas
- Cache de estatísticas calculadas

### Frontend Optimization
- Minificação de CSS/JS
- Lazy loading de imagens
- AJAX para atualizações parciais

## Deployment

### Estrutura Recomendada
```
/srv/gestao_agencia/
├── www/                    # Código da aplicação
├── .virtualenv/           # Ambiente virtual
├── logs/                  # Logs da aplicação
├── media/                 # Arquivos de upload
└── static/               # Arquivos estáticos
```

### Configurações de Produção
- PostgreSQL como banco principal
- Nginx como proxy reverso
- uWSGI como servidor WSGI
- Redis para cache e Celery
- Supervisor para gerenciar processos

### Variáveis de Ambiente
```bash
DJANGO_SETTINGS_MODULE=gestao_agencia.settings.production
DATABASE_URL=postgresql://user:pass@localhost/db
REDIS_URL=redis://localhost:6379/0
SECRET_KEY=your-secret-key
DEBUG=False
```

## Monitoramento

### Logs
- Django: Aplicação e erros
- Celery: Tarefas assíncronas
- Nginx: Acesso e proxy
- PostgreSQL: Queries e performance

### Métricas
- Tempo de resposta das APIs
- Taxa de conclusão de tarefas
- Utilização de recursos
- Erros por módulo

## Extensibilidade

### Adicionando Novos Módulos
1. Criar app Django: `python manage.py startapp novo_modulo`
2. Definir modelos com relacionamentos apropriados
3. Criar serializers para API
4. Implementar views e templates
5. Configurar URLs e permissões
6. Adicionar testes

### Customizações Comuns
- Novos tipos de notificação
- Campos customizados em modelos
- Relatórios específicos
- Integrações com sistemas externos

## Troubleshooting

### Problemas Comuns

1. **Migrações Conflitantes**
```bash
python manage.py migrate --fake-initial
python manage.py makemigrations --merge
```

2. **Celery Não Processa Tarefas**
- Verificar se Redis está rodando
- Confirmar configurações de BROKER_URL
- Verificar logs do worker

3. **Performance Lenta**
- Analisar queries com django-debug-toolbar
- Verificar índices no banco
- Otimizar queries N+1

### Logs de Debug
```python
# settings.py
LOGGING = {
    'version': 1,
    'handlers': {
        'file': {
            'level': 'DEBUG',
            'class': 'logging.FileHandler',
            'filename': 'debug.log',
        },
    },
    'loggers': {
        'django': {
            'handlers': ['file'],
            'level': 'DEBUG',
        },
    },
}
```

## Conclusão

O sistema Gestão de Agência representa uma solução completa e escalável para gestão de projetos, implementando as melhores práticas do Django e oferecendo uma experiência de usuário moderna e intuitiva. A arquitetura modular permite fácil manutenção e extensão, enquanto o sistema de permissões garante segurança e controle de acesso adequados.

A documentação técnica serve como guia para desenvolvedores que precisam entender, manter ou estender o sistema, fornecendo informações detalhadas sobre cada componente e suas interações.

