# 🧪 Testes Automáticos - Sistema de Gestão de Agência

Este documento descreve como executar e gerenciar os testes automáticos do sistema de gestão de agência.

## 📋 Índice

- [Visão Geral](#visão-geral)
- [Tipos de Testes](#tipos-de-testes)
- [Como Executar](#como-executar)
- [Estrutura dos Testes](#estrutura-dos-testes)
- [Cobertura de Código](#cobertura-de-código)
- [CI/CD](#cicd)
- [Relatórios](#relatórios)
- [Troubleshooting](#troubleshooting)

## 🎯 Visão Geral

O sistema possui uma suite completa de testes automáticos que validam:

- ✅ **Modelos**: Validação de dados, relacionamentos e métodos
- ✅ **Views**: Funcionalidades das páginas e formulários
- ✅ **APIs**: Endpoints REST e serialização de dados
- ✅ **Formulários**: Validação e processamento de dados
- ✅ **Integração**: Fluxos completos entre componentes
- ✅ **Segurança**: Verificações de vulnerabilidades
- ✅ **Performance**: Testes de carga e resposta

## 🧪 Tipos de Testes

### 1. Testes Unitários
Testam componentes individuais isoladamente.

```python
# Exemplo: test/models.py
class TaskModelTest(TestCase):
    def test_criacao_tarefa(self):
        tarefa = Task.objects.create(nome="Teste", ...)
        self.assertEqual(tarefa.nome, "Teste")
```

### 2. Testes de Views
Validam o comportamento das páginas web.

```python
# Exemplo: test/views.py
class TaskViewsTest(TestCase):
    def test_lista_tarefas(self):
        response = self.client.get('/tasks/')
        self.assertEqual(response.status_code, 200)
```

### 3. Testes de API
Verificam os endpoints REST.

```python
# Exemplo: test/api.py
class TaskAPITest(TestCase):
    def test_api_criar_tarefa(self):
        response = self.client.post('/api/tasks/', data)
        self.assertEqual(response.status_code, 201)
```

### 4. Testes de Integração
Testam fluxos completos entre componentes.

```python
# Exemplo: test/integration.py
class TaskIntegrationTest(TestCase):
    def test_fluxo_completo_tarefa(self):
        # Criar -> Editar -> Concluir -> Deletar
        pass
```

## 🚀 Como Executar

### Opção 1: Script Principal
```bash
# Executar todos os testes
python run_tests.py

# Executar com relatórios detalhados
python run_tests.py --verbose
```

### Opção 2: Suite Interativa
```bash
# Menu interativo
python test_suite.py

# Linha de comando
python test_suite.py unit          # Testes unitários
python test_suite.py integration   # Testes de integração
python test_suite.py api           # Testes de API
python test_suite.py coverage      # Com cobertura
```

### Opção 3: Django Management
```bash
# Todos os testes
python manage.py test

# App específico
python manage.py test task

# Padrão específico
python manage.py test --pattern="*ModelTest*"

# Com verbosidade
python manage.py test --verbosity=2
```

### Opção 4: Pytest
```bash
# Instalar pytest
pip install pytest pytest-django pytest-cov

# Executar com pytest
pytest

# Com cobertura
pytest --cov=. --cov-report=html
```

## 📁 Estrutura dos Testes

```
gestao_agencia/
├── task/
│   ├── tests.py              # Testes do app task
│   └── test_*.py            # Testes específicos
├── client/
│   └── tests.py             # Testes do app client
├── collaborator/
│   └── tests.py             # Testes do app collaborator
├── run_tests.py             # Script principal
├── test_suite.py            # Suite interativa
├── pytest.ini              # Configuração pytest
└── .github/workflows/       # CI/CD
    └── tests.yml
```

### Organização dos Testes

Cada app possui testes organizados em classes:

```python
# task/tests.py
class TaskModelTest(TestCase):      # Testes de modelo
class TaskViewsTest(TestCase):      # Testes de views
class TaskAPITest(TestCase):        # Testes de API
class TaskFormTest(TestCase):       # Testes de formulários
class TaskIntegrationTest(TestCase): # Testes de integração
```

## 📊 Cobertura de Código

### Gerar Relatório de Cobertura
```bash
# Com coverage
coverage run --source="." manage.py test
coverage report --show-missing
coverage html --directory=htmlcov

# Com pytest
pytest --cov=. --cov-report=html --cov-report=term-missing
```

### Visualizar Relatório
```bash
# Abrir no navegador
open htmlcov/index.html  # macOS
xdg-open htmlcov/index.html  # Linux
start htmlcov/index.html  # Windows
```

### Metas de Cobertura
- **Mínimo**: 80% de cobertura
- **Ideal**: 90%+ de cobertura
- **Crítico**: 100% para funcionalidades principais

## 🔄 CI/CD

### GitHub Actions
O sistema possui workflow automático que executa:

1. **Testes em múltiplas versões** (Python 3.9, 3.10, 3.11)
2. **Verificações de segurança** (Bandit, Safety)
3. **Testes de performance** (Locust)
4. **Análise de qualidade** (Flake8, Black, isort)

### Configuração Local
```bash
# Instalar dependências de desenvolvimento
pip install -r requirements-dev.txt

# Executar verificações
flake8 . --count --select=E9,F63,F7,F82
black --check .
isort --check-only --diff .
```

## 📋 Relatórios

### Relatórios Gerados
- `test_report.json` - Relatório em JSON
- `test_report.txt` - Relatório em texto
- `htmlcov/` - Relatório de cobertura HTML
- `coverage.xml` - Relatório de cobertura XML

### Exemplo de Relatório
```
RELATÓRIO DE TESTES - SISTEMA DE GESTÃO DE AGÊNCIA
============================================================

Data/Hora: 2025-01-21T20:40:16
Duração: 0:02:15.123456

RESUMO:
- Total de apps testados: 8
- Apps com sucesso: 8
- Apps com falha: 0

DETALHES POR APP:
----------------------------------------
task: ✅ PASSOU
client: ✅ PASSOU
collaborator: ✅ PASSOU
project: ✅ PASSOU
board: ✅ PASSOU
portfolio: ✅ PASSOU
workspace: ✅ PASSOU
notification: ✅ PASSOU
```

## 🔧 Troubleshooting

### Problemas Comuns

#### 1. Erro de Importação
```bash
# Erro: ModuleNotFoundError
# Solução: Verificar PYTHONPATH
export PYTHONPATH="${PYTHONPATH}:$(pwd)"
```

#### 2. Erro de Banco de Dados
```bash
# Erro: Database connection
# Solução: Executar migrações
python manage.py migrate
```

#### 3. Timeout nos Testes
```bash
# Erro: Timeout
# Solução: Aumentar timeout ou otimizar testes
python manage.py test --timeout=600
```

#### 4. Falha de Cobertura
```bash
# Erro: Coverage < 80%
# Solução: Adicionar mais testes
# Verificar arquivos não cobertos em htmlcov/
```

### Debug de Testes

#### Modo Verbose
```bash
python manage.py test --verbosity=3
```

#### Teste Específico
```bash
python manage.py test task.tests.TaskModelTest.test_criacao_tarefa
```

#### Com Debugger
```python
import pdb; pdb.set_trace()  # Adicionar no teste
```

## 📈 Métricas de Qualidade

### Indicadores
- **Cobertura de Código**: > 80%
- **Taxa de Sucesso**: > 95%
- **Tempo de Execução**: < 5 minutos
- **Vulnerabilidades**: 0 críticas

### Monitoramento
```bash
# Verificar métricas
python run_tests.py --metrics

# Histórico de execuções
cat test_report.json | jq '.summary'
```

## 🎯 Boas Práticas

### 1. Nomenclatura
```python
# ✅ Bom
def test_criacao_tarefa_com_dados_validos(self):
def test_edicao_tarefa_inexistente_retorna_404(self):

# ❌ Ruim
def test1(self):
def test_task(self):
```

### 2. Organização
```python
# ✅ Estrutura clara
class TaskModelTest(TestCase):
    def setUp(self):
        # Configuração comum
    
    def test_cenario_1(self):
        # Teste específico
    
    def test_cenario_2(self):
        # Outro teste
```

### 3. Assertions
```python
# ✅ Assertions específicos
self.assertEqual(result, expected)
self.assertIn(item, collection)
self.assertTrue(condition)

# ❌ Assertions genéricos
self.assertTrue(result == expected)
```

### 4. Dados de Teste
```python
# ✅ Usar factories ou fixtures
from .factories import TaskFactory

def test_criacao_tarefa(self):
    tarefa = TaskFactory()
    self.assertIsNotNone(tarefa.pk)
```

## 📚 Recursos Adicionais

### Documentação
- [Django Testing](https://docs.djangoproject.com/en/stable/topics/testing/)
- [Pytest Django](https://pytest-django.readthedocs.io/)
- [Coverage.py](https://coverage.readthedocs.io/)

### Ferramentas
- **pytest**: Framework de testes avançado
- **coverage**: Análise de cobertura
- **factory_boy**: Geração de dados de teste
- **faker**: Dados fictícios realistas

### Comandos Úteis
```bash
# Limpar cache de testes
find . -name "*.pyc" -delete
find . -name "__pycache__" -delete

# Executar testes em paralelo
python manage.py test --parallel

# Testes com dados aleatórios
python manage.py test --random-seed=42
```

---

## 🤝 Contribuição

Para contribuir com os testes:

1. **Siga os padrões** estabelecidos
2. **Mantenha a cobertura** acima de 80%
3. **Documente** novos tipos de teste
4. **Execute** todos os testes antes do commit
5. **Atualize** este documento se necessário

---

**Última atualização**: Janeiro 2025  
**Versão**: 1.0.0  
**Mantido por**: Equipe de Desenvolvimento 