from rest_framework import viewsets, status
from rest_framework.decorators import action
from rest_framework.response import Response
from rest_framework.permissions import IsAuthenticated
from django_filters.rest_framework import DjangoFilterBackend
from rest_framework.filters import SearchFilter, OrderingFilter

from .models import Board
from .serializers import (
    BoardSerializer, 
    BoardDetailSerializer, 
    BoardListSerializer,
    BoardCreateSerializer
)


class BoardViewSet(viewsets.ModelViewSet):
    """
    ViewSet para operações CRUD de boards.
    """
    
    queryset = Board.objects.all()
    permission_classes = [IsAuthenticated]
    filter_backends = [DjangoFilterBackend, SearchFilter, OrderingFilter]
    filterset_fields = ['ativo', 'project', 'workspace', 'cliente']
    search_fields = ['nome', 'descricao', 'project__nome', 'workspace__nome', 'cliente__nome']
    ordering_fields = ['nome', 'project__nome', 'ordem', 'created_at']
    ordering = ['project__nome', 'ordem', 'nome']

    def get_serializer_class(self):
        if self.action == 'create':
            return BoardCreateSerializer
        if self.action == 'retrieve':
            return BoardDetailSerializer
        if self.action == 'list':
            return BoardListSerializer
        return BoardSerializer

    def get_queryset(self):
        queryset = Board.objects.select_related('project', 'workspace', 'cliente').all()
        
        # Filtro por status ativo
        ativo = self.request.query_params.get('ativo', None)
        if ativo is not None:
            queryset = queryset.filter(ativo=ativo.lower() == 'true')
            
        # Filtro por projeto
        project_id = self.request.query_params.get('project_id', None)
        if project_id:
            queryset = queryset.filter(project_id=project_id)
            
        # Filtro por workspace
        workspace_id = self.request.query_params.get('workspace_id', None)
        if workspace_id:
            queryset = queryset.filter(workspace_id=workspace_id)
            
        # Filtro por cliente
        cliente_id = self.request.query_params.get('cliente_id', None)
        if cliente_id:
            queryset = queryset.filter(cliente_id=cliente_id)
            
        return queryset

    @action(detail=True, methods=['post'])
    def toggle_status(self, request, pk=None):
        """
        Alterna o status ativo/inativo do board.
        """
        board = self.get_object()
        board.ativo = not board.ativo
        board.save()
        
        serializer = self.get_serializer(board)
        return Response(serializer.data)

    @action(detail=True, methods=['get'])
    def groups(self, request, pk=None):
        """
        Retorna os grupos do board.
        """
        board = self.get_object()
        groups = board.groups.filter(ativo=True).order_by('ordem', 'nome')
        
        from task.serializers import GroupListSerializer
        serializer = GroupListSerializer(groups, many=True)
        return Response(serializer.data)

    @action(detail=True, methods=['get'])
    def tasks(self, request, pk=None):
        """
        Retorna todas as tarefas do board.
        """
        board = self.get_object()
        
        from task.models import Task
        tasks = Task.objects.filter(group__board=board)
        
        from task.serializers import TaskListSerializer
        serializer = TaskListSerializer(tasks, many=True)
        return Response(serializer.data)

    @action(detail=True, methods=['patch'])
    def reorder(self, request, pk=None):
        """
        Reordena a posição do board.
        """
        board = self.get_object()
        nova_ordem = request.data.get('ordem')
        
        if nova_ordem is not None:
            try:
                nova_ordem = int(nova_ordem)
                board.ordem = nova_ordem
                board.save()
                
                serializer = self.get_serializer(board)
                return Response(serializer.data)
            except ValueError:
                return Response(
                    {"error": "Ordem deve ser um número inteiro"}, 
                    status=status.HTTP_400_BAD_REQUEST
                )
        
        return Response(
            {"error": "Campo 'ordem' é obrigatório"}, 
            status=status.HTTP_400_BAD_REQUEST
        )

    @action(detail=True, methods=['get'])
    def kanban(self, request, pk=None):
        """
        Retorna dados estruturados para visualização Kanban.
        """
        board = self.get_object()
        
        kanban_data = {
            'board': BoardListSerializer(board).data,
            'groups': []
        }
        
        groups = board.groups.filter(ativo=True).order_by('ordem', 'nome')
        
        for group in groups:
            from task.serializers import GroupListSerializer, TaskListSerializer
            
            group_data = GroupListSerializer(group).data
            group_data['tasks'] = TaskListSerializer(
                group.tasks.all().order_by('cronograma_inicio', 'nome'), 
                many=True
            ).data
            
            kanban_data['groups'].append(group_data)
        
        return Response(kanban_data)

    @action(detail=True, methods=['get'])
    def statistics(self, request, pk=None):
        """
        Retorna estatísticas do board.
        """
        board = self.get_object()
        
        stats = {
            'total_groups': board.total_groups,
            'total_tasks': board.total_tasks,
            'tasks_concluidas': board.tasks_concluidas,
            'progresso_percentual': board.progresso_percentual,
            'tasks_por_status': {},
            'tasks_por_prioridade': {},
            'tasks_por_responsavel': {},
        }
        
        # Estatísticas de tarefas
        from task.models import Task
        tasks = Task.objects.filter(group__board=board)
        
        for task in tasks:
            # Por status
            status_key = task.status
            if status_key not in stats['tasks_por_status']:
                stats['tasks_por_status'][status_key] = 0
            stats['tasks_por_status'][status_key] += 1
            
            # Por prioridade
            prioridade_key = task.prioridade
            if prioridade_key not in stats['tasks_por_prioridade']:
                stats['tasks_por_prioridade'][prioridade_key] = 0
            stats['tasks_por_prioridade'][prioridade_key] += 1
            
            # Por responsável
            responsavel_key = task.responsavel.nome if task.responsavel else 'Sem responsável'
            if responsavel_key not in stats['tasks_por_responsavel']:
                stats['tasks_por_responsavel'][responsavel_key] = 0
            stats['tasks_por_responsavel'][responsavel_key] += 1
        
        return Response(stats)

