from django.db import models
from client.models import Client
from workspace.models import Workspace
from project.models import Project


class Board(models.Model):
    """
    Modelo para representar quadros (boards).
    Um board organiza grupos de tarefas dentro de um projeto.
    """
    
    nome = models.CharField(
        max_length=200,
        verbose_name="Nome",
        help_text="Nome do quadro"
    )
    
    project = models.ForeignKey(
        Project,
        on_delete=models.CASCADE,
        verbose_name="Projeto",
        help_text="Projeto ao qual o quadro pertence",
        related_name="boards"
    )
    
    workspace = models.ForeignKey(
        Workspace,
        on_delete=models.CASCADE,
        verbose_name="Workspace",
        help_text="Workspace ao qual o quadro pertence",
        related_name="boards"
    )
    
    cliente = models.ForeignKey(
        Client,
        on_delete=models.CASCADE,
        verbose_name="Cliente",
        help_text="Cliente proprietário do quadro",
        related_name="boards"
    )
    
    descricao = models.TextField(
        blank=True,
        null=True,
        verbose_name="Descrição",
        help_text="Descrição do quadro"
    )
    
    cor = models.CharField(
        max_length=7,
        default="#28a745",
        verbose_name="Cor",
        help_text="Cor do quadro (formato hexadecimal)"
    )
    
    ordem = models.IntegerField(
        default=0,
        verbose_name="Ordem",
        help_text="Ordem de exibição do quadro"
    )
    
    ativo = models.BooleanField(
        default=True,
        verbose_name="Ativo",
        help_text="Indica se o quadro está ativo"
    )
    
    created_at = models.DateTimeField(
        auto_now_add=True,
        verbose_name="Criado em"
    )
    
    updated_at = models.DateTimeField(
        auto_now=True,
        verbose_name="Atualizado em"
    )

    class Meta:
        verbose_name = "Quadro"
        verbose_name_plural = "Quadros"
        ordering = ['project', 'ordem', 'nome']
        unique_together = ['project', 'nome']

    def __str__(self):
        return f"{self.project.nome} - {self.nome}"

    def get_absolute_url(self):
        from django.urls import reverse
        return reverse('board:detail', kwargs={'pk': self.pk})

    @property
    def total_groups(self):
        return self.groups.count()

    @property
    def total_tasks(self):
        total = 0
        for group in self.groups.all():
            total += group.tasks.count()
        return total

    @property
    def tasks_concluidas(self):
        total = 0
        for group in self.groups.all():
            total += group.tasks.filter(status='concluida').count()
        return total

    @property
    def progresso_percentual(self):
        """Calcula o progresso percentual do quadro baseado nas tarefas."""
        total = self.total_tasks
        if total == 0:
            return 0
        concluidas = self.tasks_concluidas
        return round((concluidas / total) * 100, 2)

    def save(self, *args, **kwargs):
        # Garante que workspace e cliente sejam consistentes com o projeto
        if self.project:
            self.workspace = self.project.workspace
            self.cliente = self.project.cliente
        super().save(*args, **kwargs)

