from rest_framework import serializers
from .models import Board
from project.serializers import ProjectListSerializer
from workspace.serializers import WorkspaceListSerializer
from client.serializers import ClientListSerializer


class BoardSerializer(serializers.ModelSerializer):
    """
    Serializer para o modelo Board.
    """
    
    project_nome = serializers.CharField(source='project.nome', read_only=True)
    workspace_nome = serializers.CharField(source='workspace.nome', read_only=True)
    cliente_nome = serializers.CharField(source='cliente.nome', read_only=True)
    total_groups = serializers.ReadOnlyField()
    total_tasks = serializers.ReadOnlyField()
    tasks_concluidas = serializers.ReadOnlyField()
    progresso_percentual = serializers.ReadOnlyField()
    
    class Meta:
        model = Board
        fields = [
            'id', 'nome', 'project', 'project_nome', 'workspace', 'workspace_nome',
            'cliente', 'cliente_nome', 'descricao', 'cor', 'ordem', 'ativo',
            'total_groups', 'total_tasks', 'tasks_concluidas', 'progresso_percentual',
            'created_at', 'updated_at'
        ]
        read_only_fields = ['id', 'workspace', 'cliente', 'created_at', 'updated_at']


class BoardDetailSerializer(serializers.ModelSerializer):
    """
    Serializer detalhado para o modelo Board.
    """
    
    project = ProjectListSerializer(read_only=True)
    workspace = WorkspaceListSerializer(read_only=True)
    cliente = ClientListSerializer(read_only=True)
    groups = serializers.SerializerMethodField()
    progresso_percentual = serializers.ReadOnlyField()
    
    class Meta:
        model = Board
        fields = [
            'id', 'nome', 'project', 'workspace', 'cliente', 'descricao',
            'cor', 'ordem', 'ativo', 'groups', 'progresso_percentual',
            'created_at', 'updated_at'
        ]
        read_only_fields = ['id', 'workspace', 'cliente', 'created_at', 'updated_at']
    
    def get_groups(self, obj):
        from task.serializers import GroupListSerializer
        groups = obj.groups.filter(ativo=True).order_by('ordem', 'nome')
        return GroupListSerializer(groups, many=True).data


class BoardListSerializer(serializers.ModelSerializer):
    """
    Serializer simplificado para listagem de boards.
    """
    
    project_nome = serializers.CharField(source='project.nome', read_only=True)
    total_groups = serializers.ReadOnlyField()
    total_tasks = serializers.ReadOnlyField()
    progresso_percentual = serializers.ReadOnlyField()
    
    class Meta:
        model = Board
        fields = [
            'id', 'nome', 'project_nome', 'cor', 'ordem', 'ativo',
            'total_groups', 'total_tasks', 'progresso_percentual'
        ]


class BoardCreateSerializer(serializers.ModelSerializer):
    """
    Serializer para criação de boards.
    """
    
    class Meta:
        model = Board
        fields = ['nome', 'project', 'descricao', 'cor', 'ordem']
    
    def create(self, validated_data):
        # O workspace e cliente são automaticamente definidos pelo projeto
        project = validated_data['project']
        validated_data['workspace'] = project.workspace
        validated_data['cliente'] = project.cliente
        return super().create(validated_data)

