from django.shortcuts import render, get_object_or_404, redirect
from django.contrib import messages
from django.contrib.auth.decorators import login_required
from django.core.paginator import Paginator
from django.db.models import Q, Count
from django.http import JsonResponse
from django.views.generic import CreateView, UpdateView
from django.contrib.auth.mixins import LoginRequiredMixin
from django.urls import reverse_lazy
from .models import Board
from .forms import BoardForm
from project.models import Project
from workspace.models import Workspace
from task.models import Group, Task
from collaborator.models import Collaborator
from permissions.models import VisibilityPermission
from gestao_agencia.decorators import superuser_required
from gestao_agencia.mixins import SuperuserRequiredMixin


@login_required
def board_list(request):
    """
    Lista todos os quadros com filtros e paginação.
    """
    # Obter parâmetros de filtro
    search = request.GET.get('search', '')
    project_id = request.GET.get('project', '')
    workspace_id = request.GET.get('workspace', '')
    
    # Query base - filtra apenas quadros acessíveis
    boards = VisibilityPermission.get_accessible_objects(request.user, Board)
    boards = boards.select_related('project', 'project__portfolio', 'project__workspace')
    
    # Aplicar filtros
    if search:
        boards = boards.filter(
            Q(nome__icontains=search) |
            Q(project__nome__icontains=search) |
            Q(project__portfolio__nome__icontains=search) |
            Q(descricao__icontains=search)
        )
    
    if project_id:
        boards = boards.filter(project_id=project_id)
    
    if workspace_id:
        boards = boards.filter(project__workspace_id=workspace_id)
    
    # Ordenar por projeto e nome
    boards = boards.order_by('project__nome', 'nome')
    
    # Paginação
    paginator = Paginator(boards, 12)
    page_number = request.GET.get('page')
    page_obj = paginator.get_page(page_number)
    
    # Estatísticas baseadas nos quadros acessíveis
    accessible_boards = VisibilityPermission.get_accessible_objects(request.user, Board)
    total_boards = accessible_boards.count()
    boards_ativos = accessible_boards.filter(ativo=True).count()
    boards_com_tarefas = 0  # Será implementado quando Group estiver disponível
    total_groups = 0  # Será implementado quando Group estiver disponível
    
    # Lista de projetos e workspaces acessíveis para o filtro
    accessible_projects = VisibilityPermission.get_accessible_objects(request.user, Project)
    accessible_workspaces = VisibilityPermission.get_accessible_objects(request.user, Workspace)
    projects = accessible_projects.order_by('portfolio__nome', 'nome')
    workspaces = accessible_workspaces.order_by('nome')
    
    context = {
        'boards': page_obj,
        'page_obj': page_obj,
        'is_paginated': page_obj.has_other_pages(),
        'search': search,
        'project_id': project_id,
        'workspace_id': workspace_id,
        'total_boards': total_boards,
        'boards_ativos': boards_ativos,
        'boards_com_tarefas': boards_com_tarefas,
        'total_groups': total_groups,
        'projects': projects,
        'workspaces': workspaces,
    }
    
    return render(request, 'board/list.html', context)


@login_required
@superuser_required
def board_create(request):
    """
    Cria um novo quadro.
    """
    if request.method == 'POST':
        # Implementação básica - será expandida com forms
        nome = request.POST.get('nome')
        project_id = request.POST.get('project')
        descricao = request.POST.get('descricao', '')
        cor = request.POST.get('cor', '#007bff')
        
        if nome and project_id:
            try:
                project = Project.objects.get(id=project_id)
                board = Board.objects.create(
                    nome=nome,
                    project=project,
                    descricao=descricao,
                    cor=cor
                )
                messages.success(request, f'Quadro "{board.nome}" criado com sucesso!')
                return redirect('board:list')
            except Project.DoesNotExist:
                messages.error(request, 'Projeto não encontrado.')
            except Exception as e:
                messages.error(request, f'Erro ao criar quadro: {str(e)}')
        else:
            messages.error(request, 'Nome e projeto são obrigatórios.')
    
    # GET request - mostrar formulário
    projects = Project.objects.all().order_by('portfolio__nome', 'nome')
    
    context = {
        'projects': projects,
        'board': None,
    }
    
    return render(request, 'board/form.html', context)


@login_required
def board_detail(request, pk):
    """
    Exibe os detalhes de um quadro.
    """
    board = get_object_or_404(Board, pk=pk)
    
    # Verifica se o usuário tem acesso ao quadro
    if not VisibilityPermission.has_access(request.user, board):
        messages.error(request, "Acesso negado. Você não tem permissão para visualizar este quadro.")
        return redirect('board:list')
    
    context = {
        'board': board,
    }
    
    return render(request, 'board/detail.html', context)


@login_required
@superuser_required
def board_edit(request, pk):
    """
    Edita um quadro existente.
    """
    board = get_object_or_404(Board, pk=pk)
    
    if request.method == 'POST':
        # Implementação básica - será expandida com forms
        nome = request.POST.get('nome')
        descricao = request.POST.get('descricao', '')
        cor = request.POST.get('cor', board.cor)
        ativo = request.POST.get('ativo') == 'on'
        
        if nome:
            board.nome = nome
            board.descricao = descricao
            board.cor = cor
            board.ativo = ativo
            board.save()
            messages.success(request, f'Quadro "{board.nome}" atualizado com sucesso!')
            return redirect('board:detail', pk=board.pk)
        else:
            messages.error(request, 'Nome é obrigatório.')
    
    # GET request - mostrar formulário
    projects = Project.objects.filter(ativo=True).order_by('portfolio__nome', 'nome')
    
    context = {
        'board': board,
        'projects': projects,
    }
    
    return render(request, 'board/form.html', context)


@login_required
@superuser_required
def board_delete(request, pk):
    """
    Exclui um quadro.
    """
    board = get_object_or_404(Board, pk=pk)
    
    if request.method == 'POST':
        nome = board.nome
        board.delete()
        messages.success(request, f'Quadro "{nome}" excluído com sucesso!')
        return redirect('board:list')
    
    context = {
        'board': board,
    }
    
    return render(request, 'board/delete.html', context)


@login_required
@superuser_required
def board_add_group(request, pk):
    """
    Adiciona um grupo ao quadro.
    """
    board = get_object_or_404(Board, pk=pk)
    
    if request.method == 'POST':
        nome = request.POST.get('nome')
        tipo = request.POST.get('tipo', 'desenvolvimento')
        descricao = request.POST.get('descricao', '')
        cor = request.POST.get('cor', '#6c757d')
        ordem = request.POST.get('ordem', 0)
        limite_tarefas = request.POST.get('limite_tarefas', '')
        data_inicio_prevista = request.POST.get('data_inicio_prevista', '')
        data_fim_prevista = request.POST.get('data_fim_prevista', '')
        ativo = request.POST.get('ativo') == 'on'
        
        if nome and tipo:
            try:
                group = Group.objects.create(
                    nome=nome,
                    board=board,
                    tipo=tipo,
                    descricao=descricao,
                    cor=cor,
                    ordem=ordem,
                    ativo=ativo
                )
                
                # Campos opcionais
                if limite_tarefas:
                    group.limite_tarefas = int(limite_tarefas)
                
                if data_inicio_prevista:
                    group.data_inicio_prevista = data_inicio_prevista
                
                if data_fim_prevista:
                    group.data_fim_prevista = data_fim_prevista
                
                group.save()
                
                messages.success(request, f'Grupo "{group.nome}" criado com sucesso!')
                return redirect('board:detail', pk=board.pk)
            except Exception as e:
                messages.error(request, f'Erro ao criar grupo: {str(e)}')
        else:
            messages.error(request, 'Nome e tipo do grupo são obrigatórios.')
    
    context = {
        'board': board,
    }
    
    return render(request, 'board/add_group.html', context)


@login_required
@superuser_required
def board_add_task(request, pk):
    """
    Adiciona uma tarefa ao quadro.
    """
    board = get_object_or_404(Board, pk=pk)
    
    if request.method == 'POST':
        nome = request.POST.get('nome')
        group_id = request.POST.get('group')
        descricao = request.POST.get('descricao', '')
        status = request.POST.get('status', 'nao_iniciada')
        prioridade = request.POST.get('prioridade', 'media')
        responsavel_id = request.POST.get('responsavel', '')
        
        if nome and group_id:
            try:
                group = Group.objects.get(id=group_id, board=board)
                task = Task.objects.create(
                    nome=nome,
                    group=group,
                    descricao=descricao,
                    status=status,
                    prioridade=prioridade,
                    cronograma_inicio=request.POST.get('cronograma_inicio'),
                    cronograma_fim=request.POST.get('cronograma_fim'),
                    duracao_dias=request.POST.get('duracao_dias', 1),
                    esforco_previsto=request.POST.get('esforco_previsto', 0)
                )
                
                if responsavel_id:
                    try:
                        responsavel = Collaborator.objects.get(id=responsavel_id)
                        task.responsavel = responsavel
                        task.save()
                    except Collaborator.DoesNotExist:
                        pass
                
                messages.success(request, f'Tarefa "{task.nome}" criada com sucesso!')
                return redirect('board:detail', pk=board.pk)
            except Group.DoesNotExist:
                messages.error(request, 'Grupo não encontrado.')
            except Exception as e:
                messages.error(request, f'Erro ao criar tarefa: {str(e)}')
        else:
            messages.error(request, 'Nome e grupo são obrigatórios.')
    
    # GET request - mostrar formulário
    groups = board.groups.filter(ativo=True).order_by('ordem', 'nome')
    colaboradores = Collaborator.objects.filter(ativo=True).order_by('nome')
    
    context = {
        'board': board,
        'groups': groups,
        'colaboradores': colaboradores,
    }
    
    return render(request, 'board/add_task.html', context)


class BoardCreateModalView(LoginRequiredMixin, SuperuserRequiredMixin, CreateView):
    """
    View para criação de board via modal.
    """
    model = Board
    form_class = BoardForm
    template_name = 'board/modal_create.html'
    
    def get_context_data(self, **kwargs):
        context = super().get_context_data(**kwargs)
        context.update({
            'modal_id': 'createBoardModal',
            'modal_title': 'Novo Board',
            'modal_icon': 'bi-kanban',
            'form_id': 'boardForm',
            'submit_text': 'Criar Board'
        })
        return context
    
    def form_valid(self, form):
        board = form.save()
        
        if self.request.headers.get('X-Requested-With') == 'XMLHttpRequest':
            return JsonResponse({
                'success': True,
                'message': 'Board criado com sucesso!',
                'obj': {
                    'id': board.id,
                    'nome': board.nome,
                    'detail_url': reverse_lazy('board:detail', kwargs={'pk': board.id})
                }
            })
        return super().form_valid(form)
    
    def form_invalid(self, form):
        if self.request.headers.get('X-Requested-With') == 'XMLHttpRequest':
            return JsonResponse({
                'success': False,
                'errors': form.errors
            })
        return super().form_invalid(form)


class BoardUpdateModalView(LoginRequiredMixin, SuperuserRequiredMixin, UpdateView):
    """
    View para edição de board via modal.
    """
    model = Board
    form_class = BoardForm
    template_name = 'board/modal_edit.html'
    
    def get_context_data(self, **kwargs):
        context = super().get_context_data(**kwargs)
        board = self.get_object()
        context.update({
            'modal_id': 'editBoardModal',
            'modal_title': f'Editar Board: {board.nome}',
            'modal_icon': 'bi-kanban-gear',
            'form_id': 'boardEditForm',
            'submit_text': 'Atualizar Board'
        })
        return context
    
    def form_valid(self, form):
        board = form.save()
        
        if self.request.headers.get('X-Requested-With') == 'XMLHttpRequest':
            return JsonResponse({
                'success': True,
                'message': 'Board atualizado com sucesso!',
                'obj': {
                    'id': board.id,
                    'nome': board.nome,
                    'ativo': board.ativo,
                    'detail_url': reverse_lazy('board:detail', kwargs={'pk': board.id})
                }
            })
        return super().form_valid(form)
    
    def form_invalid(self, form):
        if self.request.headers.get('X-Requested-With') == 'XMLHttpRequest':
            return JsonResponse({
                'success': False,
                'errors': form.errors
            })
        return super().form_invalid(form)
