from rest_framework import viewsets, status
from rest_framework.decorators import action
from rest_framework.response import Response
from rest_framework.permissions import IsAuthenticated
from django_filters.rest_framework import DjangoFilterBackend
from rest_framework.filters import SearchFilter, OrderingFilter

from .models import Client
from .serializers import (
    ClientSerializer, 
    ClientListSerializer
)


class ClientViewSet(viewsets.ModelViewSet):
    """
    ViewSet para operações CRUD de clientes.
    """
    
    queryset = Client.objects.all()
    permission_classes = [IsAuthenticated]
    filter_backends = [DjangoFilterBackend, SearchFilter, OrderingFilter]
    filterset_fields = ['ativo']
    search_fields = ['nome', 'cnpj', 'email']
    ordering_fields = ['nome', 'created_at']
    ordering = ['nome']
    
    def get_serializer_class(self):
        if self.action == 'list':
            return ClientListSerializer
        return ClientSerializer
    
    def get_queryset(self):
        queryset = Client.objects.all()
        
        # Filtro por status ativo
        ativo = self.request.query_params.get('ativo', None)
        if ativo is not None:
            queryset = queryset.filter(ativo=ativo.lower() == 'true')
        
        return queryset
    
    @action(detail=True, methods=['post'])
    def toggle_status(self, request, pk=None):
        """
        Alterna o status ativo/inativo do cliente.
        """
        client = self.get_object()
        client.ativo = not client.ativo
        client.save()
        
        serializer = self.get_serializer(client)
        return Response(serializer.data)
    
    @action(detail=True, methods=['get'])
    def workspaces(self, request, pk=None):
        """
        Retorna os workspaces do cliente.
        """
        client = self.get_object()
        workspaces = client.workspaces.filter(ativo=True)
        
        from workspace.serializers import WorkspaceListSerializer
        serializer = WorkspaceListSerializer(workspaces, many=True)
        return Response(serializer.data)
    
    @action(detail=True, methods=['get'])
    def projects(self, request, pk=None):
        """
        Retorna os projetos do cliente.
        """
        client = self.get_object()
        projects = client.projects.filter(ativo=True)
        
        from project.serializers import ProjectListSerializer
        serializer = ProjectListSerializer(projects, many=True)
        return Response(serializer.data)
    
    @action(detail=True, methods=['get'])
    def statistics(self, request, pk=None):
        """
        Retorna estatísticas do cliente.
        """
        client = self.get_object()
        
        stats = {
            'total_workspaces': client.workspaces.count(),
            'workspaces_ativos': client.workspaces.filter(ativo=True).count(),
            'total_portfolios': client.portfolios.count(),
            'portfolios_ativos': client.portfolios.filter(ativo=True).count(),
            'total_projects': client.projects.count(),
            'projects_ativos': client.projects.filter(ativo=True).count(),
            'projects_concluidos': client.projects.filter(status='concluido').count(),
            'projects_em_andamento': client.projects.filter(status='em_andamento').count(),
            'total_boards': client.boards.count(),
            'boards_ativos': client.boards.filter(ativo=True).count(),
        }
        
        return Response(stats)

