from django import forms
from .models import Client


class ClientForm(forms.ModelForm):
    """
    Formulário para criação e edição de clientes.
    """
    
    class Meta:
        model = Client
        fields = ['nome', 'cnpj', 'email', 'telefone', 'endereco', 'ativo']
        widgets = {
            'nome': forms.TextInput(attrs={
                'class': 'form-control',
                'placeholder': 'Nome do cliente'
            }),
            'cnpj': forms.TextInput(attrs={
                'class': 'form-control',
                'placeholder': 'XX.XXX.XXX/XXXX-XX',
                'data-mask': '00.000.000/0000-00'
            }),
            'email': forms.EmailInput(attrs={
                'class': 'form-control',
                'placeholder': 'email@exemplo.com'
            }),
            'telefone': forms.TextInput(attrs={
                'class': 'form-control',
                'placeholder': '(11) 99999-9999',
                'data-mask': '(00) 00000-0000'
            }),
            'endereco': forms.Textarea(attrs={
                'class': 'form-control',
                'rows': 3,
                'placeholder': 'Endereço completo do cliente'
            }),
            'ativo': forms.CheckboxInput(attrs={
                'class': 'form-check-input'
            })
        }
    
    def __init__(self, *args, **kwargs):
        super().__init__(*args, **kwargs)
        
        # Adiciona classes CSS aos labels
        for field_name, field in self.fields.items():
            field.widget.attrs.update({'class': field.widget.attrs.get('class', '') + ' form-control'})
            
        # Remove a classe form-control do checkbox
        self.fields['ativo'].widget.attrs['class'] = 'form-check-input'
    
    def clean_cnpj(self):
        cnpj = self.cleaned_data.get('cnpj')
        if cnpj:
            # Remove caracteres não numéricos
            import re
            cnpj_numbers = re.sub(r'\D', '', cnpj)
            
            # Verifica se tem 14 dígitos
            if len(cnpj_numbers) != 14:
                raise forms.ValidationError("CNPJ deve ter 14 dígitos.")
            
            # Formata o CNPJ
            formatted_cnpj = f"{cnpj_numbers[:2]}.{cnpj_numbers[2:5]}.{cnpj_numbers[5:8]}/{cnpj_numbers[8:12]}-{cnpj_numbers[12:14]}"
            
            # Verifica se já existe outro cliente com este CNPJ
            existing_client = Client.objects.filter(cnpj=formatted_cnpj).exclude(pk=self.instance.pk if self.instance else None)
            if existing_client.exists():
                raise forms.ValidationError("Já existe um cliente com este CNPJ.")
            
            return formatted_cnpj
        
        return cnpj

