from django.db import models
from django.core.validators import RegexValidator


class Client(models.Model):
    """
    Modelo para representar clientes da agência.
    """
    nome = models.CharField(
        max_length=200,
        verbose_name="Nome",
        help_text="Nome do cliente"
    )
    
    cnpj = models.CharField(
        max_length=18,
        unique=True,
        verbose_name="CNPJ",
        help_text="CNPJ do cliente (formato: XX.XXX.XXX/XXXX-XX)",
        validators=[
            RegexValidator(
                regex=r'^\d{2}\.\d{3}\.\d{3}\/\d{4}-\d{2}$',
                message='CNPJ deve estar no formato XX.XXX.XXX/XXXX-XX'
            )
        ]
    )
    
    email = models.EmailField(
        blank=True,
        null=True,
        verbose_name="E-mail",
        help_text="E-mail de contato do cliente"
    )
    
    telefone = models.CharField(
        max_length=20,
        blank=True,
        null=True,
        verbose_name="Telefone",
        help_text="Telefone de contato do cliente"
    )
    
    endereco = models.TextField(
        blank=True,
        null=True,
        verbose_name="Endereço",
        help_text="Endereço completo do cliente"
    )
    
    ativo = models.BooleanField(
        default=True,
        verbose_name="Ativo",
        help_text="Indica se o cliente está ativo"
    )
    
    created_at = models.DateTimeField(
        auto_now_add=True,
        verbose_name="Criado em"
    )
    
    updated_at = models.DateTimeField(
        auto_now=True,
        verbose_name="Atualizado em"
    )

    class Meta:
        verbose_name = "Cliente"
        verbose_name_plural = "Clientes"
        ordering = ['nome']

    def __str__(self):
        return self.nome

    def get_absolute_url(self):
        from django.urls import reverse
        return reverse('client:detail', kwargs={'pk': self.pk})

