from rest_framework import serializers
from .models import Client


class ClientSerializer(serializers.ModelSerializer):
    """
    Serializer para o modelo Client.
    """
    
    class Meta:
        model = Client
        fields = [
            'id', 'nome', 'cnpj', 'email', 'telefone', 'endereco', 
            'ativo', 'created_at', 'updated_at'
        ]
        read_only_fields = ['id', 'created_at', 'updated_at']

    def validate_cnpj(self, value):
        """Validação customizada para CNPJ."""
        import re
        
        # Remove caracteres não numéricos
        cnpj_numbers = re.sub(r'\D', '', value)
        
        # Verifica se tem 14 dígitos
        if len(cnpj_numbers) != 14:
            raise serializers.ValidationError("CNPJ deve ter 14 dígitos.")
        
        # Formata o CNPJ
        formatted_cnpj = f"{cnpj_numbers[:2]}.{cnpj_numbers[2:5]}.{cnpj_numbers[5:8]}/{cnpj_numbers[8:12]}-{cnpj_numbers[12:14]}"
        
        return formatted_cnpj


class ClientListSerializer(serializers.ModelSerializer):
    """
    Serializer simplificado para listagem de clientes.
    """
    
    total_workspaces = serializers.SerializerMethodField()
    total_projects = serializers.SerializerMethodField()
    
    class Meta:
        model = Client
        fields = ['id', 'nome', 'cnpj', 'ativo', 'total_workspaces', 'total_projects']
    
    def get_total_workspaces(self, obj):
        return obj.workspaces.count()
    
    def get_total_projects(self, obj):
        return obj.projects.count()

