from django.test import TestCase, Client as DjangoTestClient
from django.contrib.auth.models import User
from django.urls import reverse
from django.core.exceptions import ValidationError
import json

from .models import Client


class ClientModelTest(TestCase):
    """Testes para o modelo Client"""
    
    def setUp(self):
        """Configuração inicial para os testes"""
        self.client_data = {
            'nome': 'Empresa Teste LTDA',
            'cnpj': '12.345.678/0001-90',
            'email': 'contato@empresateste.com',
            'telefone': '(11) 99999-9999',
            'endereco': 'Rua Teste, 123 - São Paulo/SP',
            'ativo': True
        }

    def test_criacao_cliente(self):
        """Testa a criação básica de um cliente"""
        cliente = Client.objects.create(**self.client_data)
        self.assertEqual(cliente.nome, 'Empresa Teste LTDA')
        self.assertEqual(cliente.cnpj, '12.345.678/0001-90')
        self.assertEqual(cliente.email, 'contato@empresateste.com')
        self.assertTrue(cliente.ativo)

    def test_cnpj_unico(self):
        """Testa se o CNPJ é único"""
        Client.objects.create(**self.client_data)
        
        # Tentar criar outro cliente com o mesmo CNPJ
        with self.assertRaises(Exception):
            Client.objects.create(
                nome='Outra Empresa LTDA',
                cnpj='12.345.678/0001-90',  # Mesmo CNPJ
                email='outra@empresa.com',
                telefone='(11) 88888-8888',
                endereco='Rua Outra, 456 - São Paulo/SP',
                ativo=True
            )

    def test_validacao_cnpj_formato(self):
        """Testa a validação do formato do CNPJ"""
        # CNPJ com formato inválido
        with self.assertRaises(ValidationError):
            cliente = Client(
                nome='Empresa Formato Inválido LTDA',
                cnpj='123456789',  # Formato inválido
                email='formato@empresa.com',
                telefone='(11) 77777-7777',
                endereco='Rua Formato, 789 - São Paulo/SP',
                ativo=True
            )
            cliente.full_clean()

    def test_cnpj_formato_valido(self):
        """Testa CNPJ com formato válido"""
        cliente = Client(
            nome='Empresa Formato Válido LTDA',
            cnpj='98.765.432/0001-10',
            email='valido@empresa.com',
            telefone='(11) 66666-6666',
            endereco='Rua Válido, 321 - São Paulo/SP',
            ativo=True
        )
        cliente.full_clean()  # Não deve levantar exceção

    def test_cliente_inativo(self):
        """Testa cliente inativo"""
        client_data_inativo = self.client_data.copy()
        client_data_inativo['ativo'] = False
        cliente = Client.objects.create(**client_data_inativo)
        self.assertFalse(cliente.ativo)

    def test_str_representation(self):
        """Testa a representação string do cliente"""
        cliente = Client.objects.create(**self.client_data)
        self.assertEqual(str(cliente), 'Empresa Teste LTDA')

    def test_get_absolute_url(self):
        """Testa a URL absoluta do cliente"""
        cliente = Client.objects.create(**self.client_data)
        expected_url = f'/clients/{cliente.pk}/'
        self.assertEqual(cliente.get_absolute_url(), expected_url)

    def test_ordering(self):
        """Testa a ordenação dos clientes por nome"""
        # Criar clientes em ordem não alfabética
        Client.objects.create(
            nome='Zebra Empresa LTDA',
            cnpj='11.111.111/0001-11',
            email='zebra@empresa.com',
            telefone='(11) 11111-1111',
            endereco='Rua Zebra, 111 - São Paulo/SP',
            ativo=True
        )
        
        Client.objects.create(
            nome='Alpha Empresa LTDA',
            cnpj='22.222.222/0001-22',
            email='alpha@empresa.com',
            telefone='(11) 22222-2222',
            endereco='Rua Alpha, 222 - São Paulo/SP',
            ativo=True
        )
        
        # Criar o cliente do setUp também
        Client.objects.create(**self.client_data)
        
        clientes = Client.objects.all()
        self.assertEqual(clientes[0].nome, 'Alpha Empresa LTDA')
        self.assertEqual(clientes[1].nome, 'Empresa Teste LTDA')
        self.assertEqual(clientes[2].nome, 'Zebra Empresa LTDA')


class ClientViewsTest(TestCase):
    """Testes para as views do app client"""
    
    def setUp(self):
        """Configuração inicial para os testes"""
        self.client_instance = DjangoTestClient()
        self.user = User.objects.create_user(
            username='testuser',
            password='testpass123',
            email='test@example.com'
        )
        
        self.client_data = {
            'nome': 'Empresa Teste LTDA',
            'cnpj': '12.345.678/0001-90',
            'email': 'contato@empresateste.com',
            'telefone': '(11) 99999-9999',
            'endereco': 'Rua Teste, 123 - São Paulo/SP',
            'ativo': True
        }
        
        self.cliente = Client.objects.create(**self.client_data)

    def test_lista_clientes_sem_login(self):
        """Testa acesso à lista de clientes sem login"""
        response = self.client_instance.get(reverse('client:list'))
        self.assertEqual(response.status_code, 302)  # Redirecionamento para login

    def test_lista_clientes_com_login(self):
        """Testa acesso à lista de clientes com login"""
        self.client_instance.force_login(self.user)
        response = self.client_instance.get(reverse('client:list'))
        self.assertEqual(response.status_code, 200)
        self.assertContains(response, 'Empresa Teste LTDA')

    def test_detalhe_cliente(self):
        """Testa a view de detalhe do cliente"""
        self.client_instance.force_login(self.user)
        response = self.client_instance.get(reverse('client:detail', kwargs={'pk': self.cliente.pk}))
        self.assertEqual(response.status_code, 200)
        self.assertContains(response, 'Empresa Teste LTDA')

    def test_criar_cliente(self):
        """Testa a criação de um novo cliente"""
        self.client_instance.force_login(self.user)
        
        data = {
            'nome': 'Nova Empresa LTDA',
            'cnpj': '98.765.432/0001-10',
            'email': 'nova@empresa.com',
            'telefone': '(11) 88888-8888',
            'endereco': 'Rua Nova, 456 - São Paulo/SP',
            'ativo': True
        }
        
        response = self.client_instance.post(reverse('client:create'), data, follow=True)
        self.assertEqual(response.status_code, 200)  # Página de sucesso
        
        # Verificar se o cliente foi criado
        novo_cliente = Client.objects.get(nome='Nova Empresa LTDA')
        self.assertEqual(novo_cliente.cnpj, '98.765.432/0001-10')

    def test_editar_cliente(self):
        """Testa a edição de um cliente"""
        self.client_instance.force_login(self.user)
        
        data = {
            'nome': 'Empresa Editada LTDA',
            'cnpj': '12.345.678/0001-90',
            'email': 'editado@empresa.com',
            'telefone': '(11) 77777-7777',
            'endereco': 'Rua Editada, 789 - São Paulo/SP',
            'ativo': False
        }
        
        response = self.client_instance.post(
            reverse('client:edit', kwargs={'pk': self.cliente.pk}),
            data
        )
        self.assertEqual(response.status_code, 302)
        
        # Verificar se o cliente foi editado
        self.cliente.refresh_from_db()
        self.assertEqual(self.cliente.nome, 'Empresa Editada LTDA')
        self.assertFalse(self.cliente.ativo)

    def test_deletar_cliente(self):
        """Testa a exclusão de um cliente"""
        self.client_instance.force_login(self.user)
        
        response = self.client_instance.post(
            reverse('client:delete', kwargs={'pk': self.cliente.pk})
        )
        self.assertEqual(response.status_code, 302)
        
        # Verificar se o cliente foi deletado
        with self.assertRaises(Client.DoesNotExist):
            Client.objects.get(pk=self.cliente.pk)

    def test_modal_criar_cliente(self):
        """Testa o modal de criação de cliente"""
        self.client_instance.force_login(self.user)
        response = self.client_instance.get(reverse('client:create_modal'))
        self.assertEqual(response.status_code, 200)

    def test_modal_editar_cliente(self):
        """Testa o modal de edição de cliente"""
        self.client_instance.force_login(self.user)
        response = self.client_instance.get(
            reverse('client:edit_modal', kwargs={'pk': self.cliente.pk})
        )
        self.assertEqual(response.status_code, 200)

    def test_cliente_nao_encontrado(self):
        """Testa acesso a cliente inexistente"""
        self.client_instance.force_login(self.user)
        response = self.client_instance.get(reverse('client:detail', kwargs={'pk': 99999}))
        self.assertEqual(response.status_code, 404)


class ClientAPITest(TestCase):
    """Testes para a API do app client"""
    
    def setUp(self):
        """Configuração inicial para os testes"""
        self.client_instance = DjangoTestClient()
        self.user = User.objects.create_user(
            username='testuser',
            password='testpass123',
            email='test@example.com'
        )
        
        self.client_data = {
            'nome': 'Empresa Teste LTDA',
            'cnpj': '12.345.678/0001-90',
            'email': 'contato@empresateste.com',
            'telefone': '(11) 99999-9999',
            'endereco': 'Rua Teste, 123 - São Paulo/SP',
            'ativo': True
        }
        
        self.cliente = Client.objects.create(**self.client_data)

    def test_api_lista_clientes(self):
        """Testa a API de listagem de clientes"""
        self.client_instance.force_login(self.user)
        response = self.client_instance.get('/api/clients/')
        self.assertEqual(response.status_code, 200)
        
        data = json.loads(response.content)
        # Verificar se o cliente de teste está na lista
        cliente_encontrado = any(isinstance(item, dict) and item.get('nome') == 'Empresa Teste LTDA' for item in data)
        self.assertTrue(cliente_encontrado, "Cliente de teste não encontrado na lista")

    def test_api_detalhe_cliente(self):
        """Testa a API de detalhe do cliente"""
        self.client_instance.force_login(self.user)
        response = self.client_instance.get(f'/api/clients/{self.cliente.pk}/')
        self.assertEqual(response.status_code, 200)
        
        data = json.loads(response.content)
        self.assertEqual(data['nome'], 'Empresa Teste LTDA')
        self.assertEqual(data['cnpj'], '12.345.678/0001-90')

    def test_api_criar_cliente(self):
        """Testa a API de criação de cliente"""
        self.client_instance.force_login(self.user)
        
        data = {
            'nome': 'Nova Empresa API LTDA',
            'cnpj': '98.765.432/0001-10',
            'email': 'api@novaempresa.com',
            'telefone': '(11) 88888-8888',
            'endereco': 'Av. API, 123 - São Paulo/SP',
            'ativo': True
        }
        
        response = self.client_instance.post(
            '/api/clients/',
            data=json.dumps(data),
            content_type='application/json'
        )
        self.assertEqual(response.status_code, 201)
        
        # Verificar se o cliente foi criado
        novo_cliente = Client.objects.get(nome='Nova Empresa API LTDA')
        self.assertEqual(novo_cliente.cnpj, '98.765.432/0001-10')

    def test_api_atualizar_cliente(self):
        """Testa a API de atualização de cliente"""
        self.client_instance.force_login(self.user)
        
        data = {
            'nome': 'Empresa Atualizada API LTDA',
            'cnpj': '12.345.678/0001-90',
            'email': 'atualizada@empresa.com',
            'telefone': '(11) 77777-7777',
            'endereco': 'Rua Atualizada, 456 - São Paulo/SP',
            'ativo': False
        }
        
        response = self.client_instance.put(
            f'/api/clients/{self.cliente.pk}/',
            data=json.dumps(data),
            content_type='application/json'
        )
        self.assertEqual(response.status_code, 200)
        
        # Verificar se o cliente foi atualizado
        self.cliente.refresh_from_db()
        self.assertEqual(self.cliente.nome, 'Empresa Atualizada API LTDA')

    def test_api_deletar_cliente(self):
        """Testa a API de exclusão de cliente"""
        self.client_instance.force_login(self.user)
        
        response = self.client_instance.delete(f'/api/clients/{self.cliente.pk}/')
        self.assertEqual(response.status_code, 204)
        
        # Verificar se o cliente foi deletado
        with self.assertRaises(Client.DoesNotExist):
            Client.objects.get(pk=self.cliente.pk)

    def test_api_cliente_nao_encontrado(self):
        """Testa API para cliente inexistente"""
        self.client_instance.force_login(self.user)
        response = self.client_instance.get('/api/clients/99999/')
        self.assertEqual(response.status_code, 404)


class ClientFormTest(TestCase):
    """Testes para os formulários do app client"""
    
    def test_form_cliente_valido(self):
        """Testa formulário com dados válidos"""
        from .forms import ClientForm
        
        data = {
            'nome': 'Empresa Form Teste LTDA',
            'cnpj': '12.345.678/0001-90',
            'email': 'form@teste.com',
            'telefone': '(11) 99999-9999',
            'endereco': 'Rua Form, 123 - São Paulo/SP',
            'ativo': True
        }
        
        form = ClientForm(data=data)
        self.assertTrue(form.is_valid())
        
        if form.is_valid():
            cliente = form.save()
            self.assertEqual(cliente.nome, 'Empresa Form Teste LTDA')
            self.assertEqual(cliente.cnpj, '12.345.678/0001-90')

    def test_form_cnpj_invalido(self):
        """Testa formulário com CNPJ inválido"""
        from .forms import ClientForm
        
        data = {
            'nome': 'Empresa Form Teste LTDA',
            'cnpj': '123456789',  # CNPJ com menos de 14 dígitos
            'email': 'form@teste.com',
            'telefone': '(11) 99999-9999',
            'endereco': 'Rua Form, 123 - São Paulo/SP',
            'ativo': True
        }
        
        form = ClientForm(data=data)
        self.assertFalse(form.is_valid())
        self.assertIn('cnpj', form.errors)

    def test_form_email_invalido(self):
        """Testa formulário com email inválido"""
        from .forms import ClientForm
        
        data = {
            'nome': 'Empresa Form Teste LTDA',
            'cnpj': '12.345.678/0001-90',
            'email': 'email_invalido',  # Email inválido
            'telefone': '(11) 99999-9999',
            'endereco': 'Rua Form, 123 - São Paulo/SP',
            'ativo': True
        }
        
        form = ClientForm(data=data)
        self.assertFalse(form.is_valid())
        self.assertIn('email', form.errors)

    def test_form_campos_obrigatorios(self):
        """Testa formulário com campos obrigatórios vazios"""
        from .forms import ClientForm
        
        data = {
            'nome': '',  # Campo obrigatório vazio
            'cnpj': '',  # Campo obrigatório vazio
            'email': 'form@teste.com',
            'telefone': '(11) 99999-9999',
            'endereco': 'Rua Form, 123 - São Paulo/SP',
            'ativo': True
        }
        
        form = ClientForm(data=data)
        self.assertFalse(form.is_valid())
        self.assertIn('nome', form.errors)
        self.assertIn('cnpj', form.errors)


class ClientIntegrationTest(TestCase):
    """Testes de integração para o app client"""
    
    def setUp(self):
        """Configuração inicial para os testes"""
        self.client_instance = DjangoTestClient()
        self.user = User.objects.create_user(
            username='testuser',
            password='testpass123',
            email='test@example.com'
        )

    def test_fluxo_completo_cliente(self):
        """Testa o fluxo completo de um cliente"""
        self.client_instance.force_login(self.user)
        
        # 1. Criar cliente
        data = {
            'nome': 'Empresa Fluxo Completo LTDA',
            'cnpj': '12.345.678/0001-90',
            'email': 'fluxo@empresa.com',
            'telefone': '(11) 99999-9999',
            'endereco': 'Rua Fluxo, 123 - São Paulo/SP',
            'ativo': True
        }
        
        response = self.client_instance.post(reverse('client:create'), data)
        self.assertEqual(response.status_code, 302)
        
        cliente = Client.objects.get(nome='Empresa Fluxo Completo LTDA')
        self.assertEqual(cliente.cnpj, '12.345.678/0001-90')
        self.assertTrue(cliente.ativo)
        
        # 2. Editar cliente
        data_edit = {
            'nome': 'Empresa Fluxo Editada LTDA',
            'cnpj': '12.345.678/0001-90',
            'email': 'editado@empresa.com',
            'telefone': '(11) 88888-8888',
            'endereco': 'Rua Editada, 456 - São Paulo/SP',
            'ativo': False
        }
        
        response = self.client_instance.post(
            reverse('client:edit', kwargs={'pk': cliente.pk}),
            data_edit
        )
        self.assertEqual(response.status_code, 302)
        
        cliente.refresh_from_db()
        self.assertEqual(cliente.nome, 'Empresa Fluxo Editada LTDA')
        self.assertFalse(cliente.ativo)
        
        # 3. Deletar cliente
        response = self.client_instance.post(
            reverse('client:delete', kwargs={'pk': cliente.pk})
        )
        self.assertEqual(response.status_code, 302)
        
        with self.assertRaises(Client.DoesNotExist):
            Client.objects.get(pk=cliente.pk)

    def test_multiplos_clientes(self):
        """Testa operações com múltiplos clientes"""
        self.client_instance.force_login(self.user)
        
        # Criar múltiplos clientes
        clientes_data = [
            {
                'nome': 'Empresa A LTDA',
                'cnpj': '11.111.111/0001-11',
                'email': 'empresa_a@teste.com',
                'telefone': '(11) 11111-1111',
                'endereco': 'Rua A, 111 - São Paulo/SP',
                'ativo': True
            },
            {
                'nome': 'Empresa B LTDA',
                'cnpj': '22.222.222/0001-22',
                'email': 'empresa_b@teste.com',
                'telefone': '(11) 22222-2222',
                'endereco': 'Rua B, 222 - São Paulo/SP',
                'ativo': True
            },
            {
                'nome': 'Empresa C LTDA',
                'cnpj': '33.333.333/0001-33',
                'email': 'empresa_c@teste.com',
                'telefone': '(11) 33333-3333',
                'endereco': 'Rua C, 333 - São Paulo/SP',
                'ativo': False
            }
        ]
        
        for data in clientes_data:
            response = self.client_instance.post(reverse('client:create'), data)
            self.assertEqual(response.status_code, 302)
        
        # Verificar se todos foram criados
        self.assertEqual(Client.objects.count(), 3)
        
        # Verificar clientes ativos
        clientes_ativos = Client.objects.filter(ativo=True)
        self.assertEqual(clientes_ativos.count(), 2)
        
        # Verificar ordenação
        clientes = Client.objects.all()
        self.assertEqual(clientes[0].nome, 'Empresa A LTDA')
        self.assertEqual(clientes[1].nome, 'Empresa B LTDA')
        self.assertEqual(clientes[2].nome, 'Empresa C LTDA')
