from django.shortcuts import render, get_object_or_404, redirect
from django.views.generic import ListView, DetailView, CreateView, UpdateView, DeleteView
from django.contrib.auth.mixins import LoginRequiredMixin
from django.contrib import messages
from django.urls import reverse_lazy
from django.http import JsonResponse
from django.views import View

from .models import Client
from .forms import ClientForm
from system_logs.utils import log_object_created, log_object_updated, log_object_deleted, log_system_action


class SuperuserRequiredMixin:
    """
    Mixin para verificar se o usuário é superuser.
    """
    def dispatch(self, request, *args, **kwargs):
        if not request.user.is_superuser:
            messages.error(request, "Acesso negado. Apenas administradores podem acessar esta área.")
            return redirect('home')
        return super().dispatch(request, *args, **kwargs)


class ClientListView(LoginRequiredMixin, SuperuserRequiredMixin, ListView):
    """
    View para listagem de clientes.
    """
    model = Client
    template_name = 'client/list.html'
    context_object_name = 'clients'
    paginate_by = 20
    
    def get_queryset(self):
        queryset = Client.objects.all()
        
        # Filtro por status
        status = self.request.GET.get('status')
        if status == 'ativo':
            queryset = queryset.filter(ativo=True)
        elif status == 'inativo':
            queryset = queryset.filter(ativo=False)
        
        # Busca
        search = self.request.GET.get('search')
        if search:
            queryset = queryset.filter(
                nome__icontains=search
            ) | queryset.filter(
                cnpj__icontains=search
            ) | queryset.filter(
                email__icontains=search
            )
        
        return queryset.order_by('nome')
    
    def get_context_data(self, **kwargs):
        context = super().get_context_data(**kwargs)
        context['breadcrumb'] = [
            {'name': 'Clientes', 'url': None}
        ]
        context['search'] = self.request.GET.get('search', '')
        context['status'] = self.request.GET.get('status', '')
        return context


class ClientDetailView(LoginRequiredMixin, SuperuserRequiredMixin, DetailView):
    """
    View para detalhes do cliente.
    """
    model = Client
    template_name = 'client/detail.html'
    context_object_name = 'client'
    
    def get_context_data(self, **kwargs):
        context = super().get_context_data(**kwargs)
        client = self.get_object()
        
        context['breadcrumb'] = [
            {'name': 'Clientes', 'url': reverse_lazy('client:list')},
            {'name': client.nome, 'url': None}
        ]
        
        # Estatísticas do cliente
        context['stats'] = {
            'total_workspaces': client.workspaces.count(),
            'workspaces_ativos': client.workspaces.filter(ativo=True).count(),
            'total_portfolios': client.portfolios.count(),
            'portfolios_ativos': client.portfolios.filter(ativo=True).count(),
            'total_projects': client.projects.count(),
            'projects_ativos': client.projects.filter(ativo=True).count(),
            'projects_concluidos': client.projects.filter(status='concluido').count(),
        }
        
        # Workspaces recentes
        context['recent_workspaces'] = client.workspaces.filter(ativo=True)[:5]
        
        # Projetos recentes
        context['recent_projects'] = client.projects.filter(ativo=True).order_by('-created_at')[:5]
        
        return context


class ClientCreateView(LoginRequiredMixin, SuperuserRequiredMixin, CreateView):
    """
    View para criação de cliente.
    """
    model = Client
    form_class = ClientForm
    template_name = 'client/form.html'
    success_url = reverse_lazy('client:list')
    
    def get_context_data(self, **kwargs):
        context = super().get_context_data(**kwargs)
        context['breadcrumb'] = [
            {'name': 'Clientes', 'url': reverse_lazy('client:list')},
            {'name': 'Novo Cliente', 'url': None}
        ]
        context['title'] = 'Novo Cliente'
        return context
    
    def form_valid(self, form):
        response = super().form_valid(form)
        log_object_created(self.request, self.object)
        messages.success(self.request, 'Cliente criado com sucesso!')
        return response


class ClientCreateModalView(LoginRequiredMixin, SuperuserRequiredMixin, CreateView):
    """
    View para criação de cliente via modal.
    """
    model = Client
    form_class = ClientForm
    template_name = 'client/modal_create.html'
    
    def get_context_data(self, **kwargs):
        context = super().get_context_data(**kwargs)
        context.update({
            'modal_id': 'createClientModal',
            'modal_title': 'Novo Cliente',
            'modal_icon': 'bi-person-plus',
            'form_id': 'clientForm',
            'submit_text': 'Criar Cliente'
        })
        return context
    
    def form_valid(self, form):
        client = form.save()
        log_object_created(self.request, client)
        if self.request.headers.get('X-Requested-With') == 'XMLHttpRequest':
            return JsonResponse({
                'success': True,
                'message': 'Cliente criado com sucesso!',
                'client': {
                    'id': client.id,
                    'nome': client.nome,
                    'email': client.email,
                    'detail_url': reverse_lazy('client:detail', kwargs={'pk': client.id})
                }
            })
        return super().form_valid(form)
    
    def form_invalid(self, form):
        if self.request.headers.get('X-Requested-With') == 'XMLHttpRequest':
            return JsonResponse({
                'success': False,
                'errors': form.errors
            })
        return super().form_invalid(form)


class ClientUpdateModalView(LoginRequiredMixin, SuperuserRequiredMixin, UpdateView):
    """
    View para edição de cliente via modal.
    """
    model = Client
    form_class = ClientForm
    template_name = 'client/modal_edit.html'
    
    def get_context_data(self, **kwargs):
        context = super().get_context_data(**kwargs)
        client = self.get_object()
        context.update({
            'modal_id': 'editClientModal',
            'modal_title': f'Editar Cliente: {client.nome}',
            'modal_icon': 'bi-person-gear',
            'form_id': 'clientEditForm',
            'submit_text': 'Atualizar Cliente'
        })
        return context
    
    def form_valid(self, form):
        client = form.save()
        log_object_updated(self.request, client)
        if self.request.headers.get('X-Requested-With') == 'XMLHttpRequest':
            return JsonResponse({
                'success': True,
                'message': 'Cliente atualizado com sucesso!',
                'client': {
                    'id': client.id,
                    'nome': client.nome,
                    'email': client.email,
                    'ativo': client.ativo,
                    'detail_url': reverse_lazy('client:detail', kwargs={'pk': client.id})
                }
            })
        return super().form_valid(form)
    
    def form_invalid(self, form):
        if self.request.headers.get('X-Requested-With') == 'XMLHttpRequest':
            return JsonResponse({
                'success': False,
                'errors': form.errors
            })
        return super().form_invalid(form)


class ClientUpdateView(LoginRequiredMixin, SuperuserRequiredMixin, UpdateView):
    """
    View para edição de cliente.
    """
    model = Client
    form_class = ClientForm
    template_name = 'client/form.html'
    
    def get_success_url(self):
        return reverse_lazy('client:detail', kwargs={'pk': self.object.pk})
    
    def get_context_data(self, **kwargs):
        context = super().get_context_data(**kwargs)
        client = self.get_object()
        
        context['breadcrumb'] = [
            {'name': 'Clientes', 'url': reverse_lazy('client:list')},
            {'name': client.nome, 'url': reverse_lazy('client:detail', kwargs={'pk': client.pk})},
            {'name': 'Editar', 'url': None}
        ]
        context['title'] = f'Editar Cliente: {client.nome}'
        return context
    
    def form_valid(self, form):
        response = super().form_valid(form)
        log_object_updated(self.request, self.object)
        messages.success(self.request, 'Cliente atualizado com sucesso!')
        return response


class ClientDeleteView(LoginRequiredMixin, SuperuserRequiredMixin, DeleteView):
    """
    View para exclusão de cliente.
    """
    model = Client
    template_name = 'client/delete.html'
    success_url = reverse_lazy('client:list')
    
    def get_context_data(self, **kwargs):
        context = super().get_context_data(**kwargs)
        client = self.get_object()
        
        context['breadcrumb'] = [
            {'name': 'Clientes', 'url': reverse_lazy('client:list')},
            {'name': client.nome, 'url': reverse_lazy('client:detail', kwargs={'pk': client.pk})},
            {'name': 'Excluir', 'url': None}
        ]
        return context
    
    def delete(self, request, *args, **kwargs):
        client = self.get_object()
        response = super().delete(request, *args, **kwargs)
        log_object_deleted(request, client)
        messages.success(request, f'Cliente "{client.nome}" excluído com sucesso!')
        return response


class ClientToggleStatusView(LoginRequiredMixin, SuperuserRequiredMixin, View):
    """
    View para alternar status ativo/inativo do cliente.
    """
    
    def post(self, request, pk):
        client = get_object_or_404(Client, pk=pk)
        old_status = client.ativo
        client.ativo = not client.ativo
        client.save()
        
        status_text = 'ativado' if client.ativo else 'desativado'
        log_object_updated(request, client, details={'status_change': f'{old_status} -> {client.ativo}'})
        messages.success(request, f'Cliente "{client.nome}" {status_text} com sucesso!')
        
        if request.headers.get('X-Requested-With') == 'XMLHttpRequest':
            return JsonResponse({
                'success': True,
                'ativo': client.ativo,
                'message': f'Cliente {status_text} com sucesso!'
            })
        
        return redirect('client:detail', pk=pk)

