from rest_framework import viewsets, status
from rest_framework.decorators import action
from rest_framework.response import Response
from rest_framework.permissions import IsAuthenticated, IsAdminUser
from django_filters.rest_framework import DjangoFilterBackend
from rest_framework.filters import SearchFilter, OrderingFilter

from .models import Area, Collaborator
from .serializers import (
    AreaSerializer, 
    CollaboratorSerializer, 
    CollaboratorCreateSerializer, 
    CollaboratorListSerializer
)


class AreaViewSet(viewsets.ModelViewSet):
    """
    ViewSet para operações CRUD de áreas.
    """
    
    queryset = Area.objects.all()
    serializer_class = AreaSerializer
    permission_classes = [IsAdminUser]  # Apenas admins podem gerenciar áreas
    filter_backends = [DjangoFilterBackend, SearchFilter, OrderingFilter]
    filterset_fields = ['ativo']
    search_fields = ['nome', 'descricao']
    ordering_fields = ['nome', 'created_at']
    ordering = ['nome']

    @action(detail=True, methods=['get'])
    def collaborators(self, request, pk=None):
        """
        Retorna os colaboradores de uma área.
        """
        area = self.get_object()
        collaborators = area.collaborator_set.filter(ativo=True)
        
        serializer = CollaboratorListSerializer(collaborators, many=True)
        return Response(serializer.data)


class CollaboratorViewSet(viewsets.ModelViewSet):
    """
    ViewSet para operações CRUD de colaboradores.
    """
    
    queryset = Collaborator.objects.all()
    permission_classes = [IsAuthenticated]
    filter_backends = [DjangoFilterBackend, SearchFilter, OrderingFilter]
    filterset_fields = ['ativo', 'area', 'cargo']
    search_fields = ['nome', 'email', 'user__username']
    ordering_fields = ['nome', 'area__nome', 'created_at']
    ordering = ['nome']

    def get_serializer_class(self):
        if self.action == 'create':
            return CollaboratorCreateSerializer
        if self.action == 'list':
            return CollaboratorListSerializer
        return CollaboratorSerializer

    def get_queryset(self):
        queryset = Collaborator.objects.select_related('user', 'area').all()
        
        # Filtro por status ativo
        ativo = self.request.query_params.get('ativo', None)
        if ativo is not None:
            queryset = queryset.filter(ativo=ativo.lower() == 'true')
            
        # Filtro por área
        area_id = self.request.query_params.get('area_id', None)
        if area_id:
            queryset = queryset.filter(area_id=area_id)
            
        return queryset

    @action(detail=True, methods=['post'])
    def toggle_status(self, request, pk=None):
        """
        Alterna o status ativo/inativo do colaborador.
        """
        collaborator = self.get_object()
        collaborator.ativo = not collaborator.ativo
        collaborator.save()
        
        serializer = self.get_serializer(collaborator)
        return Response(serializer.data)

    @action(detail=True, methods=['get'])
    def tasks(self, request, pk=None):
        """
        Retorna as tarefas do colaborador.
        """
        collaborator = self.get_object()
        tasks = collaborator.tasks.all()
        
        from task.serializers import TaskListSerializer
        serializer = TaskListSerializer(tasks, many=True)
        return Response(serializer.data)

    @action(detail=False, methods=['get'])
    def me(self, request):
        """
        Retorna o perfil do colaborador logado.
        """
        try:
            collaborator = request.user.collaborator
            serializer = self.get_serializer(collaborator)
            return Response(serializer.data)
        except Collaborator.DoesNotExist:
            return Response({"detail": "Colaborador não encontrado para este usuário."}, status=status.HTTP_404_NOT_FOUND)

    @action(detail=True, methods=['get'])
    def statistics(self, request, pk=None):
        """
        Retorna estatísticas do colaborador.
        """
        collaborator = self.get_object()
        
        stats = {
            'total_tasks': collaborator.tasks.count(),
            'tasks_concluidas': collaborator.tasks.filter(status='concluida').count(),
            'tasks_em_andamento': collaborator.tasks.filter(status='em_andamento').count(),
            'tasks_atrasadas': collaborator.tasks.filter(is_atrasada=True).count(),
            'total_esforco_previsto': collaborator.tasks.aggregate(total=models.Sum('esforco_previsto'))['total'] or 0,
            'total_esforco_utilizado': collaborator.tasks.aggregate(total=models.Sum('esforco_utilizado'))['total'] or 0,
        }
        
        return Response(stats)


