from django import forms
from .models import Collaborator, Area


class CollaboratorForm(forms.ModelForm):
    """
    Formulário para criação e edição de colaboradores.
    """
    
    class Meta:
        model = Collaborator
        fields = ['nome', 'email', 'telefone', 'area', 'cargo', 'data_admissao', 'ativo']
        widgets = {
            'nome': forms.TextInput(attrs={
                'class': 'form-control',
                'placeholder': 'Nome completo do colaborador'
            }),
            'email': forms.EmailInput(attrs={
                'class': 'form-control',
                'placeholder': 'email@exemplo.com'
            }),
            'telefone': forms.TextInput(attrs={
                'class': 'form-control',
                'placeholder': '(11) 99999-9999'
            }),
            'area': forms.Select(attrs={
                'class': 'form-select'
            }),
            'cargo': forms.TextInput(attrs={
                'class': 'form-control',
                'placeholder': 'Cargo do colaborador'
            }),
            'data_admissao': forms.DateInput(attrs={
                'class': 'form-control',
                'type': 'date'
            }),
            'ativo': forms.CheckboxInput(attrs={
                'class': 'form-check-input'
            })
        }
    
    def __init__(self, *args, **kwargs):
        super().__init__(*args, **kwargs)
        
        # Filtrar apenas áreas ativas
        self.fields['area'].queryset = Area.objects.filter(ativo=True).order_by('nome')
        
        # Adiciona classes CSS aos labels
        for field_name, field in self.fields.items():
            if field_name != 'ativo':
                field.widget.attrs.update({'class': field.widget.attrs.get('class', '') + ' form-control'})
    
    def clean_email(self):
        email = self.cleaned_data.get('email')
        if email:
            # Verificar se já existe outro colaborador com este email
            existing_collaborator = Collaborator.objects.filter(email=email).exclude(pk=self.instance.pk if self.instance else None)
            if existing_collaborator.exists():
                raise forms.ValidationError("Já existe um colaborador com este e-mail.")
        return email 