from django.db import models
from django.contrib.auth.models import User


class Area(models.Model):
    """
    Modelo para representar áreas de trabalho/departamentos.
    """
    nome = models.CharField(
        max_length=100,
        unique=True,
        verbose_name="Nome",
        help_text="Nome da área/departamento"
    )
    
    descricao = models.TextField(
        blank=True,
        null=True,
        verbose_name="Descrição",
        help_text="Descrição da área"
    )
    
    ativo = models.BooleanField(
        default=True,
        verbose_name="Ativo",
        help_text="Indica se a área está ativa"
    )
    
    created_at = models.DateTimeField(
        auto_now_add=True,
        verbose_name="Criado em"
    )
    
    updated_at = models.DateTimeField(
        auto_now=True,
        verbose_name="Atualizado em"
    )

    class Meta:
        verbose_name = "Área"
        verbose_name_plural = "Áreas"
        ordering = ['nome']

    def __str__(self):
        return self.nome


class Collaborator(models.Model):
    """
    Modelo para representar colaboradores da agência.
    Estende o modelo User do Django com informações adicionais.
    """
    user = models.OneToOneField(
        User,
        on_delete=models.CASCADE,
        verbose_name="Usuário",
        help_text="Usuário Django associado"
    )
    
    nome = models.CharField(
        max_length=200,
        verbose_name="Nome Completo",
        help_text="Nome completo do colaborador"
    )
    
    email = models.EmailField(
        unique=True,
        verbose_name="E-mail",
        help_text="E-mail do colaborador"
    )
    
    area = models.ForeignKey(
        Area,
        on_delete=models.PROTECT,
        verbose_name="Área",
        help_text="Área de trabalho do colaborador"
    )
    
    telefone = models.CharField(
        max_length=20,
        blank=True,
        null=True,
        verbose_name="Telefone",
        help_text="Telefone do colaborador"
    )
    
    cargo = models.CharField(
        max_length=100,
        blank=True,
        null=True,
        verbose_name="Cargo",
        help_text="Cargo do colaborador"
    )
    
    data_admissao = models.DateField(
        blank=True,
        null=True,
        verbose_name="Data de Admissão",
        help_text="Data de admissão do colaborador"
    )
    
    ativo = models.BooleanField(
        default=True,
        verbose_name="Ativo",
        help_text="Indica se o colaborador está ativo"
    )
    
    foto = models.ImageField(
        upload_to='collaborators/photos/',
        blank=True,
        null=True,
        verbose_name="Foto",
        help_text="Foto do colaborador"
    )
    
    created_at = models.DateTimeField(
        auto_now_add=True,
        verbose_name="Criado em"
    )
    
    updated_at = models.DateTimeField(
        auto_now=True,
        verbose_name="Atualizado em"
    )

    class Meta:
        verbose_name = "Colaborador"
        verbose_name_plural = "Colaboradores"
        ordering = ['nome']

    def __str__(self):
        return self.nome

    def get_absolute_url(self):
        from django.urls import reverse
        return reverse('collaborator:detail', kwargs={'pk': self.pk})

    @property
    def get_full_name(self):
        return self.nome

    @property
    def get_short_name(self):
        return self.nome.split()[0] if self.nome else ""

    def save(self, *args, **kwargs):
        # Sincroniza email com o User
        if self.user:
            self.user.email = self.email
            self.user.first_name = self.get_short_name
            self.user.save()
        super().save(*args, **kwargs)

