from rest_framework import serializers
from django.contrib.auth.models import User
from .models import Area, Collaborator


class AreaSerializer(serializers.ModelSerializer):
    """
    Serializer para o modelo Area.
    """
    
    total_collaborators = serializers.SerializerMethodField()
    
    class Meta:
        model = Area
        fields = [
            'id', 'nome', 'descricao', 'ativo', 
            'total_collaborators', 'created_at', 'updated_at'
        ]
        read_only_fields = ['id', 'created_at', 'updated_at']
    
    def get_total_collaborators(self, obj):
        return obj.collaborator_set.filter(ativo=True).count()


class UserSerializer(serializers.ModelSerializer):
    """
    Serializer para o modelo User (Django).
    """
    
    class Meta:
        model = User
        fields = ['id', 'username', 'email', 'first_name', 'last_name', 'is_active']
        read_only_fields = ['id']


class CollaboratorSerializer(serializers.ModelSerializer):
    """
    Serializer para o modelo Collaborator.
    """
    
    user = UserSerializer(read_only=True)
    area_nome = serializers.CharField(source='area.nome', read_only=True)
    total_tasks = serializers.SerializerMethodField()
    tasks_concluidas = serializers.SerializerMethodField()
    
    class Meta:
        model = Collaborator
        fields = [
            'id', 'user', 'nome', 'email', 'area', 'area_nome', 
            'telefone', 'cargo', 'data_admissao', 'ativo', 'foto',
            'total_tasks', 'tasks_concluidas', 'created_at', 'updated_at'
        ]
        read_only_fields = ['id', 'created_at', 'updated_at']
    
    def get_total_tasks(self, obj):
        return obj.tasks.count()
    
    def get_tasks_concluidas(self, obj):
        return obj.tasks.filter(status='concluida').count()


class CollaboratorCreateSerializer(serializers.ModelSerializer):
    """
    Serializer para criação de colaboradores.
    """
    
    username = serializers.CharField(write_only=True)
    password = serializers.CharField(write_only=True, min_length=8)
    
    class Meta:
        model = Collaborator
        fields = [
            'nome', 'email', 'area', 'telefone', 'cargo', 
            'data_admissao', 'ativo', 'foto', 'username', 'password'
        ]
    
    def create(self, validated_data):
        username = validated_data.pop('username')
        password = validated_data.pop('password')
        
        # Cria o usuário Django
        user = User.objects.create_user(
            username=username,
            email=validated_data['email'],
            password=password,
            first_name=validated_data['nome'].split()[0] if validated_data['nome'] else '',
        )
        
        # Cria o colaborador
        collaborator = Collaborator.objects.create(user=user, **validated_data)
        
        return collaborator


class CollaboratorListSerializer(serializers.ModelSerializer):
    """
    Serializer simplificado para listagem de colaboradores.
    """
    
    area_nome = serializers.CharField(source='area.nome', read_only=True)
    username = serializers.CharField(source='user.username', read_only=True)
    
    class Meta:
        model = Collaborator
        fields = ['id', 'nome', 'email', 'area_nome', 'cargo', 'ativo', 'username']

