from django.test import TestCase, Client as DjangoTestClient
from django.contrib.auth.models import User
from django.urls import reverse
from django.core.exceptions import ValidationError
from datetime import date
import json

from .models import Collaborator, Area


class AreaModelTest(TestCase):
    """Testes para o modelo Area"""
    
    def setUp(self):
        """Configuração inicial para os testes"""
        self.area_data = {
            'nome': 'Desenvolvimento',
            'descricao': 'Área responsável pelo desenvolvimento de software',
            'ativo': True
        }

    def test_criacao_area(self):
        """Testa a criação básica de uma área"""
        area = Area.objects.create(**self.area_data)
        self.assertEqual(area.nome, 'Desenvolvimento')
        self.assertEqual(area.descricao, 'Área responsável pelo desenvolvimento de software')
        self.assertTrue(area.ativo)

    def test_nome_unico(self):
        """Testa se o nome da área deve ser único"""
        Area.objects.create(**self.area_data)
        
        # Tentar criar outra área com o mesmo nome
        with self.assertRaises(Exception):  # IntegrityError
            Area.objects.create(
                nome='Desenvolvimento',  # Mesmo nome
                descricao='Outra descrição'
            )

    def test_area_inativa(self):
        """Testa área inativa"""
        self.area_data['ativo'] = False
        area = Area.objects.create(**self.area_data)
        self.assertFalse(area.ativo)

    def test_str_representation(self):
        """Testa a representação string da área"""
        area = Area.objects.create(**self.area_data)
        self.assertEqual(str(area), 'Desenvolvimento')

    def test_ordering(self):
        """Testa a ordenação das áreas por nome"""
        # Criar áreas em ordem não alfabética
        Area.objects.create(nome='Zebra', descricao='Área Z')
        Area.objects.create(nome='Alpha', descricao='Área A')
        Area.objects.create(nome='Beta', descricao='Área B')
        
        areas = Area.objects.all()
        self.assertEqual(areas[0].nome, 'Alpha')
        self.assertEqual(areas[1].nome, 'Beta')
        self.assertEqual(areas[2].nome, 'Zebra')


class CollaboratorModelTest(TestCase):
    """Testes para o modelo Collaborator"""
    
    def setUp(self):
        """Configuração inicial para os testes"""
        self.user = User.objects.create_user(
            username='testuser',
            password='testpass123',
            email='test@example.com'
        )
        
        self.area = Area.objects.create(
            nome='Desenvolvimento',
            descricao='Área de desenvolvimento'
        )
        
        self.collaborator_data = {
            'user': self.user,
            'nome': 'João Silva Santos',
            'email': 'joao.silva@empresa.com',
            'area': self.area,
            'telefone': '(11) 99999-9999',
            'cargo': 'Desenvolvedor Full Stack',
            'data_admissao': date(2023, 1, 15),
            'ativo': True
        }

    def test_criacao_colaborador(self):
        """Testa a criação básica de um colaborador"""
        colaborador = Collaborator.objects.create(**self.collaborator_data)
        self.assertEqual(colaborador.nome, 'João Silva Santos')
        self.assertEqual(colaborador.email, 'joao.silva@empresa.com')
        self.assertEqual(colaborador.cargo, 'Desenvolvedor Full Stack')
        self.assertTrue(colaborador.ativo)

    def test_email_unico(self):
        """Testa se o email deve ser único"""
        Collaborator.objects.create(**self.collaborator_data)
        
        # Criar outro usuário
        outro_user = User.objects.create_user(
            username='outrouser',
            password='testpass123',
            email='outro@example.com'
        )
        
        # Tentar criar outro colaborador com o mesmo email
        with self.assertRaises(Exception):  # IntegrityError
            Collaborator.objects.create(
                user=outro_user,
                nome='Maria Silva',
                email='joao.silva@empresa.com',  # Mesmo email
                area=self.area
            )

    def test_colaborador_inativo(self):
        """Testa colaborador inativo"""
        self.collaborator_data['ativo'] = False
        colaborador = Collaborator.objects.create(**self.collaborator_data)
        self.assertFalse(colaborador.ativo)

    def test_str_representation(self):
        """Testa a representação string do colaborador"""
        colaborador = Collaborator.objects.create(**self.collaborator_data)
        self.assertEqual(str(colaborador), 'João Silva Santos')

    def test_get_absolute_url(self):
        """Testa o método get_absolute_url"""
        colaborador = Collaborator.objects.create(**self.collaborator_data)
        expected_url = reverse('collaborator:detail', kwargs={'pk': colaborador.pk})
        self.assertEqual(colaborador.get_absolute_url(), expected_url)

    def test_get_full_name(self):
        """Testa a propriedade get_full_name"""
        colaborador = Collaborator.objects.create(**self.collaborator_data)
        self.assertEqual(colaborador.get_full_name, 'João Silva Santos')

    def test_get_short_name(self):
        """Testa a propriedade get_short_name"""
        colaborador = Collaborator.objects.create(**self.collaborator_data)
        self.assertEqual(colaborador.get_short_name, 'João')

    def test_sincronizacao_user(self):
        """Testa a sincronização com o modelo User"""
        colaborador = Collaborator.objects.create(**self.collaborator_data)
        
        # Verificar se o email foi sincronizado
        self.assertEqual(colaborador.user.email, 'joao.silva@empresa.com')
        
        # Verificar se o first_name foi sincronizado
        self.assertEqual(colaborador.user.first_name, 'João')

    def test_ordering(self):
        """Testa a ordenação dos colaboradores por nome"""
        # Criar colaboradores em ordem não alfabética
        user2 = User.objects.create_user(username='user2', email='maria@teste.com')
        user3 = User.objects.create_user(username='user3', email='carlos@teste.com')
        
        Collaborator.objects.create(
            user=user2,
            nome='Maria Silva',
            email='maria@teste.com',
            area=self.area
        )
        Collaborator.objects.create(
            user=user3,
            nome='Carlos Santos',
            email='carlos@teste.com',
            area=self.area
        )
        
        # Buscar apenas os colaboradores criados neste teste
        colaboradores = Collaborator.objects.filter(
            nome__in=['Carlos Santos', 'João Silva Santos', 'Maria Silva']
        ).order_by('nome')
        
        self.assertEqual(len(colaboradores), 3)
        self.assertEqual(colaboradores[0].nome, 'Carlos Santos')
        self.assertEqual(colaboradores[1].nome, 'João Silva Santos')
        self.assertEqual(colaboradores[2].nome, 'Maria Silva')


class CollaboratorViewsTest(TestCase):
    """Testes para as views do app collaborator"""
    
    def setUp(self):
        """Configuração inicial para os testes"""
        self.client = DjangoTestClient()
        self.user = User.objects.create_user(
            username='testuser',
            password='testpass123',
            email='test@example.com'
        )
        
        self.area = Area.objects.create(
            nome='Desenvolvimento',
            descricao='Área de desenvolvimento'
        )
        
        self.colaborador = Collaborator.objects.create(
            user=self.user,
            nome='João Silva Santos',
            email='joao.silva@empresa.com',
            area=self.area,
            telefone='(11) 99999-9999',
            cargo='Desenvolvedor Full Stack',
            data_admissao=date(2023, 1, 15),
            ativo=True
        )

    def test_lista_colaboradores_sem_login(self):
        """Testa acesso à lista de colaboradores sem login"""
        response = self.client.get(reverse('collaborator:list'))
        self.assertEqual(response.status_code, 302)  # Redirecionamento para login

    def test_lista_colaboradores_com_login(self):
        """Testa acesso à lista de colaboradores com login"""
        self.client.force_login(self.user)
        response = self.client.get(reverse('collaborator:list'))
        self.assertEqual(response.status_code, 200)
        self.assertContains(response, 'João Silva Santos')

    def test_detalhe_colaborador(self):
        """Testa a view de detalhe do colaborador"""
        self.client.force_login(self.user)
        response = self.client.get(reverse('collaborator:detail', kwargs={'pk': self.colaborador.pk}))
        self.assertEqual(response.status_code, 200)
        self.assertContains(response, 'João Silva Santos')

    def test_criar_colaborador(self):
        """Testa a criação de um novo colaborador"""
        self.client.force_login(self.user)
        
        # Criar novo usuário
        novo_user = User.objects.create_user(
            username='novouser',
            password='testpass123',
            email='novo@empresa.com'
        )
        
        data = {
            'user': novo_user.pk,
            'nome': 'Maria Silva Santos',
            'email': 'maria.silva@empresa.com',
            'area': self.area.pk,
            'telefone': '(11) 88888-8888',
            'cargo': 'Designer UX/UI',
            'data_admissao': date(2023, 2, 1),
            'ativo': True
        }
        
        response = self.client.post(reverse('collaborator:create'), data)
        self.assertEqual(response.status_code, 302)  # Redirecionamento após criação
        
        # Verificar se o colaborador foi criado
        novo_colaborador = Collaborator.objects.get(nome='Maria Silva Santos')
        self.assertEqual(novo_colaborador.email, 'maria.silva@empresa.com')

    def test_editar_colaborador(self):
        """Testa a edição de um colaborador"""
        self.client.force_login(self.user)
        
        data = {
            'user': self.user.pk,
            'nome': 'João Silva Santos Editado',
            'email': 'joao.editado@empresa.com',
            'area': self.area.pk,
            'telefone': '(11) 77777-7777',
            'cargo': 'Desenvolvedor Senior',
            'data_admissao': date(2023, 1, 15),
            'ativo': True
        }
        
        response = self.client.post(
            reverse('collaborator:edit', kwargs={'pk': self.colaborador.pk}),
            data
        )
        self.assertEqual(response.status_code, 302)
        
        # Verificar se o colaborador foi editado
        self.colaborador.refresh_from_db()
        self.assertEqual(self.colaborador.nome, 'João Silva Santos Editado')
        self.assertEqual(self.colaborador.cargo, 'Desenvolvedor Senior')

    def test_deletar_colaborador(self):
        """Testa a exclusão de um colaborador"""
        self.client.force_login(self.user)
        
        response = self.client.post(
            reverse('collaborator:delete', kwargs={'pk': self.colaborador.pk})
        )
        self.assertEqual(response.status_code, 302)
        
        # Verificar se o colaborador foi deletado
        with self.assertRaises(Collaborator.DoesNotExist):
            Collaborator.objects.get(pk=self.colaborador.pk)

    def test_modal_criar_colaborador(self):
        """Testa o modal de criação de colaborador"""
        self.client.login(username='testuser', password='testpass123')
        response = self.client.get(reverse('collaborator:create_modal'))
        self.assertEqual(response.status_code, 200)

    def test_modal_editar_colaborador(self):
        """Testa o modal de edição de colaborador"""
        self.client.login(username='testuser', password='testpass123')
        response = self.client.get(
            reverse('collaborator:edit_modal', kwargs={'pk': self.colaborador.pk})
        )
        self.assertEqual(response.status_code, 200)

    def test_colaborador_nao_encontrado(self):
        """Testa acesso a colaborador inexistente"""
        self.client.login(username='testuser', password='testpass123')
        response = self.client.get(reverse('collaborator:detail', kwargs={'pk': 99999}))
        self.assertEqual(response.status_code, 404)

    def test_perfil_colaborador(self):
        """Testa a view de perfil do colaborador"""
        self.client.login(username='testuser', password='testpass123')
        response = self.client.get(reverse('collaborator:detail', kwargs={'pk': self.colaborador.pk}))
        self.assertEqual(response.status_code, 200)


class CollaboratorAPITest(TestCase):
    """Testes para a API do app collaborator"""
    
    def setUp(self):
        """Configuração inicial para os testes"""
        self.client = DjangoTestClient()
        self.user = User.objects.create_user(
            username='testuser',
            password='testpass123',
            email='test@example.com'
        )
        
        self.area = Area.objects.create(
            nome='Desenvolvimento',
            descricao='Área de desenvolvimento'
        )
        
        self.colaborador = Collaborator.objects.create(
            user=self.user,
            nome='João Silva Santos',
            email='joao.silva@empresa.com',
            area=self.area,
            telefone='(11) 99999-9999',
            cargo='Desenvolvedor Full Stack',
            data_admissao=date(2023, 1, 15),
            ativo=True
        )

    def test_api_lista_colaboradores(self):
        """Testa a API de listagem de colaboradores"""
        self.client.force_login(self.user)
        response = self.client.get('/api/collaborators/')
        self.assertEqual(response.status_code, 200)
        
        data = json.loads(response.content)
        # Verificar se pelo menos o colaborador do teste está na lista
        colaborador_encontrado = False
        for item in data:
            if isinstance(item, dict) and item.get('nome') == 'João Silva Santos':
                colaborador_encontrado = True
                break
        
        self.assertTrue(colaborador_encontrado, "Colaborador de teste não encontrado na lista")

    def test_api_detalhe_colaborador(self):
        """Testa a API de detalhe do colaborador"""
        self.client.force_login(self.user)
        response = self.client.get(f'/api/collaborators/{self.colaborador.pk}/')
        self.assertEqual(response.status_code, 200)
        
        data = json.loads(response.content)
        self.assertEqual(data['nome'], 'João Silva Santos')
        self.assertEqual(data['email'], 'joao.silva@empresa.com')

    def test_api_criar_colaborador(self):
        """Testa a API de criação de colaborador"""
        self.client.force_login(self.user)
        
        data = {
            'nome': 'Maria Silva Santos API',
            'email': 'maria.api@empresa.com',
            'area': self.area.pk,
            'telefone': '(11) 88888-8888',
            'cargo': 'Designer UX/UI',
            'data_admissao': '2023-02-01',
            'ativo': True,
            'username': 'mariasilva',
            'password': 'senha123456'
        }
        
        response = self.client.post(
            '/api/collaborators/',
            data=json.dumps(data),
            content_type='application/json'
        )
        self.assertEqual(response.status_code, 201)
        
        # Verificar se o colaborador foi criado
        novo_colaborador = Collaborator.objects.get(nome='Maria Silva Santos API')
        self.assertEqual(novo_colaborador.email, 'maria.api@empresa.com')

    def test_api_atualizar_colaborador(self):
        """Testa a API de atualização de colaborador"""
        self.client.force_login(self.user)
        
        data = {
            'user': self.user.pk,
            'nome': 'João Silva Santos API Editado',
            'email': 'joao.api@empresa.com',
            'area': self.area.pk,
            'telefone': '(11) 77777-7777',
            'cargo': 'Desenvolvedor Senior',
            'data_admissao': '2023-01-15',
            'ativo': True
        }
        
        response = self.client.put(
            f'/api/collaborators/{self.colaborador.pk}/',
            data=json.dumps(data),
            content_type='application/json'
        )
        self.assertEqual(response.status_code, 200)
        
        # Verificar se o colaborador foi atualizado
        self.colaborador.refresh_from_db()
        self.assertEqual(self.colaborador.nome, 'João Silva Santos API Editado')

    def test_api_deletar_colaborador(self):
        """Testa a API de exclusão de colaborador"""
        self.client.force_login(self.user)
        
        response = self.client.delete(f'/api/collaborators/{self.colaborador.pk}/')
        self.assertEqual(response.status_code, 204)
        
        # Verificar se o colaborador foi deletado
        with self.assertRaises(Collaborator.DoesNotExist):
            Collaborator.objects.get(pk=self.colaborador.pk)

    def test_api_colaborador_nao_encontrado(self):
        """Testa API para colaborador inexistente"""
        self.client.force_login(self.user)
        response = self.client.get('/api/collaborators/99999/')
        self.assertEqual(response.status_code, 404)


class CollaboratorFormTest(TestCase):
    """Testes para os formulários do app collaborator"""
    
    def setUp(self):
        """Configuração inicial para os testes"""
        self.user = User.objects.create_user(
            username='testuser',
            password='testpass123',
            email='test@example.com'
        )
        
        self.area = Area.objects.create(
            nome='Desenvolvimento',
            descricao='Área de desenvolvimento'
        )

    def test_form_colaborador_valido(self):
        """Testa formulário com dados válidos"""
        from .forms import CollaboratorForm
        
        data = {
            'user': self.user.pk,
            'nome': 'João Silva Santos Form',
            'email': 'joao.form@empresa.com',
            'area': self.area.pk,
            'telefone': '(11) 99999-9999',
            'cargo': 'Desenvolvedor Full Stack',
            'data_admissao': date(2023, 1, 15),
            'ativo': True
        }
        
        form = CollaboratorForm(data=data)
        self.assertTrue(form.is_valid())

    def test_form_email_invalido(self):
        """Testa formulário com email inválido"""
        from .forms import CollaboratorForm
        
        data = {
            'user': self.user.pk,
            'nome': 'João Silva Santos Form',
            'email': 'email_invalido',  # Email inválido
            'area': self.area.pk,
            'telefone': '(11) 99999-9999',
            'cargo': 'Desenvolvedor Full Stack',
            'data_admissao': date(2023, 1, 15),
            'ativo': True
        }
        
        form = CollaboratorForm(data=data)
        self.assertFalse(form.is_valid())
        self.assertIn('email', form.errors)

    def test_form_campos_obrigatorios(self):
        """Testa formulário com campos obrigatórios vazios"""
        from .forms import CollaboratorForm
        
        data = {
            'user': self.user.pk,
            'nome': '',  # Campo obrigatório vazio
            'email': '',  # Campo obrigatório vazio
            'area': self.area.pk,
            'telefone': '(11) 99999-9999',
            'cargo': 'Desenvolvedor Full Stack',
            'data_admissao': date(2023, 1, 15),
            'ativo': True
        }
        
        form = CollaboratorForm(data=data)
        self.assertFalse(form.is_valid())
        self.assertIn('nome', form.errors)
        self.assertIn('email', form.errors)


class CollaboratorIntegrationTest(TestCase):
    """Testes de integração para o app collaborator"""
    
    def setUp(self):
        """Configuração inicial para os testes"""
        self.client = DjangoTestClient()
        self.user = User.objects.create_user(
            username='testuser',
            password='testpass123',
            email='test@example.com'
        )
        
        self.area = Area.objects.create(
            nome='Desenvolvimento',
            descricao='Área de desenvolvimento'
        )

    def test_fluxo_completo_colaborador(self):
        """Testa o fluxo completo de um colaborador"""
        self.client.force_login(self.user)
        
        # 1. Criar colaborador
        data = {
            'nome': 'João Silva Santos Fluxo',
            'email': 'joao.fluxo@empresa.com',
            'area': self.area.pk,
            'telefone': '(11) 99999-9999',
            'cargo': 'Desenvolvedor Full Stack',
            'data_admissao': date(2023, 1, 15),
            'ativo': True,
            'username': 'joaofluxo',
            'password': 'senha123456'
        }
        
        response = self.client.post(reverse('collaborator:create'), data)
        self.assertEqual(response.status_code, 302)
        
        colaborador = Collaborator.objects.get(nome='João Silva Santos Fluxo')
        self.assertEqual(colaborador.email, 'joao.fluxo@empresa.com')
        self.assertTrue(colaborador.ativo)
        
        # 2. Editar colaborador
        data_edit = {
            'user': self.user.pk,
            'nome': 'João Silva Santos Fluxo Editado',
            'email': 'joao.fluxo.editado@empresa.com',
            'area': self.area.pk,
            'telefone': '(11) 88888-8888',
            'cargo': 'Desenvolvedor Senior',
            'data_admissao': date(2023, 1, 15),
            'ativo': False
        }
        
        response = self.client.post(
            reverse('collaborator:edit', kwargs={'pk': colaborador.pk}),
            data_edit
        )
        self.assertEqual(response.status_code, 302)
        
        colaborador.refresh_from_db()
        self.assertEqual(colaborador.nome, 'João Silva Santos Fluxo Editado')
        self.assertFalse(colaborador.ativo)
        
        # 3. Deletar colaborador
        response = self.client.post(
            reverse('collaborator:delete', kwargs={'pk': colaborador.pk})
        )
        self.assertEqual(response.status_code, 302)
        
        # Verificar se o colaborador foi deletado
        with self.assertRaises(Collaborator.DoesNotExist):
            Collaborator.objects.get(nome='João Silva Santos Fluxo Editado')

    def test_multiplos_colaboradores(self):
        """Testa operações com múltiplos colaboradores"""
        self.client.force_login(self.user)
        
        # Criar múltiplos colaboradores
        colaboradores_data = [
            {
                'user': self.user,
                'nome': 'João Silva',
                'email': 'joao@empresa.com',
                'area': self.area,
                'ativo': True
            },
            {
                'user': User.objects.create_user(username='user2', email='maria@empresa.com'),
                'nome': 'Maria Santos',
                'email': 'maria@empresa.com',
                'area': self.area,
                'ativo': True
            },
            {
                'user': User.objects.create_user(username='user3', email='carlos@empresa.com'),
                'nome': 'Carlos Oliveira',
                'email': 'carlos@empresa.com',
                'area': self.area,
                'ativo': False
            }
        ]
        
        for data in colaboradores_data:
            response = self.client.post(reverse('collaborator:create'), {
                'user': data['user'].pk,
                'nome': data['nome'],
                'email': data['email'],
                'area': data['area'].pk,
                'telefone': '(11) 99999-9999',
                'cargo': 'Desenvolvedor',
                'data_admissao': date(2023, 1, 15),
                'ativo': data['ativo']
            })
            self.assertEqual(response.status_code, 302)
        
        # Verificar se todos foram criados
        self.assertEqual(Collaborator.objects.count(), 3)
        
        # Verificar colaboradores ativos
        colaboradores_ativos = Collaborator.objects.filter(ativo=True)
        self.assertEqual(colaboradores_ativos.count(), 2)
        
        # Verificar ordenação
        colaboradores = Collaborator.objects.all()
        self.assertEqual(colaboradores[0].nome, 'Carlos Oliveira')
        self.assertEqual(colaboradores[1].nome, 'João Silva')
        self.assertEqual(colaboradores[2].nome, 'Maria Santos')
