from django.shortcuts import render, get_object_or_404, redirect
from django.contrib import messages
from django.contrib.auth.decorators import login_required
from django.core.paginator import Paginator
from django.db.models import Q, Count
from django.contrib.auth.models import User
from django.contrib.auth.hashers import make_password
from django.http import JsonResponse
from django.views.generic import CreateView, UpdateView
from django.contrib.auth.mixins import LoginRequiredMixin
from django.urls import reverse_lazy
from .models import Collaborator, Area
from .forms import CollaboratorForm


def superuser_required(view_func):
    """
    Decorator para verificar se o usuário é superuser.
    """
    def wrapper(request, *args, **kwargs):
        if not request.user.is_superuser:
            messages.error(request, "Acesso negado. Apenas administradores podem acessar esta área.")
            return redirect('home')
        return view_func(request, *args, **kwargs)
    return wrapper


class SuperuserRequiredMixin:
    """
    Mixin para verificar se o usuário é superuser.
    """
    def dispatch(self, request, *args, **kwargs):
        if not request.user.is_superuser:
            messages.error(request, "Acesso negado. Apenas administradores podem acessar esta área.")
            return redirect('home')
        return super().dispatch(request, *args, **kwargs)


@login_required
@superuser_required
def collaborator_list(request):
    """
    Lista todos os colaboradores com filtros e paginação.
    """
    # Obter parâmetros de filtro
    search = request.GET.get('search', '')
    area_id = request.GET.get('area', '')
    status = request.GET.get('status', '')
    
    # Query base
    collaborators = Collaborator.objects.select_related('user', 'area').all()
    
    # Aplicar filtros
    if search:
        collaborators = collaborators.filter(
            Q(nome__icontains=search) |
            Q(email__icontains=search) |
            Q(cargo__icontains=search) |
            Q(area__nome__icontains=search)
        )
    
    if area_id:
        collaborators = collaborators.filter(area_id=area_id)
    
    if status:
        if status == 'ativo':
            collaborators = collaborators.filter(ativo=True)
        elif status == 'inativo':
            collaborators = collaborators.filter(ativo=False)
    
    # Ordenar por nome
    collaborators = collaborators.order_by('nome')
    
    # Paginação
    paginator = Paginator(collaborators, 12)
    page_number = request.GET.get('page')
    page_obj = paginator.get_page(page_number)
    
    # Estatísticas
    total_collaborators = Collaborator.objects.count()
    collaborators_ativos = Collaborator.objects.filter(ativo=True).count()
    collaborators_inativos = Collaborator.objects.filter(ativo=False).count()
    areas_count = Area.objects.filter(ativo=True).count()
    
    # Lista de áreas para o filtro
    areas = Area.objects.filter(ativo=True).order_by('nome')
    
    context = {
        'collaborators': page_obj,
        'page_obj': page_obj,
        'is_paginated': page_obj.has_other_pages(),
        'search': search,
        'area_id': area_id,
        'status': status,
        'total_collaborators': total_collaborators,
        'collaborators_ativos': collaborators_ativos,
        'collaborators_inativos': collaborators_inativos,
        'areas_count': areas_count,
        'areas': areas,
    }
    
    return render(request, 'collaborator/list.html', context)


@login_required
@superuser_required
def collaborator_create(request):
    """
    Cria um novo colaborador.
    """
    if request.method == 'POST':
        # Implementação básica - será expandida com forms
        nome = request.POST.get('nome')
        email = request.POST.get('email')
        area_id = request.POST.get('area')
        cargo = request.POST.get('cargo', '')
        telefone = request.POST.get('telefone', '')
        data_admissao = request.POST.get('data_admissao', '')
        senha = request.POST.get('senha', '123456')  # Senha padrão
        
        if nome and email and area_id:
            try:
                # Verificar se o email já existe
                if User.objects.filter(email=email).exists():
                    messages.error(request, 'Este e-mail já está em uso.')
                    return render(request, 'collaborator/form.html', {'areas': Area.objects.all()})
                
                # Criar usuário Django
                username = email.split('@')[0]  # Usar parte do email como username
                user = User.objects.create_user(
                    username=username,
                    email=email,
                    password=senha,
                    first_name=nome.split()[0] if nome else '',
                    last_name=' '.join(nome.split()[1:]) if len(nome.split()) > 1 else ''
                )
                
                # Criar colaborador
                area = Area.objects.get(id=area_id)
                collaborator = Collaborator.objects.create(
                    user=user,
                    nome=nome,
                    email=email,
                    area=area,
                    cargo=cargo,
                    telefone=telefone,
                    data_admissao=data_admissao if data_admissao else None
                )
                
                messages.success(request, f'Colaborador "{collaborator.nome}" criado com sucesso!')
                return redirect('collaborator:detail', pk=collaborator.pk)
            except Area.DoesNotExist:
                messages.error(request, 'Área não encontrada.')
            except Exception as e:
                messages.error(request, f'Erro ao criar colaborador: {str(e)}')
        else:
            messages.error(request, 'Nome, e-mail e área são obrigatórios.')
    
    # GET request - mostrar formulário
    areas = Area.objects.filter(ativo=True).order_by('nome')
    
    context = {
        'areas': areas,
        'collaborator': None,
    }
    
    return render(request, 'collaborator/form.html', context)


@login_required
@superuser_required
def collaborator_detail(request, pk):
    """
    Exibe os detalhes de um colaborador.
    """
    collaborator = get_object_or_404(Collaborator, pk=pk)
    
    context = {
        'collaborator': collaborator,
    }
    
    return render(request, 'collaborator/detail.html', context)


@login_required
@superuser_required
def collaborator_edit(request, pk):
    """
    Edita um colaborador existente.
    """
    collaborator = get_object_or_404(Collaborator, pk=pk)
    
    if request.method == 'POST':
        # Implementação básica - será expandida com forms
        nome = request.POST.get('nome')
        email = request.POST.get('email')
        area_id = request.POST.get('area')
        cargo = request.POST.get('cargo', '')
        telefone = request.POST.get('telefone', '')
        data_admissao = request.POST.get('data_admissao', '')
        ativo = request.POST.get('ativo') == 'on'
        
        if nome and email and area_id:
            try:
                # Verificar se o email já existe (exceto para o próprio colaborador)
                if User.objects.filter(email=email).exclude(id=collaborator.user.id).exists():
                    messages.error(request, 'Este e-mail já está em uso.')
                    return render(request, 'collaborator/form.html', {
                        'collaborator': collaborator,
                        'areas': Area.objects.all()
                    })
                
                # Atualizar colaborador
                area = Area.objects.get(id=area_id)
                collaborator.nome = nome
                collaborator.email = email
                collaborator.area = area
                collaborator.cargo = cargo
                collaborator.telefone = telefone
                collaborator.data_admissao = data_admissao if data_admissao else None
                collaborator.ativo = ativo
                collaborator.save()
                
                messages.success(request, f'Colaborador "{collaborator.nome}" atualizado com sucesso!')
                return redirect('collaborator:detail', pk=collaborator.pk)
            except Area.DoesNotExist:
                messages.error(request, 'Área não encontrada.')
            except Exception as e:
                messages.error(request, f'Erro ao atualizar colaborador: {str(e)}')
        else:
            messages.error(request, 'Nome, e-mail e área são obrigatórios.')
    
    # GET request - mostrar formulário
    areas = Area.objects.filter(ativo=True).order_by('nome')
    
    context = {
        'collaborator': collaborator,
        'areas': areas,
    }
    
    return render(request, 'collaborator/form.html', context)


@login_required
@superuser_required
def collaborator_delete(request, pk):
    """
    Exclui um colaborador.
    """
    collaborator = get_object_or_404(Collaborator, pk=pk)
    
    if request.method == 'POST':
        nome = collaborator.nome
        user = collaborator.user
        
        # Excluir colaborador (isso também excluirá o usuário devido ao CASCADE)
        collaborator.delete()
        
        messages.success(request, f'Colaborador "{nome}" excluído com sucesso!')
        return redirect('collaborator:list')
    
    context = {
        'collaborator': collaborator,
    }
    
    return render(request, 'collaborator/delete.html', context)


@login_required
def profile_view(request):
    """
    Exibe o perfil do colaborador logado.
    """
    try:
        collaborator = Collaborator.objects.get(user=request.user)
    except Collaborator.DoesNotExist:
        messages.error(request, 'Perfil de colaborador não encontrado.')
        return redirect('home')
    
    context = {
        'collaborator': collaborator,
    }
    
    return render(request, 'collaborator/profile.html', context)


class ColaboradorCreateModalView(LoginRequiredMixin, SuperuserRequiredMixin, CreateView):
    """
    View para criação de colaborador via modal.
    """
    model = Collaborator
    form_class = CollaboratorForm
    template_name = 'collaborator/modal_create.html'
    
    def get_context_data(self, **kwargs):
        context = super().get_context_data(**kwargs)
        context.update({
            'modal_id': 'createColaboradorModal',
            'modal_title': 'Novo Colaborador',
            'modal_icon': 'bi-person-badge',
            'form_id': 'collaboratorForm',
            'submit_text': 'Criar Colaborador',
            'form': context.get('form')  # Garantir que o form está no contexto
        })
        return context
    
    def form_valid(self, form):
        # Criar usuário Django primeiro
        email = form.cleaned_data['email']
        nome = form.cleaned_data['nome']
        
        # Verificar se o email já existe
        if User.objects.filter(email=email).exists():
            if self.request.headers.get('X-Requested-With') == 'XMLHttpRequest':
                return JsonResponse({
                    'success': False,
                    'errors': {'email': ['Este e-mail já está em uso.']}
                })
            form.add_error('email', 'Este e-mail já está em uso.')
            return self.form_invalid(form)
        
        # Criar usuário Django
        username = email.split('@')[0]
        user = User.objects.create_user(
            username=username,
            email=email,
            password='123456',  # Senha padrão
            first_name=nome.split()[0] if nome else '',
            last_name=' '.join(nome.split()[1:]) if len(nome.split()) > 1 else ''
        )
        
        # Salvar colaborador
        collaborator = form.save(commit=False)
        collaborator.user = user
        collaborator.save()
        
        if self.request.headers.get('X-Requested-With') == 'XMLHttpRequest':
            return JsonResponse({
                'success': True,
                'message': 'Colaborador criado com sucesso!',
                'obj': {
                    'id': collaborator.id,
                    'nome': collaborator.nome,
                    'email': collaborator.email,
                    'detail_url': reverse_lazy('collaborator:detail', kwargs={'pk': collaborator.id})
                }
            })
        return super().form_valid(form)
    
    def form_invalid(self, form):
        if self.request.headers.get('X-Requested-With') == 'XMLHttpRequest':
            return JsonResponse({
                'success': False,
                'errors': form.errors
            })
        return super().form_invalid(form)


class ColaboradorUpdateModalView(LoginRequiredMixin, SuperuserRequiredMixin, UpdateView):
    """
    View para edição de colaborador via modal.
    """
    model = Collaborator
    form_class = CollaboratorForm
    template_name = 'collaborator/modal_edit.html'
    
    def get_context_data(self, **kwargs):
        context = super().get_context_data(**kwargs)
        collaborator = self.get_object()
        context.update({
            'modal_id': 'editColaboradorModal',
            'modal_title': f'Editar Colaborador: {collaborator.nome}',
            'modal_icon': 'bi-person-gear',
            'form_id': 'collaboratorEditForm',
            'submit_text': 'Atualizar Colaborador',
            'form': context.get('form')  # Garantir que o form está no contexto
        })
        return context
    
    def form_valid(self, form):
        collaborator = form.save()
        
        # Atualizar dados do usuário Django se necessário
        if collaborator.user:
            user = collaborator.user
            user.email = form.cleaned_data['email']
            user.first_name = form.cleaned_data['nome'].split()[0] if form.cleaned_data['nome'] else ''
            user.last_name = ' '.join(form.cleaned_data['nome'].split()[1:]) if len(form.cleaned_data['nome'].split()) > 1 else ''
            user.save()
        
        if self.request.headers.get('X-Requested-With') == 'XMLHttpRequest':
            return JsonResponse({
                'success': True,
                'message': 'Colaborador atualizado com sucesso!',
                'obj': {
                    'id': collaborator.id,
                    'nome': collaborator.nome,
                    'email': collaborator.email,
                    'ativo': collaborator.ativo,
                    'detail_url': reverse_lazy('collaborator:detail', kwargs={'pk': collaborator.id})
                }
            })
        return super().form_valid(form)
    
    def form_invalid(self, form):
        if self.request.headers.get('X-Requested-With') == 'XMLHttpRequest':
            return JsonResponse({
                'success': False,
                'errors': form.errors
            })
        return super().form_invalid(form)

