#!/bin/bash

# Script de Deploy para Sistema de Gestão de Agência
# Uso: ./deploy.sh [production|staging]

set -e  # Para em caso de erro

ENVIRONMENT=${1:-production}
PROJECT_DIR="/opt/gestao_agencia"
BACKUP_DIR="/opt/backups"

echo "============================================================"
echo "🚀 DEPLOY - SISTEMA DE GESTÃO DE AGÊNCIA"
echo "🌍 Ambiente: $ENVIRONMENT"
echo "📁 Diretório: $PROJECT_DIR"
echo "============================================================"

# Função para backup
backup_database() {
    echo "💾 Criando backup do banco de dados..."
    mkdir -p $BACKUP_DIR
    TIMESTAMP=$(date +%Y%m%d_%H%M%S)
    
    if [ "$ENVIRONMENT" = "production" ]; then
        docker-compose exec -T db pg_dump -U postgres gestao_agencia > $BACKUP_DIR/backup_$TIMESTAMP.sql
        echo "✅ Backup criado: $BACKUP_DIR/backup_$TIMESTAMP.sql"
    fi
}

# Função para verificar saúde dos serviços
health_check() {
    echo "🏥 Verificando saúde dos serviços..."
    
    # Verificar se a aplicação está respondendo
    for i in {1..30}; do
        if curl -f http://localhost/health/ > /dev/null 2>&1; then
            echo "✅ Aplicação está funcionando!"
            return 0
        fi
        echo "⏳ Aguardando aplicação... ($i/30)"
        sleep 2
    done
    
    echo "❌ Aplicação não está respondendo!"
    return 1
}

# Função para rollback
rollback() {
    echo "🔄 Executando rollback..."
    docker-compose down
    git reset --hard HEAD~1
    docker-compose up -d
    echo "✅ Rollback concluído!"
}

# Verificar se estamos no diretório correto
if [ ! -f "$PROJECT_DIR/docker-compose.yml" ]; then
    echo "❌ Erro: docker-compose.yml não encontrado em $PROJECT_DIR"
    exit 1
fi

cd $PROJECT_DIR

# Backup antes do deploy
backup_database

# Parar serviços atuais
echo "🛑 Parando serviços atuais..."
docker-compose down

# Atualizar código
echo "📥 Atualizando código..."
git fetch origin
git reset --hard origin/main

# Construir novas imagens
echo "🔨 Construindo imagens Docker..."
docker-compose build --no-cache

# Iniciar serviços
echo "🚀 Iniciando serviços..."
docker-compose up -d

# Aguardar serviços iniciarem
echo "⏳ Aguardando serviços iniciarem..."
sleep 10

# Executar migrações
echo "🗄️ Executando migrações..."
docker-compose exec -T web python manage.py migrate

# Coletar arquivos estáticos
echo "📦 Coletando arquivos estáticos..."
docker-compose exec -T web python manage.py collectstatic --noinput

# Verificar saúde
if health_check; then
    echo "============================================================"
    echo "✅ DEPLOY CONCLUÍDO COM SUCESSO!"
    echo "🌐 Acesse: http://localhost"
    echo "🔐 Admin: http://localhost/admin"
    echo "============================================================"
else
    echo "❌ DEPLOY FALHOU!"
    rollback
    exit 1
fi

# Limpar imagens antigas
echo "🧹 Limpando imagens antigas..."
docker system prune -f

echo "🎉 Deploy finalizado!" 