#!/bin/bash

echo "============================================================"
echo "🚀 DEPLOY CLOUDEZ - Sistema de Gestão de Agência"
echo "============================================================"
echo "Baseado na documentação: https://help.cloudez.io/subir-aplicacao-customizada-em-python/"
echo "============================================================"

# Configurações
DOMAIN="gestao.blackbeans.com.br"
PROJECT_DIR="/srv/$DOMAIN"
WWW_DIR="$PROJECT_DIR/www"

# Verificar se estamos no diretório correto
if [ ! -f "manage.py" ]; then
    echo "❌ Erro: manage.py não encontrado!"
    echo "Execute este script no diretório do projeto"
    exit 1
fi

# Verificar se o diretório da Cloudez existe
if [ ! -d "$PROJECT_DIR" ]; then
    echo "❌ Erro: Diretório $PROJECT_DIR não encontrado!"
    echo "Certifique-se de que a aplicação Python foi criada no painel da Cloudez"
    exit 1
fi

echo "✅ Diretório da Cloudez encontrado: $PROJECT_DIR"

# 1. Copiar arquivos para o diretório www
echo "📁 Copiando arquivos para $WWW_DIR..."
sudo cp -r . "$WWW_DIR/"
sudo chown -R www-data:www-data "$WWW_DIR"

# 2. Ativar ambiente virtual da Cloudez
echo "🔧 Ativando ambiente virtual da Cloudez..."
source "$PROJECT_DIR/.virtualenv/3.7/bin/activate"

# 3. Instalar dependências
echo "📦 Instalando dependências..."
cd "$WWW_DIR"
pip install -r requirements.txt

# 4. Configurar variáveis de ambiente
echo "⚙️ Configurando variáveis de ambiente..."
if [ ! -f ".env" ]; then
    cp env.example .env
    echo "📝 Arquivo .env criado. Configure as variáveis de ambiente!"
fi

# 5. Executar migrações
echo "🗄️ Executando migrações..."
python manage.py migrate --noinput

# 6. Coletar arquivos estáticos
echo "📦 Coletando arquivos estáticos..."
python manage.py collectstatic --noinput

# 7. Criar superusuário se não existir
echo "👤 Verificando superusuário..."
python manage.py shell -c "
from django.contrib.auth.models import User
if not User.objects.filter(is_superuser=True).exists():
    User.objects.create_superuser('admin', 'admin@gestaoagencia.com', 'admin123')
    print('Superusuário criado: admin/admin123')
else:
    print('Superusuário já existe')
"

# 8. Copiar script de inicialização
echo "📝 Copiando script de inicialização..."
sudo cp start_cloudez.sh "$PROJECT_DIR/"
sudo chmod +x "$PROJECT_DIR/start_cloudez.sh"
sudo chown www-data:www-data "$PROJECT_DIR/start_cloudez.sh"

# 9. Criar serviço systemd
echo "🔧 Criando serviço systemd..."
sudo cp gestao.service /etc/systemd/system/
sudo systemctl daemon-reload

# 10. Copiar configuração Nginx
echo "🌐 Configurando Nginx..."
sudo cp proxy.conf "$PROJECT_DIR/etc/nginx/conf.d/"

# 11. Iniciar serviço
echo "🚀 Iniciando serviço..."
sudo systemctl start gestao
sudo systemctl enable gestao

# 12. Reiniciar Nginx
echo "🔄 Reiniciando Nginx..."
sudo systemctl restart nginx

# 13. Verificar status
echo "📊 Verificando status..."
sleep 3
if sudo systemctl is-active --quiet gestao; then
    echo "✅ Serviço iniciado com sucesso!"
else
    echo "❌ Erro ao iniciar serviço!"
    sudo systemctl status gestao
    exit 1
fi

echo ""
echo "============================================================"
echo "✅ DEPLOY CLOUDEZ CONCLUÍDO!"
echo "============================================================"
echo ""
echo "🌐 Acesse: https://$DOMAIN"
echo "🔐 Admin: https://$DOMAIN/admin"
echo "👤 Login: admin/admin123"
echo ""
echo "📊 Comandos úteis:"
echo "   Status: sudo systemctl status gestao"
echo "   Reiniciar: sudo systemctl restart gestao"
echo "   Logs: sudo journalctl -u gestao -f"
echo "   Parar: sudo systemctl stop gestao"
echo "" 