#!/bin/bash

echo "============================================================"
echo "🚀 DEPLOY UWSGI - Sistema de Gestão de Agência"
echo "============================================================"

# Configurações
PROJECT_DIR="/opt/gestao_agencia"
DOMAIN="seu-subdominio.seudominio.com"

# Verificar se estamos no diretório correto
if [ ! -f "manage.py" ]; then
    echo "❌ Erro: manage.py não encontrado!"
    echo "Execute este script no diretório do projeto"
    exit 1
fi

# Ativar ambiente virtual
echo "🔧 Ativando ambiente virtual..."
source venv/bin/activate

# Verificar configurações
echo "✅ Verificando configurações..."
python manage.py check --deploy

# Executar migrações
echo "🗄️ Executando migrações..."
python manage.py migrate

# Coletar arquivos estáticos
echo "📦 Coletando arquivos estáticos..."
python manage.py collectstatic --noinput

# Verificar/criar superusuário
echo "👤 Verificando superusuário..."
python manage.py shell -c "
from django.contrib.auth.models import User
if not User.objects.filter(is_superuser=True).exists():
    User.objects.create_superuser('admin', 'admin@gestaoagencia.com', 'admin123')
    print('Superusuário criado: admin/admin123')
else:
    print('Superusuário já existe')
"

# Criar diretórios de log
echo "📝 Criando diretórios de log..."
sudo mkdir -p /var/log/uwsgi
sudo chown www-data:www-data /var/log/uwsgi

# Configurar uWSGI
echo "⚙️ Configurando uWSGI..."
if [ -f "uwsgi.ini" ]; then
    echo "✅ Arquivo uwsgi.ini encontrado"
else
    echo "❌ Arquivo uwsgi.ini não encontrado!"
    exit 1
fi

# Parar uWSGI se estiver rodando
echo "🛑 Parando uWSGI se estiver rodando..."
sudo pkill -f uwsgi || true

# Iniciar uWSGI
echo "🚀 Iniciando uWSGI..."
sudo uwsgi --ini uwsgi.ini --daemonize

# Verificar se uWSGI está rodando
sleep 3
if pgrep -f uwsgi > /dev/null; then
    echo "✅ uWSGI iniciado com sucesso!"
else
    echo "❌ Erro ao iniciar uWSGI!"
    exit 1
fi

echo ""
echo "============================================================"
echo "✅ DEPLOY UWSGI CONCLUÍDO!"
echo "============================================================"
echo ""
echo "🌐 Acesse: http://$DOMAIN"
echo "🔐 Admin: http://$DOMAIN/admin"
echo "👤 Login: admin/admin123"
echo ""
echo "📊 Status do uWSGI:"
echo "   sudo uwsgi --stop /tmp/uwsgi.pid"
echo "   sudo uwsgi --reload /tmp/uwsgi.pid"
echo "   sudo tail -f /var/log/uwsgi/gestao_agencia.log"
echo "" 