from django.core.management.base import BaseCommand
from django.contrib.auth.models import User, Group
from django.utils import timezone
from datetime import datetime, timedelta
import random

from client.models import Client
from collaborator.models import Area, Collaborator
from workspace.models import Workspace
from portfolio.models import Portfolio
from project.models import Project
from board.models import Board
from task.models import Group as TaskGroup, Task
from notification.models import Notification


class Command(BaseCommand):
    help = 'Carrega dados de exemplo para demonstração'

    def add_arguments(self, parser):
        parser.add_argument(
            '--reset',
            action='store_true',
            help='Remove todos os dados antes de carregar os novos',
        )

    def handle(self, *args, **options):
        if options['reset']:
            self.stdout.write('Removendo dados existentes...')
            self.reset_data()
        
        self.stdout.write('Carregando dados de exemplo...')
        
        # Carregar fixtures básicas
        self.load_basic_fixtures()
        
        # Criar usuários e colaboradores
        self.create_users_and_collaborators()
        
        # Criar workspaces e projetos
        self.create_workspaces_and_projects()
        
        # Criar boards e tarefas
        self.create_boards_and_tasks()
        
        # Criar algumas notificações
        self.create_notifications()
        
        self.stdout.write(self.style.SUCCESS('Dados de exemplo carregados com sucesso!'))
    
    def reset_data(self):
        """Remove todos os dados para começar limpo."""
        Task.objects.all().delete()
        TaskGroup.objects.all().delete()
        Board.objects.all().delete()
        Project.objects.all().delete()
        Portfolio.objects.all().delete()
        Workspace.objects.all().delete()
        Collaborator.objects.all().delete()
        Client.objects.all().delete()
        Notification.objects.all().delete()
        
        # Manter apenas superusuários
        User.objects.filter(is_superuser=False).delete()
    
    def load_basic_fixtures(self):
        """Carrega fixtures básicas."""
        from django.core.management import call_command
        
        try:
            call_command('loaddata', 'areas.json')
            self.stdout.write('Áreas carregadas')
        except:
            self.stdout.write(self.style.WARNING('Erro ao carregar áreas'))
        
        try:
            call_command('loaddata', 'clients.json')
            self.stdout.write('Clientes carregados')
        except:
            self.stdout.write(self.style.WARNING('Erro ao carregar clientes'))
    
    def create_users_and_collaborators(self):
        """Cria usuários e colaboradores de exemplo."""
        self.stdout.write('Criando usuários e colaboradores...')
        
        # Grupos
        gerentes_group, _ = Group.objects.get_or_create(name='Gerentes')
        colaboradores_group, _ = Group.objects.get_or_create(name='Colaboradores')
        
        # Áreas
        areas = list(Area.objects.all())
        
        # Gerente
        if not User.objects.filter(username='gerente').exists():
            user_gerente = User.objects.create_user(
                username='gerente',
                email='gerente@agencia.com',
                password='123456',
                first_name='João',
                last_name='Silva'
            )
            user_gerente.groups.add(gerentes_group)
            
            # Encontrar área de Gestão de Projetos ou usar a primeira disponível
            area_gerente = areas[5] if len(areas) > 5 else areas[0]
            Collaborator.objects.create(
                user=user_gerente,
                nome='João Silva',
                email='gerente@agencia.com',
                area=area_gerente,
                cargo='Gerente de Projetos',
                ativo=True
            )
        
        # Colaboradores
        colaboradores_data = [
            ('ana.dev', 'Ana', 'Santos', 'ana@agencia.com', areas[0] if len(areas) > 0 else areas[0], 'Desenvolvedora Senior'),
            ('carlos.design', 'Carlos', 'Oliveira', 'carlos@agencia.com', areas[1] if len(areas) > 1 else areas[0], 'Designer UX/UI'),
            ('maria.marketing', 'Maria', 'Costa', 'maria@agencia.com', areas[2] if len(areas) > 2 else areas[0], 'Analista de Marketing'),
            ('pedro.dev', 'Pedro', 'Almeida', 'pedro@agencia.com', areas[0] if len(areas) > 0 else areas[0], 'Desenvolvedor Pleno'),
            ('lucia.qa', 'Lúcia', 'Ferreira', 'lucia@agencia.com', areas[8] if len(areas) > 8 else areas[0], 'Analista de Qualidade'),
        ]
        
        for username, first_name, last_name, email, area, cargo in colaboradores_data:
            if not User.objects.filter(username=username).exists():
                user = User.objects.create_user(
                    username=username,
                    email=email,
                    password='123456',
                    first_name=first_name,
                    last_name=last_name
                )
                user.groups.add(colaboradores_group)
                
                Collaborator.objects.create(
                    user=user,
                    nome=f'{first_name} {last_name}',
                    email=email,
                    area=area,
                    cargo=cargo,
                    ativo=True
                )
    
    def create_workspaces_and_projects(self):
        """Cria workspaces, portfolios e projetos de exemplo."""
        self.stdout.write('Criando workspaces e projetos...')
        
        clients = list(Client.objects.all())
        
        # Cores para workspaces
        cores = ['#007bff', '#28a745', '#dc3545', '#ffc107', '#6f42c1']
        
        for i, client in enumerate(clients[:3]):  # Apenas 3 primeiros clientes
            # Workspace
            workspace = Workspace.objects.create(
                nome=f'Workspace {client.nome}',
                descricao=f'Workspace principal para projetos do cliente {client.nome}',
                cliente=client,
                cor=cores[i % len(cores)],
                ativo=True
            )
            
            # Portfolio
            portfolio = Portfolio.objects.create(
                nome=f'Portfolio Digital {client.nome}',
                descricao=f'Portfolio de projetos digitais para {client.nome}',
                workspace=workspace,
                cliente=client,
                ativo=True
            )
            
            # Projetos
            projetos_data = [
                ('Website Institucional', 'Desenvolvimento do novo website institucional', 'planejamento'),
                ('Sistema de CRM', 'Implementação de sistema de gestão de clientes', 'em_andamento'),
                ('App Mobile', 'Desenvolvimento de aplicativo mobile', 'em_andamento'),
            ]
            
            for j, (nome, desc, status) in enumerate(projetos_data):
                inicio = timezone.now().date() - timedelta(days=random.randint(1, 30))
                fim = inicio + timedelta(days=random.randint(30, 90))
                
                Project.objects.create(
                    nome=nome,
                    descricao=desc,
                    portfolio=portfolio,
                    workspace=workspace,
                    cliente=client,
                    cronograma_inicio=inicio,
                    cronograma_fim=fim,
                    status=status,
                    prioridade=random.choice(['baixa', 'media', 'alta']),
                    orcamento=random.randint(10000, 100000),
                    esforco_previsto=random.randint(100, 500),
                    ativo=True
                )
    
    def create_boards_and_tasks(self):
        """Cria boards, grupos e tarefas de exemplo."""
        self.stdout.write('Criando boards e tarefas...')
        
        projects = Project.objects.all()
        collaborators = list(Collaborator.objects.all())
        
        for project in projects:
            # Boards
            boards_data = [
                ('Planejamento', 'Board para atividades de planejamento'),
                ('Desenvolvimento', 'Board para atividades de desenvolvimento'),
                ('Testes', 'Board para atividades de testes'),
            ]
            
            for i, (nome, desc) in enumerate(boards_data):
                board = Board.objects.create(
                    nome=nome,
                    descricao=desc,
                    project=project,
                    workspace=project.workspace,
                    cliente=project.cliente,
                    ordem=i + 1,
                    ativo=True
                )
                
                # Grupos
                grupos_data = [
                    ('A Fazer', 'Tarefas a serem iniciadas'),
                    ('Em Andamento', 'Tarefas em execução'),
                    ('Concluído', 'Tarefas finalizadas'),
                ]
                
                for j, (nome_grupo, desc_grupo) in enumerate(grupos_data):
                    group = TaskGroup.objects.create(
                        nome=nome_grupo,
                        descricao=desc_grupo,
                        board=board,
                        ordem=j + 1,
                        ativo=True
                    )
                    
                    # Tarefas
                    num_tasks = random.randint(2, 5)
                    for k in range(num_tasks):
                        inicio = timezone.now().date() + timedelta(days=random.randint(-10, 10))
                        fim = inicio + timedelta(days=random.randint(1, 14))
                        
                        status_map = {
                            'A Fazer': 'nao_iniciada',
                            'Em Andamento': 'em_andamento',
                            'Concluído': 'concluida'
                        }
                        
                        Task.objects.create(
                            nome=f'Tarefa {k+1} - {nome} - {project.nome}',
                            descricao=f'Descrição da tarefa {k+1} do board {nome}',
                            group=group,
                            responsavel=random.choice(collaborators),
                            cronograma_inicio=inicio,
                            cronograma_fim=fim,
                            status=status_map[nome_grupo],
                            prioridade=random.choice(['baixa', 'media', 'alta', 'urgente']),
                            esforco_previsto=random.randint(4, 40),
                            esforco_utilizado=random.randint(0, 20) if status_map[nome_grupo] != 'nao_iniciada' else 0
                        )
    
    def create_notifications(self):
        """Cria algumas notificações de exemplo."""
        self.stdout.write('Criando notificações...')
        
        collaborators = list(Collaborator.objects.all())
        tasks = list(Task.objects.all()[:10])  # Apenas 10 primeiras tarefas
        
        for i, task in enumerate(tasks):
            if task.responsavel:
                Notification.objects.create(
                    titulo=f'Nova tarefa atribuída: {task.nome}',
                    mensagem=f'Você foi designado como responsável pela tarefa "{task.nome}" no projeto {task.group.board.project.nome}.',
                    destinatario=task.responsavel,
                    tipo='responsavel_adicionado',
                    tarefa=task,
                    lida=random.choice([True, False]),
                    status='pendente'
                )

